package org.bdware.server.http;

import com.google.gson.JsonElement;
import com.google.gson.JsonObject;
import com.google.gson.JsonParser;
import io.netty.buffer.ByteBuf;
import io.netty.buffer.ByteBufInputStream;
import io.netty.handler.codec.http.FullHttpRequest;
import io.netty.handler.codec.http.QueryStringDecoder;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.bdware.sc.util.ExceptionUtil;
import org.bouncycastle.crypto.params.ECPublicKeyParameters;
import org.bouncycastle.pqc.math.linearalgebra.ByteUtils;
import org.zz.gmhelper.BCECUtil;
import org.zz.gmhelper.SM2Util;

import java.io.InputStreamReader;
import java.net.URLDecoder;
import java.util.List;
import java.util.Map;

public class ArgParser {

    public interface VerifiedCallback {
        void onResult(boolean verified, JsonObject param);
    }

    private static final Logger LOGGER = LogManager.getLogger(ArgParser.class);

    public static JsonObject parseGetAndVerify(FullHttpRequest msg, VerifiedCallback cb)
            throws Exception {
        QueryStringDecoder decoderQuery = new QueryStringDecoder(msg.uri());
        Map<String, List<String>> parameters = decoderQuery.parameters();
        JsonObject transformedParam = new JsonObject();
        for (String key : parameters.keySet()) {
            List<String> val = parameters.get(key);
            if (null != val) {
                transformedParam.addProperty(key, val.get(0));
            }
        }
        // 匿名用户权限为0
        transformedParam.addProperty("permission", 0);
        transformedParam.remove("verifiedPubKey");
        // 验签 有pubKey就必须有sign
        String uri = URLDecoder.decode(msg.uri(), "UTF-8").split("\\?")[1];
        int index = uri.lastIndexOf('&');
        if (index >= 0)
            verifyParam(transformedParam, uri.substring(0, index), cb);
        return transformedParam;
    }

    private static void verifyParam(JsonObject transformedParam, String toVerifyStr,
            VerifiedCallback cb) {
        boolean verify = false;
        if (transformedParam.has("pubKey")) {
            LOGGER.info("before verifying: " + toVerifyStr);
            try {
                ECPublicKeyParameters pubKey = BCECUtil.createECPublicKeyFromStrParameters(
                        transformedParam.get("pubKey").getAsString(), SM2Util.CURVE,
                        SM2Util.DOMAIN_PARAMS);
                verify = SM2Util.verify(pubKey, toVerifyStr.getBytes(),
                        ByteUtils.fromHexString(transformedParam.get("sign").getAsString()));

            } catch (Exception e) {
                LOGGER.error(e.getMessage());
                LOGGER.debug(ExceptionUtil.exceptionToString(e));
            }
        }
        if (cb != null)
            cb.onResult(verify, transformedParam);
        return;
    }

    public static JsonObject parsePostAndVerify(FullHttpRequest msg, VerifiedCallback cb)
            throws Exception {
        ByteBuf content = msg.content();
        JsonObject map =
                JsonParser.parseReader(new InputStreamReader(new ByteBufInputStream(content)))
                        .getAsJsonObject();
        StringBuilder toSign = new StringBuilder();
        boolean isFirst = true;
        for (String key : map.keySet()) {
            if (!key.equals("sign")) {
                if (!isFirst)
                    toSign.append("&");
                isFirst = false;
                toSign.append(key).append("=");
                JsonElement je = map.get(key);
                if (je.isJsonPrimitive())
                    toSign.append(je.getAsString());
                else
                    toSign.append(je.toString());
            }
        }
        verifyParam(map, toSign.toString(), cb);
        return map;

    }
}
