package org.bdware.server.http;

import io.netty.buffer.Unpooled;
import io.netty.channel.*;
import io.netty.handler.codec.http.*;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.bdware.server.action.HttpResultCallback;

import java.io.File;
import java.io.RandomAccessFile;

import static io.netty.handler.codec.http.HttpResponseStatus.OK;

public class FileDownloaderCallback extends HttpResultCallback {
    private static final Logger LOGGER = LogManager.getLogger(FileDownloaderCallback.class);
    HttpRequest req;

    public FileDownloaderCallback(ChannelHandlerContext ctx, HttpRequest req) {
        super(ctx, null);
        this.req = req;
    }

    public void onResult(final String filePath) {
        try {
            final RandomAccessFile file = new RandomAccessFile(filePath, "r");
            HttpResponse response =
                    new DefaultHttpResponse(HttpVersion.HTTP_1_1, HttpResponseStatus.OK);
            // 设置文件格式内容
            response.headers().set(HttpHeaderNames.CONTENT_TYPE, "application/x-msdownload");
            response.headers().set(HttpHeaderNames.CONTENT_DISPOSITION,
                    "attachment;filename=" + new File(filePath).getName());
            HttpUtil.setKeepAlive(response, HttpUtil.isKeepAlive(req));
            HttpUtil.setContentLength(response, file.length());
            LOGGER.debug("FileLength:" + file.length());
            ctxField.write(response);
            ChannelFuture future =
                    ctxField.write(new DefaultFileRegion(file.getChannel(), 0, file.length()),
                            ctxField.newProgressivePromise());
            ctxField.writeAndFlush(LastHttpContent.EMPTY_LAST_CONTENT);
            future.addListener(new ChannelProgressiveFutureListener() {
                @Override
                public void operationProgressed(ChannelProgressiveFuture future, long progress,
                        long total) {}

                @Override
                public void operationComplete(ChannelProgressiveFuture future) throws Exception {
                    file.close();
                    LOGGER.debug("delete file " + filePath + ": " + new File(filePath).delete());
                }
            });
            if (!HttpUtil.isKeepAlive(req))
                future.addListener(ChannelFutureListener.CLOSE);
            ctxField.flush();
        } catch (Exception e) {
            DefaultFullHttpResponse response = new DefaultFullHttpResponse(HttpVersion.HTTP_1_1, OK,
                    Unpooled.wrappedBuffer("File Not Found".getBytes()));
            ChannelFuture f = ctxField.writeAndFlush(response);
            f.addListener(ChannelFutureListener.CLOSE);
            ctxField.close();
        }
    }
}
