package org.bdware.server.http;

import io.netty.buffer.Unpooled;
import io.netty.channel.ChannelFuture;
import io.netty.channel.ChannelFutureListener;
import io.netty.channel.ChannelHandlerContext;
import io.netty.channel.SimpleChannelInboundHandler;
import io.netty.handler.codec.http.*;
import io.netty.handler.stream.ChunkedFile;

import java.io.File;
import java.io.FileFilter;
import java.io.RandomAccessFile;

import static io.netty.handler.codec.http.HttpVersion.HTTP_1_1;

public class HttpFileHandleAdapter extends SimpleChannelInboundHandler<FullHttpRequest> {
    private final String location;
    FileFilter fileFilter;

    public HttpFileHandleAdapter(String path, FileFilter fileFilter) {
        location = path;
        this.fileFilter = fileFilter;
    }

    public static void sendError(ChannelHandlerContext ctx, HttpResponseStatus status) {
        FullHttpResponse response = new DefaultFullHttpResponse(HTTP_1_1, status,
                Unpooled.wrappedBuffer(("Failure: " + status + "\r\n").getBytes()));
        response.headers().set(HttpHeaderNames.CONTENT_TYPE, "text/plain; charset=UTF-8");
        ctx.writeAndFlush(response).addListener(ChannelFutureListener.CLOSE);
    }

    public static void appendContentType(String path, HttpHeaders headers) {
        if (path.endsWith(".html")) {
            headers.set(HttpHeaderNames.CONTENT_TYPE, "text/html; charset=UTF-8");
        } else if (path.endsWith(".js")) {
            headers.set(HttpHeaderNames.CONTENT_TYPE, "application/x-javascript; charset=utf-8");
        } else if (path.endsWith(".css")) {
            headers.set(HttpHeaderNames.CONTENT_TYPE, "text/css; charset=UTF-8");
        } else if (path.endsWith(".ico")) {
            headers.set(HttpHeaderNames.CONTENT_TYPE, "image/x-icon;");
        } else if (path.endsWith(".svg")) {
            headers.set(HttpHeaderNames.CONTENT_TYPE, "image/svg+xml;charset=utf-8");
        } else if (path.endsWith(".ypk")) {
            headers.set(HttpHeaderNames.CONTENT_TYPE, "application/ypk");
        }
    }

    private static void send100Continue(ChannelHandlerContext ctx) {
        FullHttpResponse response =
                new DefaultFullHttpResponse(HttpVersion.HTTP_1_1, HttpResponseStatus.CONTINUE);
        ctx.writeAndFlush(response);
    }

    @Override
    protected void channelRead0(ChannelHandlerContext ctx, FullHttpRequest request)
            throws Exception {
        // 获取URI
        request.retain();
        // System.out.println("[HttpFileHandlerAdapter]request.uri:" + request.uri());
        // String[] hoString = request.duplicate().toString().split("\n");
        // if (request.uri().equals("/favicon.ico")) {
        // request.setUri("/SCIDE/favicon.ico");
        // }
        String uri = request.uri();

        if (uri.contains("..")) {
            sendError(ctx, HttpResponseStatus.BAD_REQUEST);
            return;
        }

        uri = uri.replaceFirst("/SCIDE", "");
        uri = uri.replaceFirst("\\?.*$", "");
        if (uri.equals("/") || uri.equals("./") || uri.length() == 0) {
            uri = "/index.html";
        }
        boolean isTrue = isFile(uri);
        // 根据路径地址构建文件
        if (isTrue) {
            String path = location + uri;
            File html = new File(path);
            // 状态为1xx的话，继续请求
            if (HttpUtil.is100ContinueExpected(request)) {
                send100Continue(ctx);
            }
            // 当文件不存在的时候，将资源指向NOT_FOUND
            if (!html.exists() || html.isDirectory()) {
                sendError(ctx, HttpResponseStatus.NOT_FOUND);
                return;
            }
            final RandomAccessFile file = new RandomAccessFile(html, "r");
            HttpResponse response =
                    new DefaultHttpResponse(request.protocolVersion(), HttpResponseStatus.OK);
            // 设置文件格式内容
            appendContentType(path, response.headers());
            boolean keepAlive = HttpUtil.isKeepAlive(request);
            HttpUtil.setContentLength(response, file.length());
            if (keepAlive) {
                HttpUtil.setKeepAlive(response, true);
            }
            ctx.write(response);
            // File对象直接将文件写入到发送缓冲区，最后为sendFileFeature增加GenericFeatureListener，
            ctx.write(new ChunkedFile(file, 0, file.length(), 512 * 1024));
            // 写入文件尾部
            ChannelFuture future = ctx.writeAndFlush(LastHttpContent.EMPTY_LAST_CONTENT);
            if (!keepAlive) {
                future.addListener(ChannelFutureListener.CLOSE);
            }
            future.addListener(arg0 -> file.close());
        } else {
            sendError(ctx, HttpResponseStatus.NOT_FOUND);
        }
    }

    private boolean isFile(String path) {
        return fileFilter.accept(new File(path));
    }
}
