package org.bdware.server.http;

import io.netty.buffer.Unpooled;
import io.netty.channel.ChannelFutureListener;
import io.netty.channel.ChannelHandlerContext;
import io.netty.handler.codec.http.*;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;

import java.lang.reflect.Method;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import static io.netty.handler.codec.http.HttpVersion.HTTP_1_1;

public class URIHandler {
    private static final Logger LOGGER = LogManager.getLogger(URIHandler.class);
    private Map<io.netty.handler.codec.http.HttpMethod, List<Tuple<String, Method, Object>>> handlers;

    public URIHandler() {
        handlers = new HashMap<>();
    }

    public void register(Object obj) {
        register(obj.getClass(), obj);
    }

    public void register(Class<?> clz) {
        register(clz, null);
    }

    public void register(Class<?> clz, Object obj) {
        for (Method m : clz.getDeclaredMethods()) {
            URIPath path = m.getAnnotation(URIPath.class);
            if (path != null) {
                HttpMethod method = path.method();
                List<Tuple<String, Method, Object>> handlerList = getOrCreate(method);
                for (String str : path.value()) {
                    m.setAccessible(true);
                    handlerList.add(new Tuple<>(str, m, obj));
                }
            }
        }
        for (List<Tuple<String, Method, Object>> handlerList : handlers.values()) {
            sortHandlerList(handlerList);
        }
    }

    protected List<Tuple<String, Method, Object>> getOrCreate(HttpMethod method) {
        return handlers.computeIfAbsent(method.get(), k -> new ArrayList<>());
    }

    private void sortHandlerList(List<Tuple<String, Method, Object>> handlerList) {
        handlerList.sort((o1, o2) -> {
            int delta = o2.t.length() - o1.t.length();
            return delta == 0 ? o1.t.compareTo(o2.t) : delta;
        });
    }

    public Tuple<String, Method, Object> findHandler(FullHttpRequest msg) {
        List<Tuple<String, Method, Object>> handlerList = handlers.get(msg.method());
        for (Tuple<String, Method, Object> t : handlerList)
            if (msg.uri().startsWith(t.t)) {
                return t;
            }
        return null;
    }

    public void handle(ChannelHandlerContext ctx, FullHttpRequest msg) {
        Tuple<String, Method, Object> t = findHandler(msg);
        handle(ctx, msg, t);
    }

    public void handle(ChannelHandlerContext ctx, FullHttpRequest msg, Tuple<String, Method, Object> t) {
        try {
            if (t != null)
                t.u.invoke(t.s, ctx, msg);
            else
                sendUnsupported(ctx);
        } catch (Exception e) {
            e.printStackTrace();
        }
    }

    public void printURIHandlers() {
        StringBuilder sbl = new StringBuilder("URIHandlers:\n");
        for (io.netty.handler.codec.http.HttpMethod m : handlers.keySet()) {
            sbl.append("\t").append(m.name()).append("\n");
            for (Tuple<String, Method, Object> t : handlers.get(m)) {
                String className = t.u.getDeclaringClass().getSimpleName();
                sbl.append("\t\t").append(t.t.isEmpty() ? "<null>" : t.t).append(" --> ")
                        .append(className.isEmpty() ? "null" : className).append(".")
                        .append(t.u.getName()).append("\n");
            }
        }
        LOGGER.info(sbl.substring(0, sbl.length() - 1));
    }

    private void sendUnsupported(ChannelHandlerContext ctx) {
        FullHttpResponse response = new DefaultFullHttpResponse(HTTP_1_1,
                HttpResponseStatus.BAD_REQUEST, Unpooled.wrappedBuffer(
                ("Failure: " + HttpResponseStatus.BAD_REQUEST + "\r\n").getBytes()));
        response.headers().set(HttpHeaderNames.CONTENT_TYPE, "text/plain; charset=UTF-8");
        ctx.writeAndFlush(response).addListener(ChannelFutureListener.CLOSE);
    }

    public static class Tuple<T, U, S> {
        public T t;
        public U u;
        public S s;

        Tuple(T t, U u, S s) {
            this.t = t;
            this.u = u;
            this.s = s;
        }
    }
}
