/*
 * Beangle, Agile Development Scaffold and Toolkit
 *
 * Copyright (c) 2005-2016, Beangle Software.
 *
 * Beangle is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * Beangle is distributed in the hope that it will be useful.
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with Beangle.  If not, see <http://www.gnu.org/licenses/>.
 */
package org.beangle.cdi.spring.web

import org.beangle.commons.collection.Collections
import org.beangle.commons.lang.ClassLoaders.{ getResources, load }
import org.beangle.commons.lang.Strings.{ substringAfter, substringBefore }
import org.beangle.commons.lang.reflect.Reflections.newInstance
import org.beangle.commons.logging.Logging
import org.beangle.cdi.spring.context.{ BeanFactoryLoader, ContextLoader }

import javax.servlet.{ ServletContextEvent, ServletContextListener }

/**
 * 1. Disable Definition Overriding
 * 2. Default config location(spring-context.xml)
 * 3. Load children context
 */
class ContextListener extends ServletContextListener with Logging {

  var loaders = Collections.newBuffer[ContextLoader]

  val springContextAvaliable = !getResources("org/springframework/context/support/AbstractApplicationContext.class").isEmpty

  override def contextInitialized(sce: ServletContextEvent) {
    val sc = sce.getServletContext
    val contextClass = sc.getInitParameter("contextClass")
    var configLocation = sc.getInitParameter("contextConfigLocation")
    if (null == configLocation) configLocation = "classpath:spring-context.xml"
    val root = newLoader().load("WebApplicationContext:ROOT", contextClass, configLocation, null)

    //load children
    val childLocation = sc.getInitParameter("childContextConfigLocation")
    if (null != childLocation) {
      newLoader().load(substringBefore(childLocation, "@"), contextClass, substringAfter(childLocation, "@"), root)
    }
  }

  override def contextDestroyed(sce: ServletContextEvent) {
    loaders.foreach { loader => loader.close() }
  }

  private def newLoader(): ContextLoader = {
    val loader =
      if (springContextAvaliable) newInstance(load("org.beangle.cdi.spring.context.ApplicationContextLoader")).asInstanceOf[ContextLoader]
      else new BeanFactoryLoader()
    loaders += loader
    loader
  }
}
