/* ********************************************************************
    Licensed to Jasig under one or more contributor license
    agreements. See the NOTICE file distributed with this work
    for additional information regarding copyright ownership.
    Jasig licenses this file to you under the Apache License,
    Version 2.0 (the "License"); you may not use this file
    except in compliance with the License. You may obtain a
    copy of the License at:

    http://www.apache.org/licenses/LICENSE-2.0

    Unless required by applicable law or agreed to in writing,
    software distributed under the License is distributed on
    an "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
    KIND, either express or implied. See the License for the
    specific language governing permissions and limitations
    under the License.
*/

package org.bedework.calfacade.mail;

import org.bedework.calfacade.BwCalendar;
import org.bedework.calfacade.BwPrincipal;

import net.fortuna.ical4j.model.Calendar;

import java.io.Serializable;
import java.util.Collection;

/** From within the calendar system we have the following mailing needs:
 *
 * <ul><li>When an event is added/changed/deleted mail the chnage to all
 * subscribers to that events calendar.</li>
 *
 * <li>When an email alarm fires, mail something, including the event, to the
 * alarm receiver</li>
 * </ul>
 *
 * <p>Over time other needs may present themselves.</p>
 *
 * <p>The subscribed case can be thought of as a mailing list. For each calendar,
 * we have a list of subscribers who want events emailed. When a change occurs to
 * the calendar we send the message out on the list. This interface allows
 * implementors to use a list server to handle the, possibly large amount of,
 * mail generated by this system.
 *
 * <p>The second case is less easily optimized. Each email is unique in content
 * and usually has only one recipient.
 *
 * <p>In both cases it is useful to have some sort of mechanism for determining
 * bad addresses. List servers often have their own mechanisms which usually
 * result in a recipient being eventually deleted from the list. Thsi interface
 * allows the calendar system to query the state of a given recipient on the
 * list.
 *
 * <p>Because we are probably interfacing to a system which identifies users
 * solely by their email-address, we have to give that system the chance to
 * update itself correctly when the users email is changed. We present the
 * new address to the system as well as the old one.
 *
 * @author Mike Douglass douglm@bedework.edu
 */
public interface MailerIntf extends Serializable {
  /**
   * @param config MailConfigProperties object
   */
  void init(MailConfigProperties config);

  /** Mail the given calendar object to the given list of recipients
   *
   * @param cal   The rfc2445,6,7 object
   * @param originator - null if we use configured default
   * @param recipients   list of recipients
   * @param subject
   * @return boolean     true if message sent, false - probably disabled.
   */
  boolean mailEntity(Calendar cal,
                     String originator,
                     Collection<String>recipients,
                     String subject);

  /** Add a list corresponding to the given calendar.
   *
   * @param cal
   */
  void addList(BwCalendar cal);

  /** Delete a list corresponding to the given calendar.
   *
   * @param cal
   */
  void deleteList(BwCalendar cal);

  /** Return a collection of mail list ids
   *
   * @return collection of mail list ids
   */
  Collection<String> listLists();

  /** Check a list corresponding to the given calendar exists.
   *
   * @param cal
   * @return true if list exists
   */
  boolean checkList(BwCalendar cal);

  /** Post a  message to the list corresponding to the given calendar.
   *
   * @param cal
   * @param val
   */
  void postList(BwCalendar cal, Message val);

  /** Add a member to the list corresponding to the given calendar.
   *
   * @param cal
   * @param member
   */
  public void addMember(BwCalendar cal, BwPrincipal<?> member);

  /** Remove a member from the list corresponding to the given calendar.
   *
   * @param cal
   * @param member
   */
  public void removeMember(BwCalendar cal, BwPrincipal<?> member);

  /** Check a member is on the list corresponding to the given calendar.
   *
   * @param cal
   * @param member
   * @return boolean
   */
  boolean checkMember(BwCalendar cal, BwPrincipal<?> member);

  /** Update a members email address on the list corresponding to the given calendar.
   *
   * @param cal
   * @param member
   * @param newEmail
   */
  void updateMember(BwCalendar cal, BwPrincipal<?> member, String newEmail)
       ;

  /** List members on the list corresponding to the given calendar. This requires
   * that the implementation has access to the database to match emails against
   * members.
   *
   * @param cal
   * @return Collection
   */
  Collection<BwPrincipal<?>> listMembers(BwCalendar cal);

  /**
   * @param val
   */
  void post(Message val);
}

