/*
BSD 2-Clause License

Copyright (c) 2019, Beigesoft™
All rights reserved.

Redistribution and use in source and binary forms, with or without
modification, are permitted provided that the following conditions are met:

* Redistributions of source code must retain the above copyright notice, this
  list of conditions and the following disclaimer.

* Redistributions in binary form must reproduce the above copyright notice,
  this list of conditions and the following disclaimer in the documentation
  and/or other materials provided with the distribution.

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE
FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

package org.beigesoft.ws.srv;

import java.util.Date;
import java.util.List;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Map;
import java.util.HashMap;
import java.math.BigDecimal;
import java.io.IOException;
import java.io.InputStream;
import java.net.URL;

import org.beigesoft.exc.ExcCode;
import org.beigesoft.mdl.IReqDt;
import org.beigesoft.mdl.ColVals;
import org.beigesoft.log.ILog;
import org.beigesoft.rdb.IOrm;
import org.beigesoft.rdb.IRdb;
import org.beigesoft.rdb.SrvClVl;
import org.beigesoft.ws.mdl.EOrdStat;
import org.beigesoft.ws.mdl.EPaymMth;
import org.beigesoft.ws.mdl.Purch;
import org.beigesoft.ws.mdlp.SeSel;
import org.beigesoft.ws.mdlp.SeSerBus;
import org.beigesoft.ws.mdlp.SerBus;
import org.beigesoft.ws.mdlp.SeItmPlc;
import org.beigesoft.ws.mdlp.ItmPlc;
import org.beigesoft.ws.mdlp.SrvPlc;
import org.beigesoft.ws.mdlp.SeSrvPlc;
import org.beigesoft.ws.mdlp.Buyer;
import org.beigesoft.ws.mdlp.CuOrSe;
import org.beigesoft.ws.mdlp.CuOrSeSrLn;
import org.beigesoft.ws.mdlp.CuOrSeGdLn;
import org.beigesoft.ws.mdlp.CuOr;
import org.beigesoft.ws.mdlp.CuOrGdLn;
import org.beigesoft.ws.mdlp.CuOrSrLn;
import org.beigesoft.ws.mdlp.AddStg;
import org.beigesoft.ws.mdlp.TrdStg;
import org.beigesoft.ws.mdlp.Itlist;

/**
 * <p>It accepts all buyer's orders.
 * It changes item's availability and orders status to PENDING.
 * If any item is unavailable, then it throws exception.
 * And so does if there are several payees for online payment.</p>
 *
 * @param <RS> platform dependent RDBMS recordset
 * @author Yury Demidenko
 */
public class AcpOrd<RS> implements IAcpOrd {

  /**
   * <p>Logger.</p>
   **/
  private ILog log;

  /**
   * <p>ORM service.</p>
   */
  private IOrm orm;

  /**
   * <p>DB service.</p>
   */
  private IRdb<RS> rdb;

  /**
   * <p>Column values service.</p>
   **/
  private SrvClVl srvClVl;

  /**
   * <p>Android configuration, RDB insert returns autogenerated ID,
   * updating with expression like "VER=VER+1" is not possible.</p>
   **/
  private boolean isAndr;

  //Cached queries:
  /**
   * <p>Query goods availability checkout.</p>
   **/
  private String quOrGdChk;

  /**
   * <p>Query services availability checkout.</p>
   **/
  private String quOrSrChk;

  /**
   * <p>It accepts all buyer's orders.
   * It changes item's availability and orders status to PENDING.
   * If any item is unavailable, then it throws exception.
   * And so does if there are several payees for online payment.</p>
   * @param pRvs additional request scoped parameters
   * @param pReqDt Request Data
   * @param pBur Buyer
   * @return list of accepted orders
   * @throws Exception - an exception
   **/
  @Override
  public final Purch accept(final Map<String, Object> pRvs,
    final IReqDt pReqDt, final Buyer pBur) throws Exception {
    Purch rez = null;
    AddStg setAdd = (AddStg) pRvs.get("tastg");
    List<CuOr> ords = null;
    List<CuOrSe> sords = null;
    Map<String, Object> vs = new HashMap<String, Object>();
    String wheStBr = "where STAS=0 and BUYR=" + pBur.getIid();
    String[] ndFlNm = new String[] {"nme"};
    vs.put("PicPlcndFds", ndFlNm);
    vs.put("BuyerdpLv", 0);
    ords = this.orm.retLstCnd(pRvs, vs, CuOr.class, wheStBr); vs.clear();
    vs.put("SeSelndFds", new String[] {"dbcr"});
    vs.put("DbCrndFds", ndFlNm);
    vs.put("SeSeldpLv", 2);
    vs.put("buyerdpLv", 0);
    sords = this.orm.retLstCnd(pRvs, vs, CuOrSe.class, wheStBr); vs.clear();
    if (ords.size() == 0 && sords.size() == 0) {
      throw new ExcCode(ExcCode.SPAM, "There is no new orders for buyer ID: "
        + pBur.getIid());
    }
    if (setAdd.getOnlMd() == 0 && sords.size() > 0) {
      //checking for several online payees:
      boolean isOwnOnlPay = false;
      for (CuOr co : ords) {
        if (co.getPaym().equals(EPaymMth.ONLINE)
          || co.getPaym().equals(EPaymMth.PARTIAL_ONLINE)
            || co.getPaym().equals(EPaymMth.PAYPAL)
              || co.getPaym().equals(EPaymMth.PAYPAL_ANY)) {
          isOwnOnlPay = true;
          break;
        }
      }
      SeSel selOnl = null;
      for (CuOrSe co : sords) {
        if (co.getPaym().equals(EPaymMth.ONLINE)
          || co.getPaym().equals(EPaymMth.PARTIAL_ONLINE)
            || co.getPaym().equals(EPaymMth.PAYPAL)
              || co.getPaym().equals(EPaymMth.PAYPAL_ANY)) {
          if (isOwnOnlPay) {
            throw new Exception("Several online payee for buyer#"
              + pBur.getIid());
          } else if (selOnl == null) {
            selOnl = co.getSelr();
          } else if (!selOnl.getIid().getIid()
            .equals(co.getSelr().getIid().getIid())) {
            throw new Exception("Several online S.E.Payee for buyer#"
              + pBur.getIid());
          }
        }
      }
    }
    //consolidated order with bookable items for farther booking:
    if (ords.size() > 0) {
      CuOr cor = check1(pRvs, ords);
      adChekBook(pRvs, cor);
    }
    if (sords.size() > 0) {
      CuOrSe cor = checkSe1(pRvs, sords);
      adChekBookSe(pRvs, cor);
    }
    //change orders status:
    if (this.isAndr) {
      String[] ndFds = new String[] {"ver", "stas"};
      Arrays.sort(ndFds);
      vs.put("ndFds", ndFds);
      for (CuOr co : ords) {
        co.setStas(EOrdStat.BOOKED);
        getOrm().update(pRvs, vs, co);
      }
      for (CuOrSe co : sords) {
        co.setStas(EOrdStat.BOOKED);
        getOrm().update(pRvs, vs, co);
      }
      vs.clear();
    } else {
      ColVals cvs = new ColVals();
      this.srvClVl.put(cvs, "ver", new Date().getTime());
      this.srvClVl.put(cvs, "stas", EOrdStat.BOOKED.ordinal());
      this.rdb.update(CuOr.class, cvs, "STAS=0 and BUYR=" + pBur.getIid());
      this.rdb.update(CuOrSe.class, cvs, "STAS=0 and BUYR=" + pBur.getIid());
    }
    rez = new Purch();
    if (ords.size() > 0) {
      rez.setOrds(ords);
    }
    if (sords.size() > 0) {
      rez.setSords(sords);
    }
    return rez;
  }

  //utils:
  /**
   * <p>It half-checks items. Order can has items from several places.</p>
   * @param pRvs additional request scoped parameters
   * @param pOrds S.E. orders
   * @return consolidated order with bookable items
   * @throws Exception - an exception if checking fail
   **/
  public final CuOrSe checkSe1(final Map<String, Object> pRvs,
    final List<CuOrSe> pOrds) throws Exception {
    StringBuffer ordIds = null;
    for (CuOrSe co : pOrds) {
      co.setGoods(new ArrayList<CuOrSeGdLn>());
      co.setServs(new ArrayList<CuOrSeSrLn>());
      if (ordIds == null) {
        ordIds = new StringBuffer();
        ordIds.append(co.getIid().toString());
      } else {
        ordIds.append("," + co.getIid());
      }
    }
    Map<String, Object> vs = new HashMap<String, Object>();
    String[] ndFlGd = new String[] {"ownr", "nme", "good", "uom", "quan", "pri",
      "tot", "toTx"};
    Arrays.sort(ndFlGd);
    String[] ndFlNm = new String[] {"nme"};
    vs.put("CuOrSeGdLnndFds", ndFlGd);
    vs.put("UomndFds", ndFlNm);
    vs.put("SeItmdpLv", 0);
    vs.put("CuOrSedpLv", 0);
    List<CuOrSeGdLn> allGoods = new ArrayList<CuOrSeGdLn>();
    List<CuOrSeSrLn> allServs = new ArrayList<CuOrSeSrLn>();
    TrdStg ts = (TrdStg) pRvs.get("tstg");
    List<CuOrSeGdLn> allGds;
    if (ts.getAi18n()) {
      String quer = lazyGetQuOrGdChk().replace(":TORLN", "CUORSEGDLN")
        .replace(":TITPL", "SEITMPLC").replace(":ORIDS", ordIds.toString());
      allGds = this.orm.retLstQu(pRvs, vs, CuOrSeGdLn.class, quer);
    } else {
      allGds = this.orm.retLstCnd(pRvs, vs, CuOrSeGdLn.class,
        "where OWNR in (:ORIDS)".replace(":ORIDS", ordIds.toString()));
    }
    vs.clear();
    for (CuOrSeGdLn gl : allGds) {
      if (gl.getQuan().compareTo(BigDecimal.ZERO) == 0) {
        this.log.error(pRvs, getClass(), "S.E.Good is not available #"
          + gl.getGood().getIid());
        throw new ExcCode(ExcCode.WRPR, "THERE_IS_NO_GOODS");
      }
    }
    for (CuOrSeGdLn gl : allGds) {
      for (CuOrSe co : pOrds) {
        if (co.getIid().equals(gl.getOwnr().getIid())) {
          gl.setOwnr(co);
          co.getGoods().add(gl);
        }
      }
      CuOrSeGdLn cgl = new CuOrSeGdLn();
      cgl.setIid(gl.getIid());
      cgl.setGood(gl.getGood());
      cgl.setQuan(gl.getQuan());
      allGoods.add(cgl);
    }
    String[] ndFlSr = new String[] {"ownr", "nme", "srv", "uom", "quan", "pri",
      "tot", "toTx", "dt1", "dt2"};
    Arrays.sort(ndFlSr);
    vs.put("CuOrSeSrLnndFds", ndFlSr);
    vs.put("SeSrvdpLv", 0);
    vs.put("UomndFds", ndFlNm);
    vs.put("CuOrSedpLv", 0);
    //non-bookable service checkout and bookable services half-checkout:
    List<CuOrSeSrLn> allSrvs;
    if (ts.getAi18n()) {
      String quer = lazyGetQuOrSrChk().replace(":TORLN", "CUORSESRLN")
        .replace(":TITPL", "SESRVPLC").replace(":ORIDS", ordIds.toString());
      allSrvs = this.orm.retLstQu(pRvs, vs, CuOrSeSrLn.class, quer);
    } else {
      allSrvs = this.orm.retLstCnd(pRvs, vs, CuOrSeSrLn.class,
        "where OWNR in (:ORIDS)".replace(":ORIDS", ordIds.toString()));
    }
    vs.clear();
    for (CuOrSeSrLn sl : allSrvs) {
      if (sl.getQuan().compareTo(BigDecimal.ZERO) == 0) {
        this.log.error(pRvs, getClass(), "S.E.Service is not available #"
          + sl.getSrv().getIid());
        throw new ExcCode(ExcCode.WRPR, "serv_no_avlb");
      }
    }
    for (CuOrSeSrLn sl : allSrvs) {
      for (CuOrSe co : pOrds) {
        if (co.getIid().equals(sl.getOwnr().getIid())) {
          sl.setOwnr(co);
          co.getServs().add(sl);
        }
      }
      CuOrSeSrLn csl = new CuOrSeSrLn();
      csl.setIid(sl.getIid());
      csl.setSrv(sl.getSrv());
      csl.setQuan(sl.getQuan());
      csl.setDt1(sl.getDt1());
      csl.setDt2(sl.getDt2());
      allServs.add(csl);
    }
    CuOrSe cor = new CuOrSe();
    cor.setGoods(allGoods);
    cor.setServs(allServs);
    return cor;
  }

  /**
   * <p>It half-checks items. Order can has items from several  places.</p>
   * @param pRvs additional request scoped parameters
   * @param pOrds orders
   * @return consolidated order with bookable items
   * @throws Exception - an exception if checking fail
   **/
  public final CuOr check1(final Map<String, Object> pRvs,
    final List<CuOr> pOrds) throws Exception {
    StringBuffer ordIds = null;
    for (CuOr co : pOrds) {
      co.setGoods(new ArrayList<CuOrGdLn>());
      co.setServs(new ArrayList<CuOrSrLn>());
      if (ordIds == null) {
        ordIds = new StringBuffer();
        ordIds.append(co.getIid().toString());
      } else {
        ordIds.append("," + co.getIid());
      }
    }
    Map<String, Object> vs = new HashMap<String, Object>();
    String[] ndFlGd = new String[] {"ownr", "nme", "good", "uom", "quan", "pri",
      "tot", "toTx"};
    Arrays.sort(ndFlGd);
    String[] ndFlNm = new String[] {"nme"};
    vs.put("CuOrGdLnndFds", ndFlGd);
    vs.put("ItmdpLv", 0);
    vs.put("CuOrdpLv", 0);
    vs.put("UomndFds", ndFlNm);
    List<CuOrGdLn> allGoods = new ArrayList<CuOrGdLn>();
    List<CuOrSrLn> allServs = new ArrayList<CuOrSrLn>();
    TrdStg ts = (TrdStg) pRvs.get("tstg");
    List<CuOrGdLn> allGds;
    if (ts.getAi18n()) {
      String quer = lazyGetQuOrGdChk().replace(":TORLN", "CUORGDLN")
        .replace(":TITPL", "ITMPLC").replace(":ORIDS", ordIds.toString());
      allGds = this.orm.retLstQu(pRvs, vs, CuOrGdLn.class, quer);
    } else {
      allGds = this.orm.retLstCnd(pRvs, vs, CuOrGdLn.class,
        "where OWNR in (:ORIDS)".replace(":ORIDS", ordIds.toString()));
    }
    vs.clear();
    for (CuOrGdLn gl : allGds) {
      if (gl.getQuan().compareTo(BigDecimal.ZERO) == 0) {
        this.log.error(pRvs, getClass(), "Good is not available #"
          + gl.getGood().getIid());
        throw new ExcCode(ExcCode.WRPR, "THERE_IS_NO_GOODS");
      }
    }
    for (CuOrGdLn gl : allGds) {
      for (CuOr co : pOrds) {
        if (co.getIid().equals(gl.getOwnr().getIid())) {
          gl.setOwnr(co);
          co.getGoods().add(gl);
        }
      }
      CuOrGdLn cgl = new CuOrGdLn();
      cgl.setIid(gl.getIid());
      cgl.setGood(gl.getGood());
      cgl.setQuan(gl.getQuan());
      allGoods.add(cgl);
    }
    String[] ndFlSr = new String[] {"ownr", "nme", "srv", "uom", "quan", "pri",
      "tot", "toTx", "dt1", "dt2"};
    Arrays.sort(ndFlSr);
    vs.put("CuOrSrLnndFds", ndFlSr);
    vs.put("UomndFds", ndFlNm);
    vs.put("SrvdpLv", 0);
    vs.put("CuOrdpLv", 0);
    List<CuOrSrLn> allSrvs;
    if (ts.getAi18n()) {
      //non-bookable service checkout and bookable services half-checkout:
      String quer = lazyGetQuOrSrChk().replace(":TORLN", "CUORSRLN")
        .replace(":TITPL", "SRVPLC").replace(":ORIDS", ordIds.toString());
      allSrvs = this.orm.retLstQu(pRvs, vs, CuOrSrLn.class, quer);
    } else {
      allSrvs = this.orm.retLstCnd(pRvs, vs, CuOrSrLn.class,
        "where OWNR in (:ORIDS)".replace(":ORIDS", ordIds.toString()));
    }
    vs.clear();
    for (CuOrSrLn sl : allSrvs) {
      if (sl.getQuan().compareTo(BigDecimal.ZERO) == 0) {
        this.log.error(pRvs, getClass(), "Service is not available #"
          + sl.getSrv().getIid());
        throw new ExcCode(ExcCode.WRPR, "serv_no_avlb");
      }
    }
    for (CuOrSrLn sl : allSrvs) {
      for (CuOr co : pOrds) {
        if (co.getIid().equals(sl.getOwnr().getIid())) {
          sl.setOwnr(co);
          co.getServs().add(sl);
        }
      }
      CuOrSrLn csl = new CuOrSrLn();
      csl.setIid(sl.getIid());
      csl.setSrv(sl.getSrv());
      csl.setQuan(sl.getQuan());
      csl.setDt1(sl.getDt1());
      csl.setDt2(sl.getDt2());
      allServs.add(csl);
    }
    CuOr cor = new CuOr();
    cor.setGoods(allGoods);
    cor.setServs(allServs);
    return cor;
  }

  /**
   * <p>It checks additionally and books S.E. items.</p>
   * @param pRvs additional request scoped parameters
   * @param pCoOr consolidate order
   * @throws Exception - an exception if incomplete
   **/
  public final void adChekBookSe(final Map<String, Object> pRvs,
    final CuOrSe pCoOr) throws Exception {
    //additional checking:
    //check availability and booking for same good in different orders:
    Map<String, Object> vs = new HashMap<String, Object>();
    List<CuOrSeGdLn> gljs = null;
    List<CuOrSeGdLn> glrs = null;
    for (CuOrSeGdLn gl : pCoOr.getGoods()) {
      //join lines with same item:
      for (CuOrSeGdLn gl0 : pCoOr.getGoods()) {
        if (!gl.getIid().equals(gl0.getIid())
          && gl.getGood().getIid().equals(gl0.getGood().getIid())) {
          if (gljs == null) {
            gljs = new ArrayList<CuOrSeGdLn>();
            glrs = new ArrayList<CuOrSeGdLn>();
          }
          glrs.add(gl0);
          if (!gljs.contains(gl)) {
            gljs.add(gl);
          }
          gl.setQuan(gl.getQuan().add(gl0.getQuan()));
        }
      }
    }
    if (gljs != null) {
      for (CuOrSeGdLn glr : glrs) {
        pCoOr.getGoods().remove(glr);
      }
      vs.put("SeItmPlcdpLv", 0); //only ID
      for (CuOrSeGdLn gl : gljs) {
        List<SeItmPlc> gps = getOrm().retLstCnd(pRvs, vs, SeItmPlc.class,
          "where ITM=" + gl.getGood().getIid() + " and QUAN>=" + gl.getQuan());
        if (gps.size() == 0) {
          this.log.error(pRvs, getClass(), "S.E.Good is not available #"
            + gl.getGood().getIid());
          throw new ExcCode(ExcCode.WRPR, "THERE_IS_NO_GOODS");
        }
      }
      vs.clear();
    }
    //bookable services final-checkout:
    String cond;
    vs.put("SeSrvPlcdpLv", 0); //only ID
    for (CuOrSeSrLn sl : pCoOr.getServs()) {
      if (sl.getDt1() != null) {
      cond = "left join (select distinct SRV from SESERBUS where FRE=0 and SRV="
        + sl.getSrv().getIid() + " and FRTM>=" + sl.getDt1().getTime()
          + " and TITM<" + sl.getDt2().getTime()
            + ") as SERBUS on SERBUS.SRV=SESRVPLC.ITM where ITM=" + sl.getSrv()
              + " and QUAN>0 and SERBUS.SRV is null";
        List<SeSrvPlc> sps = getOrm().retLstCnd(pRvs, vs, SeSrvPlc.class, cond);
        if (sps.size() == 0) {
          this.log.error(pRvs, getClass(), "BK.SeService is not available #"
            + sl.getSrv().getIid());
          throw new ExcCode(ExcCode.WRPR, "serv_no_avlb");
        }
      }
    }
    vs.clear();
    //booking:
    //changing availability (booking):
    ColVals cvsIil = null;
    String[] verQuan = new String[] {"ver", "quan"};
    Arrays.sort(verQuan);
    if (!this.isAndr) {
      cvsIil = new ColVals();
      this.srvClVl.putExpr(cvsIil, "quan");
    }
    for (CuOrSeGdLn gl : pCoOr.getGoods()) {
      vs.put("SeItmPlcndFds", verQuan);
      vs.put("SeItmPlcdpLv", 1);
      List<SeItmPlc> gps = getOrm().retLstCnd(pRvs, vs, SeItmPlc.class,
        "where ALWAY=0 and ITM=" + gl.getGood().getIid()); vs.clear();
      if (gps.size() != 0) {
        BigDecimal avQu = BigDecimal.ZERO;
        for (SeItmPlc gp : gps) {
          avQu = avQu.add(gp.getQuan());
        }
        if (avQu.compareTo(gl.getQuan()) == -1) {
          //previous test should not be passed!!!
          this.log.error(pRvs, getClass(), "S.E.Good is not available #"
            + gl.getGood().getIid());
          throw new ExcCode(ExcCode.WRPR, "THERE_IS_NO_GOODS");
        }
        BigDecimal rst = gl.getQuan();
        for (SeItmPlc gp : gps) {
          if (rst.compareTo(BigDecimal.ZERO) == 0) {
            break;
          }
          if (gp.getQuan().compareTo(gl.getQuan()) == -1) {
            rst = rst.subtract(gp.getQuan());
            gp.setQuan(BigDecimal.ZERO);
          } else {
            gp.setQuan(gp.getQuan().subtract(rst));
            rst = BigDecimal.ZERO;
          }
          //TODO PERFORM fastupd
          vs.put("ndFds", verQuan);
          getOrm().update(pRvs, vs, gp); vs.clear();
        }
        String wheTyId = "TYP=2 and ITID=" + gl.getGood().getIid();
        if (!this.isAndr) {
          //it must be RDBMS constraint: "quan>=0"!
          this.srvClVl.put(cvsIil, "ver", new Date().getTime());
          this.srvClVl.put(cvsIil, "quan",  "QUAN-" + gl.getQuan());
          try {
            this.rdb.update(Itlist.class, cvsIil, wheTyId);
          } catch (Exception e) {
            this.log.error(pRvs, getClass(), "Itlist NA SeItm: itId/quan: "
              + gl.getGood().getIid() + "/" + gl.getQuan());
            throw new ExcCode(ExcCode.WRPR, "THERE_IS_NO_GOODS");
          }
        } else {
          vs.put("ItlistndFds", verQuan);
          Itlist iil = this.orm.retEntCnd(pRvs, vs, Itlist.class, wheTyId);
          vs.clear();
          BigDecimal aq = iil.getQuan().subtract(gl.getQuan());
          if (aq.compareTo(BigDecimal.ZERO) == -1) {
    this.log.error(pRvs, getClass(), "Itlist NA SeItm: id/itId/avQua/quan: "
  + iil.getIid() + "/" + gl.getGood().getIid() + "/"
+ iil.getQuan() + "/" + gl.getQuan());
            throw new ExcCode(ExcCode.WRPR, "THERE_IS_NO_GOODS");
          } else {
            vs.put("ndFds", verQuan);
            iil.setQuan(aq);
            getOrm().update(pRvs, vs, iil); vs.clear();
          }
        }
      }
    }
    boolean tibs = false;
    for (CuOrSeSrLn sl : pCoOr.getServs()) {
      if (sl.getDt1() == null) {
        vs.put("SeSrvPlcndFds", verQuan);
        vs.put("SeSrvPlcdpLv", 1);
        List<SeSrvPlc> sps = getOrm().retLstCnd(pRvs, vs, SeSrvPlc.class,
          "where ALWAY=0 and ITM=" + sl.getSrv().getIid()); vs.clear();
        if (sps.size() != 0) {
          BigDecimal avQu = BigDecimal.ZERO;
          for (SeSrvPlc sp : sps) {
            avQu = avQu.add(sp.getQuan());
          }
          if (avQu.compareTo(sl.getQuan()) == -1) {
            //previous test should not be passed!!!
           this.log.error(pRvs, getClass(), "S.E.Service is not available #"
              + sl.getSrv().getIid());
            throw new ExcCode(ExcCode.WRPR, "serv_no_avlb");
          }
          BigDecimal rst = sl.getQuan();
          for (SeSrvPlc sp : sps) {
            if (rst.compareTo(BigDecimal.ZERO) == 0) {
              break;
            }
            if (sp.getQuan().compareTo(sl.getQuan()) == -1) {
              rst = rst.subtract(sp.getQuan());
              sp.setQuan(BigDecimal.ZERO);
            } else {
              sp.setQuan(sp.getQuan().subtract(rst));
              rst = BigDecimal.ZERO;
            }
            //TODO PERFORM fastupd
            vs.put("ndFds", verQuan);
            getOrm().update(pRvs, vs, sp); vs.clear();
          }
          String wheTyId = "TYP=3 and ITID=" + sl.getSrv().getIid();
          if (!this.isAndr) {
            this.srvClVl.put(cvsIil, "ver", new Date().getTime());
            this.srvClVl.put(cvsIil, "quan",  "QUAN-" + sl.getQuan());
            try {
              this.rdb.update(Itlist.class, cvsIil, wheTyId);
            } catch (Exception e) {
              this.log.error(pRvs, getClass(), "Itlist NA SESRV: itId/quan: "
                + sl.getSrv().getIid() + "/" + sl.getQuan());
              throw new ExcCode(ExcCode.WRPR, "serv_no_avlb");
            }
          } else {
            vs.put("ItlistndFds", verQuan);
            Itlist iil = this.orm.retEntCnd(pRvs, vs, Itlist.class, wheTyId);
            vs.clear();
            BigDecimal aq = iil.getQuan().subtract(sl.getQuan());
            if (aq.compareTo(BigDecimal.ZERO) == -1) {
        this.log.error(pRvs, getClass(), "Itlist NA SESRV: id/itId/avQua/quan: "
      + iil.getIid() + "/" + sl.getSrv().getIid() + "/"
    + iil.getQuan() + "/" + sl.getQuan());
              throw new ExcCode(ExcCode.WRPR, "serv_no_avlb");
            } else {
              vs.put("ndFds", verQuan);
              iil.setQuan(aq);
              getOrm().update(pRvs, vs, iil); vs.clear();
            }
          }
        }
      } else {
        tibs = true;
      }
    }
    if (tibs) {
      pRvs.put("SeSerBusndFds", new String[] {"ver"});
      List<SeSerBus> sbas = this.orm.retLstCnd(pRvs, vs, SeSerBus.class,
        "where FRE=1"); vs.clear();
      int i = 0;
      for (CuOrSeSrLn sl : pCoOr.getServs()) {
        if (sl.getDt1() != null) {
          SeSerBus sb;
          if (i < sbas.size()) {
            sb = sbas.get(i);
            sb.setFre(false);
          } else {
            sb = new SeSerBus();
          }
          sb.setSrv(sl.getSrv());
          sb.setFrTm(sl.getDt1());
          sb.setTiTm(sl.getDt2());
          if (i < sbas.size()) {
            getOrm().update(pRvs, vs, sb);
            i++;
          } else {
            getOrm().insIdLn(pRvs, vs, sb);
          }
        }
      }
    }
  }

  /**
   * <p>It checks additionally and books items.</p>
   * @param pRvs additional request scoped parameters
   * @param pCoOr consolidate order
   * @throws Exception - an exception if incomplete
   **/
  public final void adChekBook(final Map<String, Object> pRvs,
    final CuOr pCoOr) throws Exception {
    //additional checking:
    //check availability and booking for same good in different orders:
    Map<String, Object> vs = new HashMap<String, Object>();
    List<CuOrGdLn> gljs = null;
    List<CuOrGdLn> glrs = null;
    for (CuOrGdLn gl : pCoOr.getGoods()) {
      //join lines with same item:
      for (CuOrGdLn gl0 : pCoOr.getGoods()) {
        if (!gl.getIid().equals(gl0.getIid())
          && gl.getGood().getIid().equals(gl0.getGood().getIid())) {
          if (gljs == null) {
            gljs = new ArrayList<CuOrGdLn>();
            glrs = new ArrayList<CuOrGdLn>();
          }
          glrs.add(gl0);
          if (!gljs.contains(gl)) {
            gljs.add(gl);
          }
          gl.setQuan(gl.getQuan().add(gl0.getQuan()));
        }
      }
    }
    if (gljs != null) {
      for (CuOrGdLn glr : glrs) {
        pCoOr.getGoods().remove(glr);
      }
      vs.put("ItmPlcdpLv", 0); //only ID
      for (CuOrGdLn gl : gljs) {
        List<ItmPlc> gps = getOrm().retLstCnd(pRvs, vs, ItmPlc.class,
          "where ITM=" + gl.getGood().getIid() + " and QUAN>=" + gl.getQuan());
        if (gps.size() == 0) {
          this.log.error(pRvs, getClass(), "Good is not available #"
            + gl.getGood().getIid());
          throw new ExcCode(ExcCode.WRPR, "THERE_IS_NO_GOODS");
        }
      }
      vs.clear();
    }
    //bookable services final-checkout:
    String cond;
    vs.put("SrvPlcdpLv", 0); //only ID
    for (CuOrSrLn sl : pCoOr.getServs()) {
      if (sl.getDt1() != null) {
        cond = "left join (select distinct SRV from SERBUS where FRE=0 and SRV="
          + sl.getSrv().getIid() + " and FRTM>=" + sl.getDt1().getTime()
            + " and TITM<" + sl.getDt2().getTime()
              + ") as SERBUS on SERBUS.SRV=SRVPLC.ITM where ITM=" + sl.getSrv()
                + " and QUAN>0 and SERBUS.SRV is null";
        List<SrvPlc> sps = getOrm().retLstCnd(pRvs, vs, SrvPlc.class, cond);
        if (sps.size() == 0) {
          this.log.error(pRvs, getClass(), "BK.Service is not available #"
            + sl.getSrv().getIid());
          throw new ExcCode(ExcCode.WRPR, "serv_no_avlb");
        }
      }
    }
    vs.clear();
    //booking:
    //changing availability (booking):
    ColVals cvsIil = null;
    String[] verQuan = new String[] {"ver", "quan"};
    Arrays.sort(verQuan);
    if (!this.isAndr) {
      cvsIil = new ColVals();
      this.srvClVl.putExpr(cvsIil, "quan");
    }
    for (CuOrGdLn gl : pCoOr.getGoods()) {
      vs.put("ItmPlcndFds", verQuan);
      vs.put("ItmPlcdpLv", 1);
      List<ItmPlc> gps = getOrm().retLstCnd(pRvs, vs, ItmPlc.class,
        "where ALWAY=0 and ITM=" + gl.getGood().getIid()); vs.clear();
      if (gps.size() != 0) {
        BigDecimal avQu = BigDecimal.ZERO;
        for (ItmPlc gp : gps) {
          avQu = avQu.add(gp.getQuan());
        }
        if (avQu.compareTo(gl.getQuan()) == -1) {
          //previous test should not be passed!!!
          this.log.error(pRvs, getClass(), "Good is not available #"
            + gl.getGood().getIid());
          throw new ExcCode(ExcCode.WRPR, "THERE_IS_NO_GOODS");
        }
        BigDecimal rst = gl.getQuan();
        for (ItmPlc gp : gps) {
          if (rst.compareTo(BigDecimal.ZERO) == 0) {
            break;
          }
          if (gp.getQuan().compareTo(gl.getQuan()) == -1) {
            rst = rst.subtract(gp.getQuan());
            gp.setQuan(BigDecimal.ZERO);
          } else {
            gp.setQuan(gp.getQuan().subtract(rst));
            rst = BigDecimal.ZERO;
          }
          //TODO PERFORM fastupd
          vs.put("ndFds", verQuan);
          getOrm().update(pRvs, vs, gp); vs.clear();
        }
        String wheTyId = "TYP=0 and ITID=" + gl.getGood().getIid();
        if (!this.isAndr) {
          //it must be RDBMS constraint: "quan>=0"!
          this.srvClVl.put(cvsIil, "ver", new Date().getTime());
          this.srvClVl.put(cvsIil, "quan",  "QUAN-" + gl.getQuan());
          try {
            this.rdb.update(Itlist.class, cvsIil, wheTyId);
          } catch (Exception e) {
            this.log.error(pRvs, getClass(), "Itlist NA Itm: itId/quan: "
              + gl.getGood().getIid() + "/" + gl.getQuan());
            throw new ExcCode(ExcCode.WRPR, "THERE_IS_NO_GOODS");
          }
        } else {
          vs.put("ItlistndFds", verQuan);
          Itlist iil = this.orm.retEntCnd(pRvs, vs, Itlist.class, wheTyId);
          vs.clear();
          BigDecimal aq = iil.getQuan().subtract(gl.getQuan());
          if (aq.compareTo(BigDecimal.ZERO) == -1) {
        this.log.error(pRvs, getClass(), "Itlist NA Itm: id/itId/avQua/quan: "
      + iil.getIid() + "/" + gl.getGood().getIid() + "/"
    + iil.getQuan() + "/" + gl.getQuan());
            throw new ExcCode(ExcCode.WRPR, "THERE_IS_NO_GOODS");
          } else {
            vs.put("ndFds", verQuan);
            iil.setQuan(aq);
            getOrm().update(pRvs, vs, iil); vs.clear();
          }
        }
      }
    }
    boolean tibs = false;
    for (CuOrSrLn sl : pCoOr.getServs()) {
      if (sl.getDt1() == null) {
        vs.put("SrvPlcndFds", verQuan);
        vs.put("SrvPlcdpLv", 1);
        List<SrvPlc> sps = getOrm().retLstCnd(pRvs, vs, SrvPlc.class,
          "where ALWAY=0 and ITM=" + sl.getSrv().getIid()); vs.clear();
        if (sps.size() != 0) {
          BigDecimal avQu = BigDecimal.ZERO;
          for (SrvPlc sp : sps) {
            avQu = avQu.add(sp.getQuan());
          }
          if (avQu.compareTo(sl.getQuan()) == -1) {
            //previous test should not be passed!!!
            this.log.error(pRvs, getClass(), "Service is not available #"
              + sl.getSrv().getIid());
            throw new ExcCode(ExcCode.WRPR, "serv_no_avlb");
          }
          BigDecimal rst = sl.getQuan();
          for (SrvPlc sp : sps) {
            if (rst.compareTo(BigDecimal.ZERO) == 0) {
              break;
            }
            if (sp.getQuan().compareTo(sl.getQuan()) == -1) {
              rst = rst.subtract(sp.getQuan());
              sp.setQuan(BigDecimal.ZERO);
            } else {
              sp.setQuan(sp.getQuan().subtract(rst));
              rst = BigDecimal.ZERO;
            }
            //TODO PERFORM fastupd
            vs.put("ndFds", verQuan);
            getOrm().update(pRvs, vs, sp); vs.clear();
          }
          String wheTyId = "TYP=1 and ITID=" + sl.getSrv().getIid();
          if (!this.isAndr) {
            this.srvClVl.put(cvsIil, "ver", new Date().getTime());
            this.srvClVl.put(cvsIil, "quan",  "QUAN-" + sl.getQuan());
            try {
              this.rdb.update(Itlist.class, cvsIil, wheTyId);
            } catch (Exception e) {
              this.log.error(pRvs, getClass(), "Itlist NA Srv: itId/quan: "
                + sl.getSrv().getIid() + "/" + sl.getQuan());
              throw new ExcCode(ExcCode.WRPR, "serv_no_avlb");
            }
          } else {
            vs.put("ItlistndFds", verQuan);
            Itlist iil = this.orm.retEntCnd(pRvs, vs, Itlist.class, wheTyId);
            vs.clear();
            BigDecimal aq = iil.getQuan().subtract(sl.getQuan());
            if (aq.compareTo(BigDecimal.ZERO) == -1) {
          this.log.error(pRvs, getClass(), "Itlist NA Srv: id/itId/avQua/quan: "
        + iil.getIid() + "/" + sl.getSrv().getIid() + "/"
      + iil.getQuan() + "/" + sl.getQuan());
              throw new ExcCode(ExcCode.WRPR, "serv_no_avlb");
            } else {
              vs.put("ndFds", verQuan);
              iil.setQuan(aq);
              getOrm().update(pRvs, vs, iil); vs.clear();
            }
          }
        }
      } else {
        tibs = true;
      }
    }
    if (tibs) {
      vs.put("SerBusndFds", new String[] {"ver"});
      List<SerBus> sbas = this.orm.retLstCnd(pRvs, vs, SerBus.class,
        "where FRE=1"); vs.clear();
      int i = 0;
      for (CuOrSrLn sl : pCoOr.getServs()) {
        if (sl.getDt1() != null) {
          SerBus sb;
          if (i < sbas.size()) {
            sb = sbas.get(i);
            sb.setFre(false);
          } else {
            sb = new SerBus();
          }
          sb.setSrv(sl.getSrv());
          sb.setFrTm(sl.getDt1());
          sb.setTiTm(sl.getDt2());
          if (i < sbas.size()) {
            getOrm().update(pRvs, vs, sb);
            i++;
          } else {
            getOrm().insIdLn(pRvs, vs, sb);
          }
        }
      }
    }
  }

  /**
   * <p>Lazy Getter for quOrGdChk.</p>
   * @return String
   * @throws IOException - IO exception
   **/
  public final String lazyGetQuOrGdChk() throws IOException {
    if (this.quOrGdChk == null) {
      String flName = "/ws/ordGdChk.sql";
      this.quOrGdChk = loadString(flName);
    }
    return this.quOrGdChk;
  }

  /**
   * <p>Lazy Getter for quOrSrChk.</p>
   * @return String
   * @throws IOException - IO exception
   **/
  public final String lazyGetQuOrSrChk() throws IOException {
    if (this.quOrSrChk == null) {
      String flName = "/ws/ordSrChk.sql";
      this.quOrSrChk = loadString(flName);
    }
    return this.quOrSrChk;
  }

  /**
   * <p>Load string file (usually SQL query).</p>
   * @param pFileName file name
   * @return String usually SQL query
   * @throws IOException - IO exception
   **/
  public final String loadString(final String pFileName)
        throws IOException {
    URL urlFile = AcpOrd.class
      .getResource(pFileName);
    if (urlFile != null) {
      InputStream inputStream = null;
      try {
        inputStream = AcpOrd.class
          .getResourceAsStream(pFileName);
        byte[] bArray = new byte[inputStream.available()];
        inputStream.read(bArray, 0, inputStream.available());
        return new String(bArray, "UTF-8");
      } finally {
        if (inputStream != null) {
          inputStream.close();
        }
      }
    }
    return null;
  }

  //Simple getters and setters:
  /**
   * <p>Geter for log.</p>
   * @return ILog
   **/
  public final ILog getLog() {
    return this.log;
  }

  /**
   * <p>Setter for log.</p>
   * @param pLogger reference
   **/
  public final void setLog(final ILog pLogger) {
    this.log = pLogger;
  }

  /**
   * <p>Getter for orm.</p>
   * @return IOrm
   **/
  public final IOrm getOrm() {
    return this.orm;
  }

  /**
   * <p>Setter for orm.</p>
   * @param pOrm reference
   **/
  public final void setOrm(final IOrm pOrm) {
    this.orm = pOrm;
  }

  /**
   * <p>Getter for rdb.</p>
   * @return IRdb<RS>
   **/
  public final IRdb<RS> getRdb() {
    return this.rdb;
  }

  /**
   * <p>Setter for rdb.</p>
   * @param pRdb reference
   **/
  public final void setRdb(final IRdb<RS> pRdb) {
    this.rdb = pRdb;
  }

  /**
   * <p>Getter for isAndr.</p>
   * @return boolean
   **/
  public final boolean getIsAndr() {
    return this.isAndr;
  }

  /**
   * <p>Setter for isAndr.</p>
   * @param pIsAndr reference
   **/
  public final void setIsAndr(final boolean pIsAndr) {
    this.isAndr = pIsAndr;
  }

  /**
   * <p>Getter for srvClVl.</p>
   * @return SrvClVl
   **/
  public final SrvClVl getSrvClVl() {
    return this.srvClVl;
  }

  /**
   * <p>Setter for srvClVl.</p>
   * @param pSrvClVl reference
   **/
  public final void setSrvClVl(final SrvClVl pSrvClVl) {
    this.srvClVl = pSrvClVl;
  }
}
