/*
BSD 2-Clause License

Copyright (c) 2019, Beigesoft™
All rights reserved.

Redistribution and use in source and binary forms, with or without
modification, are permitted provided that the following conditions are met:

* Redistributions of source code must retain the above copyright notice, this
  list of conditions and the following disclaimer.

* Redistributions in binary form must reproduce the above copyright notice,
  this list of conditions and the following disclaimer in the documentation
  and/or other materials provided with the distribution.

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE
FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

package org.beigesoft.acc.srv;

import java.util.List;
import java.util.ArrayList;
import java.util.Map;
import java.util.HashMap;
import java.util.Date;
import java.util.Calendar;
import java.math.BigDecimal;
import java.io.InputStream;
import java.io.IOException;
import java.net.URL;

import org.beigesoft.exc.ExcCode;
import org.beigesoft.mdl.EPeriod;
import org.beigesoft.mdl.IRecSet;
import org.beigesoft.mdl.ColVals;
import org.beigesoft.log.ILog;
import org.beigesoft.rdb.IRdb;
import org.beigesoft.rdb.IOrm;
import org.beigesoft.rdb.SrvClVl;
import org.beigesoft.acc.mdlb.ISacnt;
import org.beigesoft.acc.mdlp.SacCh;
import org.beigesoft.acc.mdlp.AcStg;
import org.beigesoft.acc.mdlp.Acnt;
import org.beigesoft.acc.mdlp.Sacnt;
import org.beigesoft.acc.mdlp.Blnc;
import org.beigesoft.acc.mdlp.Entr;
import org.beigesoft.acc.mdlp.BlnCh;
import org.beigesoft.acc.mdl.TrBlLn;


/**
 * <p>Service that maintenance Blnc
 * and implements dirty check for all account.
 * If balance for account at given date is NULL then
 * it will be no record Blnc, this is cheap approach.
 * All work include recalculation all balances is executed
 * in single transaction. It works if there is any entry.</p>
 *
 * @param <RS> platform dependent record set type
 * @author Yury Demidenko
 */
public class SrBlnc<RS> implements ISrBlnc {

  /**
   * <p>Log.</p>
   **/
  private ILog log;

  /**
   * <p>ORM service.</p>
   **/
  private IOrm orm;

  /**
   * <p>Database service.</p>
   **/
  private IRdb<RS> rdb;

  /**
   * <p>Acc-ssttings service.</p>
   **/
  private ISrAcStg srAcStg;

  /**
   * <p>Column values service.</p>
   **/
  private SrvClVl srvClVl;

  /**
   * <p>Android configuration, RDB insert returns autogenerated ID,
   * updating with expression like "VER=VER+1" is not possible.</p>
   **/
  private boolean isAndr;

  //Private data:
  /**
   * <p>Query balance for all accounts.</p>
   **/
  private String quBlnc;

  /**
   * <p>Initialized date constant.</p>
   **/
  private final Date iniDt = new Date(BlnCh.INITDT);

    //Cached data:
  /**
   * <p>Balance info for maintain.</p>
   **/
  private BlnCh blnCh;

  /**
   * <p>Retrieve Trial Balance report.</p>
   * @param pRvs Request scoped variables
   * @param pDt date
   * @return balance lines
   * @throws Exception - an exception
   **/
  @Override
  public final synchronized List<TrBlLn> retTrBlnc(
    final Map<String, Object> pRvs, final Date pDt) throws Exception {
    chngSacsIfNd(pRvs);
    recalcIfNd(pRvs, pDt);
    List<TrBlLn> rz = new ArrayList<TrBlLn>();
    if (this.blnCh.getStDt().getTime() != BlnCh.INITDT) {
      String query = evQuBlnc(pRvs, pDt);
      IRecSet<RS> rs = null;
      try {
        rs = getRdb().retRs(query);
        AcStg as = getSrAcStg().lazAcStg(pRvs);
        if (rs.first()) {
          do {
            Double debt = rs.getDouble("DEBT");
            Double cred = rs.getDouble("CRED");
            if (debt != 0 || cred != 0) {
              TrBlLn tbl = new TrBlLn();
              tbl.setAcId(rs.getStr("ACID"));
              tbl.setAcNm(rs.getStr("ACNM"));
              tbl.setAcNb(rs.getStr("ACNB"));
              tbl.setSaNm(rs.getStr("SANM"));
              tbl.setDebt(BigDecimal.valueOf(debt)
                .setScale(as.getRpDp(), as.getRndm()));
              tbl.setCred(BigDecimal.valueOf(cred)
                .setScale(as.getRpDp(), as.getRndm()));
              if (tbl.getDebt().compareTo(BigDecimal.ZERO) != 0
                || tbl.getCred().compareTo(BigDecimal.ZERO) != 0) {
                rz.add(tbl);
              }
            }
          } while (rs.next());
        }
      } finally {
        if (rs != null) {
          rs.close();
        }
      }
      //account totals:
      BigDecimal debitAcc = null;
      BigDecimal creditAcc = null;
      String accCurr = null;
      int lineCurr = 0;
      int lineStartAcc = 0;
      for (TrBlLn tbl : rz) {
        if (!tbl.getAcId().equals(accCurr)) {
          if (accCurr != null) {
            //save to old:
            for (int j = lineStartAcc; j < lineCurr; j++) {
              rz.get(j).setDebtAcc(debitAcc);
              rz.get(j).setCredAcc(creditAcc);
            }
          }
          //init new acc:
          lineStartAcc = lineCurr;
          accCurr = tbl.getAcId();
          debitAcc = BigDecimal.ZERO;
          creditAcc = BigDecimal.ZERO;
        }
        debitAcc = debitAcc.add(tbl.getDebt());
        creditAcc = creditAcc.add(tbl.getCred());
        lineCurr++;
      }
      for (int j = lineStartAcc; j < lineCurr; j++) { //last account
        rz.get(j).setDebtAcc(debitAcc);
        rz.get(j).setCredAcc(creditAcc);
      }
    }
    return rz;
  }

  /**
   * <p>Recalculate if need for all balances for all dates less
   * or equals maximum (pDtFor, max date entry, max date balance),
   * this method is always invoked by reports.</p>
   * @param pRvs Request scoped variables
   * @param pDtFor date for
   * @return if has been recalculation
   * @throws Exception - an exception
   **/
  @Override
  public final synchronized boolean recalcIfNd(final Map<String, Object> pRvs,
    final Date pDtFor) throws Exception {
    Map<String, Object> vs = new HashMap<String, Object>();
    //lazy get start data, order is essential:
    lazBlnCh(pRvs, vs);
    evBlStPer(pRvs, vs);
    if (evBlStDt(pRvs, vs).getTime() == BlnCh.INITDT) {
      return false;
    }
    Date dtSt = evDtStPer(pRvs, pDtFor);
    Calendar calStBl = Calendar.getInstance();
    calStBl.setTime(this.blnCh.getStDt());
    calStBl.set(Calendar.MONTH, 0);
    calStBl.set(Calendar.DAY_OF_MONTH, 1);
    calStBl.set(Calendar.HOUR_OF_DAY, 0);
    calStBl.set(Calendar.MINUTE, 0);
    calStBl.set(Calendar.SECOND, 0);
    calStBl.set(Calendar.MILLISECOND, 0);
    if (this.blnCh.getLeDt().getTime() < this.blnCh.getStDt().getTime()) {
      //there is new entry with date less than balance start
      this.blnCh.setStDt(iniDt);
      this.blnCh.setLeDt(iniDt);
      if (evBlStDt(pRvs, vs).getTime() != BlnCh.INITDT) {
        recalc(pRvs, vs, pDtFor);
        return true;
      }
    } else if (this.blnCh.getPrCh() //period changed
      || dtSt.getTime() > this.blnCh.getCuDt().getTime() //new period
        //old period dirty:
        || this.blnCh.getLeDt().getTime() < this.blnCh.getCuDt().getTime()) {
      recalc(pRvs, vs, pDtFor);
      return true;
    }
    return false;
  }

  /**
   * <p>Handle new accounting entry to check dirty of stored balances.</p>
   * @param pRvs Request scoped variables
   * @param pDtAt date at
   * @throws Exception - an exception
   **/
  @Override
  public final synchronized void hndNewEntr(final Map<String, Object> pRvs,
    final Date pDtAt) throws Exception {
    Map<String, Object> vs = new HashMap<String, Object>();
    if (lazBlnCh(pRvs, vs).getLeDt().getTime() > pDtAt.getTime()) {
      boolean dbgSh = getLog().getDbgSh(this.getClass(), 11100);
      if (dbgSh) {
        getLog().debug(pRvs, SrBlnc.class, "change least last entry date from "
          + this.blnCh.getLeDt() + " to " + pDtAt);
      }
      this.blnCh.setLeDt(pDtAt);
      getOrm().update(pRvs, vs, this.blnCh);
    }
  }

  /**
   * <p>Evaluate start of period nearest to pDtFor.
   * Tested in blc org.beigesoft.test.CalendarTest.</p>
   * @param pRvs Request scoped variables
   * @param pDtFor date for
   * @return Start of period nearest to pDtFor
   * @throws Exception - an exception
   **/
  @Override
  public final synchronized Date evDtStPer(final Map<String, Object> pRvs,
    final Date pDtFor) throws Exception {
    Map<String, Object> vs = new HashMap<String, Object>();
    lazBlnCh(pRvs, vs);
    EPeriod per = evBlStPer(pRvs, vs);
    if (!(per.equals(EPeriod.MONTHLY) || per.equals(EPeriod.DAILY))) {
      throw new ExcCode(ExcCode.WRPR, "stored_balance_period_must_be_wm");
    }
    Calendar cal = Calendar.getInstance();
    cal.setTime(pDtFor);
    cal.set(Calendar.HOUR_OF_DAY, 0);
    cal.set(Calendar.MINUTE, 0);
    cal.set(Calendar.SECOND, 0);
    cal.set(Calendar.MILLISECOND, 0); //Daily is ready
    if (per.equals(EPeriod.MONTHLY)) {
      cal.set(Calendar.DAY_OF_MONTH, 1);
    }
    return cal.getTime();
  }

  /**
   * <p>Changes if need entries Entr.sacNm, Entr.sadNm, Blnc.SaNm.
   * It should be invoked together with recalcIfNd,
   * and service must be locked (synchronized).</p>
   * @param pRvs Request scoped variables
   * @return rows count updated
   * @throws Exception - an exception
   **/
  @Override
  public final synchronized int chngSacsIfNd(
    final Map<String, Object> pRvs) throws Exception {
    Map<String, Object> vs = new HashMap<String, Object>();
    List<SacCh> lsSacCh = this.orm.retLstCnd(pRvs, vs, SacCh.class,
      "where DRT=1");
    if (lsSacCh.size() > 0) {
      this.log.info(pRvs, getClass(),
    "Try to update subacc names in entries, count dirtied: " + lsSacCh.size());
      int rz = 0;
      for (SacCh sacCh : lsSacCh) {
        ColVals cv = new ColVals();
        this.srvClVl.put(cv, "sadNm", sacCh.getNme());
        if (!this.isAndr) {
          this.srvClVl.put(cv, "ver", "VER+1");
          this.srvClVl.putExpr(cv, "ver");
        }
        int esdc = this.rdb.update(Entr.class, cv, "SADID=" + sacCh.getSaId()
          + " and SADTY=" + sacCh.getSaTy());
        this.log.info(pRvs, getClass(), "Updated Entr.sadNm = " + esdc);
        cv = new ColVals();
        this.srvClVl.put(cv, "sacNm", sacCh.getNme());
        if (!this.isAndr) {
          this.srvClVl.put(cv, "ver", "VER+1");
          this.srvClVl.putExpr(cv, "ver");
        }
        int escc = this.rdb.update(Entr.class, cv, "SACID=" + sacCh.getSaId()
          + " and SACTY=" + sacCh.getSaTy());
        this.log.info(pRvs, getClass(), "Updated Entr.sacNm = " + escc);
        cv = new ColVals();
        this.srvClVl.put(cv, "saNm", sacCh.getNme());
        if (!this.isAndr) {
          this.srvClVl.put(cv, "ver", "VER+1");
          this.srvClVl.putExpr(cv, "ver");
        }
        int bsc = this.rdb.update(Blnc.class, cv, "SAID=" + sacCh.getSaId()
          + " and SATY=" + sacCh.getSaTy());
        this.log.info(pRvs, getClass(), "Updated Blnc.saNm = " + bsc);
        cv = new ColVals();
        this.srvClVl.put(cv, "saNm", sacCh.getNme());
        if (!this.isAndr) {
          this.srvClVl.put(cv, "ver", "VER+1");
          this.srvClVl.putExpr(cv, "ver");
        }
        int sacntc = this.rdb.update(Sacnt.class, cv, "SAID=" + sacCh.getSaId()
          + " and SATY=" + sacCh.getSaTy());
        this.log.info(pRvs, getClass(), "Updated Sacnt.saNm = " + sacntc);
        rz += esdc + escc + bsc + sacntc;
        sacCh.setDrt(Boolean.FALSE);
        this.orm.update(pRvs, vs, sacCh);
      }
      return rz;
    } else {
      return 0;
    }
  }

  /**
   * <p>Handle subaccount has been changed.</p>
   * @param pRvs Request scoped variables
   * @param pSacnt subaccount
   * @throws Exception - an exception
   **/
  @Override
  public final synchronized void hndSacntCh(final Map<String, Object> pRvs,
    final ISacnt pSacnt) throws Exception {
    SacCh sacCh = new SacCh();
    sacCh.setSaTy(pSacnt.cnsTy());
    sacCh.setSaId(pSacnt.getIid());
    Map<String, Object> vs = new HashMap<String, Object>();
    this.orm.refrEnt(pRvs, vs, sacCh);
    if (sacCh.getIid() == null) {
      sacCh.setIsNew(Boolean.TRUE);
      sacCh.setSaTy(pSacnt.cnsTy());
      sacCh.setSaId(pSacnt.getIid());
    }
    sacCh.setNme(pSacnt.getNme());
    sacCh.setDrt(Boolean.TRUE);
    if (sacCh.getIsNew()) {
      this.orm.insIdNln(pRvs, vs, sacCh);
    } else {
      this.orm.update(pRvs, vs, sacCh);
    }
  }

  /**
   * <p>Evaluate date start of next balance store period.
   * Tested in blc org.beigesoft.test.CalendarTest.</p>
   * @param pRvs Request scoped variables
   * @param pVs Invoker scoped variables
   * @param pDtFor date for
   * @return Start of next period nearest to pDtFor
   * @throws Exception - an exception
   **/
  @Override
  public final synchronized Date evDtNxtPerSt(final Map<String, Object> pRvs,
    final Map<String, Object> pVs, final Date pDtFor) throws Exception {
    lazBlnCh(pRvs, pVs);
    EPeriod per = evBlStPer(pRvs, pVs);
    if (!(per.equals(EPeriod.MONTHLY) || per.equals(EPeriod.DAILY))) {
      throw new ExcCode(ExcCode.WRPR, "stored_balance_period_must_be_wm");
    }
    Calendar cal = Calendar.getInstance();
    cal.setTime(pDtFor);
    cal.set(Calendar.HOUR_OF_DAY, 0);
    cal.set(Calendar.MINUTE, 0);
    cal.set(Calendar.SECOND, 0);
    cal.set(Calendar.MILLISECOND, 0);
    if (per.equals(EPeriod.MONTHLY)) {
      cal.add(Calendar.MONTH, 1);
      cal.set(Calendar.DAY_OF_MONTH, 1);
    } else if (per.equals(EPeriod.DAILY)) {
      cal.add(Calendar.DATE, 1);
    }
    return cal.getTime();
  }

  /**
   * <p>Evaluate date start of previous balance store period.
   * Tested in blc org.beigesoft.test.CalendarTest.</p>
   * @param pRvs Request scoped variables
   * @param pVs Invoker scoped variables
   * @param pDtFor date for
   * @return Start of next period nearest to pDtFor
   * @throws Exception - an exception
   **/
  @Override
  public final synchronized Date evDtPrvPerSt(final Map<String, Object> pRvs,
    final Map<String, Object> pVs, final Date pDtFor) throws Exception {
    lazBlnCh(pRvs, pVs);
    EPeriod per = evBlStPer(pRvs, pVs);
    if (!(per.equals(EPeriod.MONTHLY) || per.equals(EPeriod.DAILY))) {
      throw new ExcCode(ExcCode.WRPR, "stored_balance_period_must_be_wm");
    }
    Calendar cal = Calendar.getInstance();
    cal.setTime(pDtFor);
    cal.set(Calendar.HOUR_OF_DAY, 0);
    cal.set(Calendar.MINUTE, 0);
    cal.set(Calendar.SECOND, 0);
    cal.set(Calendar.MILLISECOND, 0);
    if (per.equals(EPeriod.MONTHLY)) {
      cal.add(Calendar.MONTH, -1);
      cal.set(Calendar.DAY_OF_MONTH, 1);
    } else if (per.equals(EPeriod.DAILY)) {
      cal.add(Calendar.DATE, -1);
    }
    return cal.getTime();
  }

  /**
   * <p>Handle rollback, e.g. clears cached data.</p>
   * @param pRvs Request scoped variables
   * @throws Exception - an exception
   **/
  @Override
  public final synchronized void hndRlBk(
    final Map<String, Object> pRvs) throws Exception {
    getLog().warn(pRvs, getClass(), "Clear cache cause transaction rollback!");
    this.blnCh = null;
  }

  //Utils:
  /**
   * <p>Forced recalculation all balances for all dates less
   * or equals maximum (pDtFor, max date entry, max date balance).
   * If balance for account at given date is NULL then
   * it will be no recorded into Blnc, this is cheap approach.</p>
   * @param pRvs Request scoped variables
   * @param pVs Invoker scoped variables
   * @param pDtFor date for
   * @throws Exception - an exception
   **/
  public final synchronized void recalc(final Map<String, Object> pRvs,
    final Map<String, Object> pVs, final Date pDtFor) throws Exception {
    Date dtEndRec = null;
    Long dtLsEnrLn = this.rdb.evLong(
      "select max(DAT) as MAXDT from ENTR where RVID is null;", "MAXDT");
    Date dtLstEntr = null;
    if (dtLsEnrLn != null) {
      dtLstEntr = new Date(dtLsEnrLn);
      if (dtLstEntr.getTime() > pDtFor.getTime()) {
        dtEndRec = dtLstEntr;
      } else {
        dtEndRec = pDtFor;
      }
    } else {
      dtEndRec = pDtFor;
    }
    Long dtLsBlnLn = this.rdb.evLong(
      "select max(DAT) as MAXDT from BLNC;", "MAXDT");
    Date dtLstBlnc = null;
    if (dtLsBlnLn != null) {
      dtLstBlnc = new Date(dtLsBlnLn);
      if (dtLstBlnc.getTime() > dtEndRec.getTime()) {
        dtEndRec = dtLstBlnc;
      }
    }
    getLog().info(pRvs, SrBlnc.class, "recalculation start for " + pDtFor
      + " BlnCh was " + this.blnCh + ", dtLstEntr = " + dtLstEntr
        + ", dtLsBlnLn = " + dtLsBlnLn);
    if (this.blnCh.getPrCh()) {
      getLog().info(pRvs, SrBlnc.class,
        "deleting all stored balances cause period has changed");
      getRdb().delete(Blnc.class.getSimpleName().toUpperCase(), null);
      this.blnCh.setPrCh(false);
      this.blnCh.setCuDt(this.iniDt);
    }
    Date curDt;
    if (this.blnCh.getLeDt().getTime() < this.blnCh.getCuDt().getTime()) {
      //recalculate from previous to changes period;
      curDt = evDtPrvPerSt(pRvs, pVs, this.blnCh.getLeDt());
      if (curDt.getTime() <= this.blnCh.getStDt().getTime()) {
        //recalculate from start;
        curDt = evDtNxtPerSt(pRvs, pVs, this.blnCh.getStDt());
        getLog().info(pRvs, SrBlnc.class, "recalculating balances from start "
          + curDt + " <- " + this.blnCh.getStDt());
      } else {
        getLog().info(pRvs, SrBlnc.class,
          "recalculating balances from previous " + curDt);
      }
    } else {
      //recalculate from current end;
      curDt = evDtNxtPerSt(pRvs, pVs, this.blnCh.getCuDt());
      getLog().info(pRvs, SrBlnc.class, "recalc balances from current end "
        + curDt + " <- " + this.blnCh.getCuDt());
    }
    Date lstBlStDt;
    do {
      lstBlStDt = curDt;
      String query = evQuBlnc(pRvs, new Date(lstBlStDt.getTime() - 1));
      List<TrBlLn> tbls = retBlnLnsToSv(query);
      pVs.put("AcntdpLv", 0);
      List<Blnc> blncs = getOrm().retLstCnd(pRvs, pVs, Blnc.class, " where DAT="
        + lstBlStDt.getTime());
      pVs.clear();
      for (TrBlLn tbl : tbls) {
        Blnc blnc = null;
        for (Blnc bl : blncs) {
         if (bl.getAcc().getIid().equals(tbl.getAcId()) && (bl.getSaTy() == null
            && tbl.getSaTy() == null) || bl.getSaTy() != null && bl.getSaTy()
              .equals(tbl.getSaTy()) && bl.getSaId().equals(tbl.getSaId())) {
            blnc = bl;
            blncs.remove(blnc);
            break;
          }
        }
        if (blnc == null) {
          blnc = new Blnc();
          blnc.setIsNew(true);
        }
        blnc.setDat(lstBlStDt);
        Acnt acc = new Acnt();
        acc.setIid(tbl.getAcId());
        blnc.setAcc(acc);
        if (tbl.getDebt().compareTo(BigDecimal.ZERO) != 0) {
          blnc.setBln(tbl.getDebt());
        } else {
          blnc.setBln(tbl.getCred());
        }
        blnc.setSaTy(tbl.getSaTy());
        blnc.setSaId(tbl.getSaId());
        blnc.setSaNm(tbl.getSaNm());
        if (blnc.getIsNew()) {
          getOrm().insIdLn(pRvs, pVs, blnc);
          blnc.setIsNew(false);
        } else {
          getOrm().update(pRvs, pVs, blnc);
        }
      }
      for (Blnc bl : blncs) {
        getOrm().del(pRvs, pVs, bl); //dirtied after reversing
      }
      curDt = evDtNxtPerSt(pRvs, pVs, curDt);
    } while (curDt.getTime() <= dtEndRec.getTime());
      getLog().info(pRvs, SrBlnc.class, "last stored balance lstBlStDt "
        + lstBlStDt + ", dtEndRecal = " + dtEndRec);
    this.blnCh.setCuDt(lstBlStDt);
    this.blnCh.setLeDt(this.blnCh.getCuDt());
    getOrm().update(pRvs, pVs, this.blnCh);
    getLog().info(pRvs, SrBlnc.class, "recalculation end BlnCh is "
      + this.blnCh);
  }

  /**
   * <p>Evaluate period of stored balances according settings,
   * if it's changed then it switch on "current balances are dirty".</p>
   * @param pRvs Request scoped variables
   * @param pVs Invoker scoped variables
   * @return pPer EPeriod e.g. MONTHLY
   * @throws Exception - an exception
   **/
  public final synchronized EPeriod evBlStPer(final Map<String, Object> pRvs,
    final Map<String, Object> pVs) throws Exception {
    AcStg as = getSrAcStg().lazAcStg(pRvs);
    if (!this.blnCh.getPrCh() && !this.blnCh.getStPr().equals(as.getBlPr())) {
      getLog().info(pRvs, SrBlnc.class, "changing period from " + this.blnCh
        .getStPr() + " to " + as.getBlPr());
      this.blnCh.setStPr(as.getBlPr());
      this.blnCh.setPrCh(true);
      this.blnCh.setLeDt(iniDt);
      getOrm().update(pRvs, pVs, this.blnCh);
    }
    return this.blnCh.getStPr();
  }

  /**
   * <p>Evaluate date start of stored balances according settings,
   * this is the first month of the first accounting entry.
   * It also may change date start in BlnCh.</p>
   * @param pRvs Request scoped variables
   * @param pVs Invoker scoped variables
   * @return Date
   * @throws Exception - an exception
   **/
  public final synchronized Date evBlStDt(final Map<String, Object> pRvs,
    final Map<String, Object> pVs) throws Exception {
    if (this.blnCh.getStDt().getTime() == BlnCh.INITDT) {
      //any situation:
      //1.the first invocation
      //2.there is new entry with date less then current start was
      Long dtFsEnrLn = this.rdb.evLong(
        "select min(DAT) as MINDT from ENTR where RVID is null;", "MINDT");
      if (dtFsEnrLn != null) {
        this.blnCh.setStDt(evDtStPer(pRvs, new Date(dtFsEnrLn)));
        getOrm().update(pRvs, pVs, this.blnCh);
      } else if (this.blnCh.getCuDt().getTime() != BlnCh.INITDT) {
        //there is Blnc without any non-reversed entry
        getLog().error(pRvs, getClass(), "Dirty balances!");
        getLog().info(pRvs, SrBlnc.class,
          "deleting all stored balances cause where is no entries");
        getRdb().delete(Blnc.class.getSimpleName().toUpperCase(), null);
        this.blnCh.setPrCh(false);
        this.blnCh.setStDt(this.iniDt);
        this.blnCh.setLeDt(this.iniDt);
        this.blnCh.setCuDt(this.iniDt);
        getOrm().update(pRvs, pVs, this.blnCh);
      }
    }
    return this.blnCh.getStDt();
  }

  /**
   * <p>Evaluate Trial Balance query.</p>
   * @param pRvs Request scoped variables
   * @param pDt date of balance
   * @return query of balance
   * @throws Exception - an exception
   **/
  public final synchronized String evQuBlnc(final Map<String, Object> pRvs,
    final Date pDt) throws Exception {
    if (this.quBlnc == null) {
      String flName = "/acc/blnc.sql";
      this.quBlnc = loadStr(flName);
    }
    String query = quBlnc.replace(":DT1",
      String.valueOf(evDtStPer(pRvs, pDt).getTime()));
    query = query.replace(":DT2", String.valueOf(pDt.getTime()));
    return query;
  }

  /**
   * <p>Retrieve Trial Balance lines with given query to save into DB.</p>
   * @param pQu date
   * @return balance lines
   * @throws Exception - an exception
   **/
  public final synchronized List<TrBlLn> retBlnLnsToSv(
    final String pQu) throws Exception {
    List<TrBlLn> rz = new ArrayList<TrBlLn>();
    IRecSet<RS> rs = null;
    try {
      rs = getRdb().retRs(pQu);
      if (rs.first()) {
        do {
          Double debt = rs.getDouble("DEBT");
          Double cred = rs.getDouble("CRED");
          TrBlLn tbl = new TrBlLn();
          tbl.setAcId(rs.getStr("ACID"));
          tbl.setAcNm(rs.getStr("ACNM"));
          tbl.setAcNb(rs.getStr("ACNB"));
          tbl.setSaId(rs.getLong("SAID"));
          tbl.setSaTy(rs.getInt("SATY"));
          tbl.setSaNm(rs.getStr("SANM"));
          tbl.setDebt(BigDecimal.valueOf(debt));
          tbl.setCred(BigDecimal.valueOf(cred));
          rz.add(tbl);
        } while (rs.next());
      }
    } finally {
      if (rs != null) {
        rs.close();
      }
    }
    return rz;
  }

  /**
   * <p>Load string file (usually SQL query).</p>
   * @param pFlNm file name
   * @return SQL query, not null
   * @throws IOException - IO exception
   **/
  public final synchronized String loadStr(
    final String pFlNm) throws IOException {
    URL urlFile = SrBlnc.class.getResource(pFlNm);
    if (urlFile != null) {
      InputStream is = null;
      try {
        is = SrBlnc.class.getResourceAsStream(pFlNm);
        byte[] bArray = new byte[is.available()];
        is.read(bArray, 0, is.available());
        return new String(bArray, "UTF-8");
      } finally {
        if (is != null) {
          is.close();
        }
      }
    }
    throw new RuntimeException("File not found: " + pFlNm);
  }

  /**
   * <p>Lazy getter for blnCh.</p>
   * @param pRvs Request scoped variables
   * @param pVs Invoker scoped variables
   * @return BlnCh
   * @throws Exception - an exception
   **/
  public final synchronized BlnCh lazBlnCh(final Map<String, Object> pRvs,
    final Map<String, Object> pVs) throws Exception {
    if (this.blnCh == null) {
      BlnCh balLoc = new BlnCh();
      balLoc.setIid(1L);
      getOrm().refrEnt(pRvs, pVs, balLoc);
      if (balLoc.getIid() == null) {
        balLoc.setIid(1L);
        getOrm().insIdLn(pRvs, pVs, balLoc);
        balLoc.setIsNew(false);
      }
      this.blnCh = balLoc;
    }
    return this.blnCh;
  }


  //Simple getters and setters:
  /**
   * <p>Getter for orm.</p>
   * @return IOrm
   **/
  public final synchronized IOrm getOrm() {
    return this.orm;
  }

  /**
   * <p>Setter for orm.</p>
   * @param pOrm reference
   **/
  public final synchronized void setOrm(final IOrm pOrm) {
    this.orm = pOrm;
  }

  /**
   * <p>Geter for rdb.</p>
   * @return IRdb
   **/
  public final synchronized IRdb<RS> getRdb() {
    return this.rdb;
  }

  /**
   * <p>Setter for rdb.</p>
   * @param pRdb reference
   **/
  public final synchronized void setRdb(
    final IRdb<RS> pRdb) {
    this.rdb = pRdb;
  }

  /**
   * <p>Getter for quBlnc.</p>
   * @return String
   **/
  public final synchronized String getQuBlnc() {
    return this.quBlnc;
  }

  /**
   * <p>Setter for quBlnc.</p>
   * @param pQuBlnc reference
   **/
  public final synchronized void setQuBlnc(final String pQuBlnc) {
    this.quBlnc = pQuBlnc;
  }

  /**
   * <p>Geter for log.</p>
   * @return ILog
   **/
  public final synchronized ILog getLog() {
    return this.log;
  }

  /**
   * <p>Setter for log.</p>
   * @param pLog reference
   **/
  public final synchronized void setLog(final ILog pLog) {
    this.log = pLog;
  }

  /**
   * <p>Getter for srAcStg.</p>
   * @return ISrAcStg
   **/
  public final synchronized ISrAcStg getSrAcStg() {
    return this.srAcStg;
  }

  /**
   * <p>Setter for srAcStg.</p>
   * @param pSrAcStg reference
   **/
  public final synchronized void setSrAcStg(final ISrAcStg pSrAcStg) {
    this.srAcStg = pSrAcStg;
  }

  /**
   * <p>Getter for srvClVl.</p>
   * @return SrvClVl
   **/
  public final synchronized SrvClVl getSrvClVl() {
    return this.srvClVl;
  }

  /**
   * <p>Setter for srvClVl.</p>
   * @param pSrvClVl reference
   **/
  public final synchronized void setSrvClVl(final SrvClVl pSrvClVl) {
    this.srvClVl = pSrvClVl;
  }

  /**
   * <p>Getter for isAndr.</p>
   * @return boolean
   **/
  public final synchronized boolean getIsAndr() {
    return this.isAndr;
  }

  /**
   * <p>Setter for isAndr.</p>
   * @param pIsAndr reference
   **/
  public final synchronized void setIsAndr(final boolean pIsAndr) {
    this.isAndr = pIsAndr;
  }
}
