/*
BSD 2-Clause License

Copyright (c) 2019, Beigesoft™
All rights reserved.

Redistribution and use in source and binary forms, with or without
modification, are permitted provided that the following conditions are met:

* Redistributions of source code must retain the above copyright notice, this
  list of conditions and the following disclaimer.

* Redistributions in binary form must reproduce the above copyright notice,
  this list of conditions and the following disclaimer in the documentation
  and/or other materials provided with the distribution.

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE
FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

package org.beigesoft.acc.srv;

import java.util.List;
import java.util.ArrayList;
import java.util.Map;
import java.util.HashMap;
import java.util.Arrays;
import java.util.Date;
import java.util.Locale;
import java.text.DateFormat;
import java.math.BigDecimal;
import java.io.InputStream;
import java.io.IOException;
import java.net.URL;

import org.beigesoft.exc.ExcCode;
import org.beigesoft.mdl.CmnPrf;
import org.beigesoft.mdl.IRecSet;
import org.beigesoft.mdl.ColVals;
import org.beigesoft.log.ILog;
import org.beigesoft.hld.IHlIntCls;
import org.beigesoft.rdb.IRdb;
import org.beigesoft.rdb.IOrm;
import org.beigesoft.rdb.SrvClVl;
import org.beigesoft.srv.II18n;
import org.beigesoft.acc.mdl.ECogsMth;
import org.beigesoft.acc.mdl.EDocTy;
import org.beigesoft.acc.mdlb.ADrItEnr;
import org.beigesoft.acc.mdlb.IMkDriEnr;
import org.beigesoft.acc.mdlb.IItmSrc;
import org.beigesoft.acc.mdlb.IDcDri;
import org.beigesoft.acc.mdlp.AcStg;
import org.beigesoft.acc.mdlp.DriEnrSr;
import org.beigesoft.acc.mdlp.WrhPl;

/**
 * <p>Service that makes, reverses, retrieves draw item entries
 * for given drawer/document.</p>
 *
 * @param <RS> platform dependent record set type
 * @author Yury Demidenko
 */
public class SrDrItEnr<RS> implements ISrDrItEnr {

  /**
   * <p>Log.</p>
   **/
  private ILog log;

  /**
   * <p>ORM service.</p>
   **/
  private IOrm orm;

  /**
   * <p>Database service.</p>
   **/
  private IRdb<RS> rdb;

  /**
   * <p>I18N service.</p>
   */
  private II18n i18n;

  /**
   * <p>Column values service.</p>
   **/
  private SrvClVl srvClVl;

  /**
   * <p>Holder of item to draw sources types.</p>
   **/
  private IHlIntCls hlTyItSr;

  /**
   * <p>Holder of entries sources types.</p>
   **/
  private IHlIntCls hlTyEnSr;

  /**
   * <p>Warehouse entries service.</p>
   **/
  private ISrWrhEnr srWrhEnr;

  /**
   * <p>Android configuration, RDB insert returns autogenerated ID,
   * updating with expression like "VER=VER+1" is not possible.</p>
   **/
  private boolean isAndr;

  //Cached data:
  /**
   * <p>Entries queries.</p>
   */
  private Map<String, String> entrQus = new HashMap<String, String>();

  /**
   * <p>Entries sources.</p>
   */
  private List<DriEnrSr> entrSrcs;

  /**
   * <p>Handles settings changed, i.e. clears cached data.</p>
   * @param pRvs Request scoped variables
   * @throws Exception - an exception
   **/
  @Override
  public final synchronized void hndStgCng(
    final Map<String, Object> pRvs) throws Exception {
    this.entrSrcs = null;
  }

  /**
   * <p>Makes drawing entries for given drawer.
   * It also makes warehouse entries exactly for warehouse place
   * as in source one.</p>
   * @param <T> draw entry type
   * @param pRvs Request scoped variables
   * @param pDrer drawer document or line
   * @throws Exception - an exception
   **/
  @Override
  public final <T extends ADrItEnr> void draw(final Map<String, Object> pRvs,
    final IMkDriEnr<T> pDrer) throws Exception {
    Map<String, Object> vs = new HashMap<String, Object>();
    if (this.entrSrcs == null) {
      synchronized (this) {
        if (this.entrSrcs == null) {
          this.entrSrcs = this.orm.retLst(pRvs, vs, DriEnrSr.class);
        }
      }
    }
    StringBuffer sb =
      new StringBuffer("select SRTY, DAT, IID, OWID, WRHP, TOLF, ITLF from(\n");
    boolean isFst = true;
    for (DriEnrSr ensr : this.entrSrcs) {
      if (ensr.getUsed() && ensr.getSrsTy() == pDrer.getSrsTy()) {
        String qu = lazEntrQu(ensr.getQuFl());
        qu = qu.replace(":DBOR", this.orm.getDbId().toString());
        qu = qu.replace(":ITM", pDrer.getItm().getIid().toString());
        qu = qu.replace(":UOM", pDrer.getUom().getIid().toString());
        if (isFst) {
          isFst = false;
        } else {
          sb.append("\nunion all\n");
        }
        sb.append(qu);
      }
    }
    if (isFst) {
      throw new ExcCode(ExcCode.WRPR, "dri_entr_src_no_set");
    }
    AcStg as = (AcStg) pRvs.get("astg");
    if (as.getCogs() == ECogsMth.FIFO) {
      sb.append("\n) as ALRC order by DAT asc");
    } else if (as.getCogs() == ECogsMth.LIFO) {
      sb.append("\n) as ALRC order by DAT desc");
    } else {
      throw new ExcCode(ExcCode.WRPR, "cogs_av_not_imp");
    }
    long lim = pDrer.getQuan().longValue() + 1L;
    sb.append(" limit " + lim + ";");
    String qu = sb.toString();
    IRecSet<RS> rs = null;
    BigDecimal itq = BigDecimal.ZERO;
    List<IItmSrc> isrs = new ArrayList<IItmSrc>();
    try {
      rs = getRdb().retRs(qu);
      if (rs.first()) {
        do {
          Integer srTy = rs.getInt("SRTY");
          @SuppressWarnings("unchecked")
          IItmSrc sr = (IItmSrc) this.hlTyItSr.get(srTy).newInstance();
          isrs.add(sr);
          sr.setIid(rs.getLong("IID"));
          sr.setDbOr(this.orm.getDbId());
          sr.setOwnrId(rs.getLong("OWID"));
          sr.setDocDt(new Date(rs.getLong("DAT")));
          sr.setItLf(BigDecimal.valueOf(rs.getDouble("ITLF")));
          sr.setToLf(BigDecimal.valueOf(rs.getDouble("TOLF")));
          WrhPl wrhp = new WrhPl();
          wrhp.setIid(rs.getLong("WRHP"));
          sr.setWrhp(wrhp);
          sr.setItm(pDrer.getItm());
          sr.setUom(pDrer.getUom());
          itq = itq.add(sr.getItLf());
          if (itq.compareTo(pDrer.getQuan()) >= 0) {
            break;
          }
        } while (rs.next());
      }
    } finally {
      if (rs != null) {
        rs.close();
      }
    }
    if (itq.compareTo(pDrer.getQuan()) == -1) {
      throw new ExcCode(ExcCode.WRPR, "THERE_IS_NO_GOODS");
    }
    itq = pDrer.getQuan();
    for (IItmSrc sr : isrs) {
      BigDecimal quan = itq.min(sr.getItLf());
      drawFr(pRvs, pDrer, sr, quan);
      itq = itq.subtract(quan);
    }
  }

  /**
   * <p>Makes drawing entries for given drawer from given source.
   * Field "toLf" mast be already made by SQL query or invoker.
   * It also makes warehouse entries exactly for warehouse place
   * as in source one.</p>
   * @param <T> draw entry type
   * @param pRvs Request scoped variables
   * @param pDrer drawer document or line
   * @param pSrc item source
   * @param pQuan quantity
   * @throws Exception - an exception
   **/
  @Override
  public final <T extends ADrItEnr> void drawFr(final Map<String, Object> pRvs,
    final IMkDriEnr<T> pDrer, final IItmSrc pSrc,
      final BigDecimal pQuan) throws Exception {
    if (pSrc.getItLf().compareTo(pQuan) == -1) {
      throw new ExcCode(ExcCode.WR, "Src has no enough items! srCls/itLf/quan: "
        + pSrc.getClass() + "/" + pSrc.getItLf() + "/" + pQuan);
    }
    Map<String, Object> vs = new HashMap<String, Object>();
    AcStg as = (AcStg) pRvs.get("astg");
    CmnPrf cpf = (CmnPrf) pRvs.get("cpf");
    DateFormat dtFr = DateFormat.getDateTimeInstance(DateFormat
      .MEDIUM, DateFormat.SHORT, new Locale(cpf.getLngDef().getIid()));
    BigDecimal tot;
    if (pQuan.compareTo(pSrc.getItLf()) == 0) {
      tot = pSrc.getToLf();
    } else {
      tot = pSrc.getToLf().divide(pSrc.getItLf(), 15, as.getRndm())
        .multiply(pQuan).setScale(as.getPrDp(), as.getRndm());
    }
    pSrc.setToLf(pSrc.getToLf().subtract(tot));
    pSrc.setItLf(pSrc.getItLf().subtract(pQuan));
    if (this.isAndr) {
      String[] ndf = new String[] {"itLf", "toLf", "ver"};
      Arrays.sort(ndf);
      vs.put("ndFds", ndf);
      this.orm.update(pRvs, vs, pSrc); vs.clear();
    } else { //use fastest locking:
      ColVals cv = new ColVals();
      this.srvClVl.put(cv, "itLf", "ITLF-" + pQuan);
      this.srvClVl.put(cv, "toLf", pSrc.getToLf().doubleValue());
      this.srvClVl.putExpr(cv, "itLf");
      this.srvClVl.put(cv, "ver", "VER+1");
      this.srvClVl.putExpr(cv, "ver");
      try {
        this.rdb.update(pSrc.getClass(), cv, "IID=" + pSrc.getIid());
      } catch (Exception e) {
        this.log.error(pRvs, getClass(), "THERE_IS_NO_GOODS", e);
        throw new ExcCode(ExcCode.WRPR, "THERE_IS_NO_GOODS");
      }
    }
    T enr = pDrer.getEnrCls().newInstance();
    enr.setDbOr(this.orm.getDbId());
    enr.setSrTy(pSrc.cnsTy());
    enr.setSrId(pSrc.getIid());
    enr.setSowTy(pSrc.getOwnrTy());
    enr.setSowId(pSrc.getOwnrId());
    enr.setDrTy(pDrer.cnsTy());
    enr.setDrId(pDrer.getIid());
    enr.setDowTy(pDrer.getOwnrTy());
    enr.setDowId(pDrer.getOwnrId());
    enr.setItm(pDrer.getItm());
    enr.setUom(pDrer.getUom());
    enr.setQuan(pQuan);
    enr.setTot(tot);
    StringBuffer sb = mkDscr(pRvs, pDrer, dtFr);
    sb.append(", " + getI18n().getMsg("from", cpf.getLngDef().getIid())
       + " " + getI18n().getMsg(pSrc.getClass().getSimpleName() + "sht",
          cpf.getLngDef().getIid()));
    sb.append(" #" + pSrc.getDbOr() + "-" + pSrc.getIid());
    if (pSrc.getOwnrTy() != null) {
      sb.append(", " + getI18n().getMsg("in", cpf.getLngDef().getIid())
         + " " + getI18n().getMsg(this.hlTyEnSr.get(pSrc.getOwnrTy())
          .getSimpleName() + "sht", cpf.getLngDef().getIid()));
      sb.append(" #" + pSrc.getDbOr() + "-" + pSrc.getOwnrId());
    }
    sb.append(", " + dtFr.format(pSrc.getDocDt()));
    enr.setDscr(sb.toString());
    this.orm.insIdLn(pRvs, vs, enr);
    this.srWrhEnr.draw(pRvs, pDrer, pSrc.getWrhp(), pQuan);
  }

  /**
   * <p>Reverses drawing entries for given drawer.</p>
   * @param <T> draw entry type
   * @param pRvs Request scoped variables
   * @param pDrer reversing drawer document or line
   * @throws Exception - an exception
   **/
  @Override
  public final <T extends ADrItEnr> void rvDraw(final Map<String, Object> pRvs,
    final IMkDriEnr<T> pDrer) throws Exception {
    Map<String, Object> vs = new HashMap<String, Object>();
    List<T> revds = this.orm.retLstCnd(pRvs, vs, pDrer.getEnrCls(),
      "where DRTY=" + pDrer.cnsTy() + " and DRID=" + pDrer.getRvId());
    if (revds.size() == 0) {
      throw new ExcCode(ExcCode.WR, "Can't reverse for CLS/RVID/ID/TY: "
        + pDrer.getClass() + "/" + pDrer.getRvId() + "/" + pDrer.getIid()
          + "/" + pDrer.cnsTy());
    }
    for (T revd : revds) {
      if (revd.getRvId() != null) {
        throw new ExcCode(ExcCode.WR, "Reverse reversed for CLS/RVID/ID/TY: "
          + pDrer.getClass() + "/" + pDrer.getRvId() + "/" + pDrer.getIid()
            + "/" + pDrer.cnsTy());
      }
      T revg = pDrer.getEnrCls().newInstance();
      revg.setDbOr(this.orm.getDbId());
      revg.setDrTy(pDrer.cnsTy());
      revg.setDrId(pDrer.getIid());
      revg.setDowTy(pDrer.getOwnrTy());
      revg.setDowId(pDrer.getOwnrId());
      revg.setSrTy(revd.getSrTy());
      revg.setSrId(revd.getSrId());
      revg.setSowTy(revd.getSowTy());
      revg.setSowId(revd.getSowId());
      revg.setRvId(revd.getIid());
      revg.setItm(revd.getItm());
      revg.setUom(revd.getUom());
      revg.setQuan(revd.getQuan().negate());
      revg.setTot(revd.getTot().negate());
      CmnPrf cpf = (CmnPrf) pRvs.get("cpf");
      DateFormat dtFr = DateFormat.getDateTimeInstance(DateFormat
        .MEDIUM, DateFormat.SHORT, new Locale(cpf.getLngDef().getIid()));
      StringBuffer sb = mkDscr(pRvs, pDrer, dtFr);
      sb.append(" ," + getI18n().getMsg("reversed", cpf.getLngDef().getIid()));
      sb.append(" #" + revd.getDbOr() + "-" + revd.getIid());
      revg.setDscr(sb.toString() + "!");
      this.orm.insIdLn(pRvs, vs, revg);
      revd.setRvId(revg.getIid());
      revd.setDscr(revd.getDscr() + ", !" + getI18n()
        .getMsg("reversing", cpf.getLngDef().getIid()) + " #" + revg.getDbOr()
          + "-" + revg.getIid() + "!");
      String[] ndFds = new String[] {"dscr", "rvId", "ver"};
      Arrays.sort(ndFds);
      vs.put("ndFds", ndFds);
      this.orm.update(pRvs, vs, revd); vs.clear();
      @SuppressWarnings("unchecked")
      IItmSrc sr = (IItmSrc) this.hlTyItSr.get(revd.getSrTy()).newInstance();
      sr.setIid(revd.getSrId());
      if (this.isAndr) {
        String[] ndf = new String[] {"itLf", "toLf", "ver"};
        Arrays.sort(ndf);
        vs.put(sr.getClass().getSimpleName() + "ndFds", ndf);
        this.orm.refrEnt(pRvs, vs, sr); vs.clear();
        sr.setItLf(sr.getItLf().add(revd.getQuan()));
        sr.setToLf(sr.getToLf().add(revd.getTot()));
        vs.put("ndFds", ndf);
        this.orm.update(pRvs, vs, sr); vs.clear();
      } else { //use fastest locking:
        ColVals cv = new ColVals();
        this.srvClVl.put(cv, "itLf", "ITLF+" + revd.getQuan());
        this.srvClVl.put(cv, "toLf", "TOLF+" + revd.getTot());
        this.srvClVl.putExpr(cv, "itLf");
        this.srvClVl.putExpr(cv, "toLf");
        this.srvClVl.put(cv, "ver", "VER+1");
        this.srvClVl.putExpr(cv, "ver");
        this.rdb.update(sr.getClass(), cv, "IID=" + sr.getIid());
      }
    }
  }

  /**
   * <p>Retrieves drawing entries for given document.</p>
   * @param <T> draw entry type
   * @param pRvs Request scoped variables
   * @param pDoc document
   * @param pEnrCls entries class
   * @return entries
   * @throws Exception - an exception
   **/
  @Override
  public final <T extends ADrItEnr> List<T> retEntrs(
    final Map<String, Object> pRvs, final IDcDri pDoc,
      final Class<T> pEnrCls) throws Exception {
    String whe;
    //resource friendly implementation:
    if (pDoc.getDocTy() == EDocTy.DRAWLN) {
      whe = "where DOWTY=" + pDoc.cnsTy() + " and DOWID=" + pDoc.getIid();
    } else if (pDoc.getDocTy() == EDocTy.DRAWBTH) {
      whe = "where (DOWTY=" + pDoc.cnsTy() + " and DOWID=" + pDoc.getIid()
        + ") or (DRTY=" + pDoc.cnsTy() + " and DRID=" + pDoc.getIid() + ")";
    } else if (pDoc.getDocTy() == EDocTy.DRAW) {
      whe = "where DRTY=" + pDoc.cnsTy() + " and DRID=" + pDoc.getIid();
    } else if (pDoc.getDocTy() == EDocTy.ITSRLN) {
      whe = "where SOWTY=" + pDoc.cnsTy() + " and SOWID=" + pDoc.getIid();
    } else if (pDoc.getDocTy() == EDocTy.ITSRBTH) {
      whe = "where (SOWTY=" + pDoc.cnsTy() + " and SOWID=" + pDoc.getIid()
        + ") or (SRTY=" + pDoc.cnsTy() + " and SRID=" + pDoc.getIid() + ")";
    } else if (pDoc.getDocTy() == EDocTy.ITSR) {
      whe = "where SRTY=" + pDoc.cnsTy() + " and SRID=" + pDoc.getIid();
    } else if (pDoc.getDocTy() == EDocTy.ITSRDRAWLN) {
      whe = "where (SRTY=" + pDoc.cnsTy() + " and SRID=" + pDoc.getIid()
        + ") or (DOWTY=" + pDoc.cnsTy() + " and DOWID=" + pDoc.getIid() + ")";
    } else if (pDoc.getDocTy() == EDocTy.ITSRDRAW) {
      whe = "where (SRTY=" + pDoc.cnsTy() + " and SRID=" + pDoc.getIid()
        + ") or (DRTY=" + pDoc.cnsTy() + " and DRID=" + pDoc.getIid() + ")";
    } else {
      throw new Exception("Not allowed!");
    }
    Map<String, Object> vs = new HashMap<String, Object>();
    List<T> rz = this.orm.retLstCnd(pRvs, vs, pEnrCls, whe);
    return rz;
  }

  /**
   * <p>Makes entry description.</p>
   * @param pRvs request scoped vars
   * @param pDrer drawer
   * @param pDtFrm date format
   * @return string buffer
   * @throws Exception - an exception
   **/
  public final StringBuffer mkDscr(final Map<String, Object> pRvs,
    final IMkDriEnr pDrer, final DateFormat pDtFrm) throws Exception {
    CmnPrf cpf = (CmnPrf) pRvs.get("cpf");
    StringBuffer sb = new StringBuffer();
    sb.append(getI18n().getMsg("Made_at", cpf.getLngDef().getIid()) + " "
      + pDtFrm.format(new Date()) + " ");
    sb.append(getI18n().getMsg("by_who", cpf.getLngDef().getIid()) + ": ");
    sb.append(getI18n().getMsg(pDrer.getClass().getSimpleName() + "sht",
  cpf.getLngDef().getIid()) + " #" + pDrer.getDbOr() + "-" + pDrer.getIid());
    if (pDrer.getOwnrId() != null) {
      sb.append(", " + getI18n().getMsg("in", cpf.getLngDef().getIid())
       + " " + getI18n().getMsg(this.hlTyEnSr.get(pDrer.getOwnrTy())
        .getSimpleName() + "sht", cpf.getLngDef().getIid()));
      sb.append(" #" + pDrer.getDbOr() + "-" + pDrer.getOwnrId());
    }
    sb.append(", " + pDtFrm.format(pDrer.getDocDt()));
    return sb;
  }

  /**
   * <p>Lazy gets SQL query.</p>
   * @param pFlNm file name
   * @return SQL query
   * @throws IOException - IO exception
   **/
  public final String lazEntrQu(final String pFlNm) throws IOException {
    String rz = this.entrQus.get(pFlNm);
    if (rz == null) {
      synchronized (this) {
        rz = this.entrQus.get(pFlNm);
        if (rz == null) {
          rz = loadStr("/acc/dri/" + pFlNm + ".sql");
          this.entrQus.put(pFlNm, rz);
        }
      }
    }
    return rz;
  }

  /**
   * <p>Loads SQL query.</p>
   * @param pFlNm file name
   * @return SQL query, not null
   * @throws IOException - IO exception
   **/
  public final String loadStr(final String pFlNm) throws IOException {
    URL urlFile = SrDrItEnr.class.getResource(pFlNm);
    if (urlFile != null) {
      InputStream is = null;
      try {
        is = SrDrItEnr.class.getResourceAsStream(pFlNm);
        byte[] bArray = new byte[is.available()];
        is.read(bArray, 0, is.available());
        return new String(bArray, "UTF-8");
      } finally {
        if (is != null) {
          is.close();
        }
      }
    }
    throw new RuntimeException("File not found: " + pFlNm);
  }

  //Simple getters and setters:
  /**
   * <p>Getter for log.</p>
   * @return ILog
   **/
  public final ILog getLog() {
    return this.log;
  }

  /**
   * <p>Setter for log.</p>
   * @param pLog reference
   **/
  public final void setLog(final ILog pLog) {
    this.log = pLog;
  }

  /**
   * <p>Getter for orm.</p>
   * @return IOrm
   **/
  public final IOrm getOrm() {
    return this.orm;
  }

  /**
   * <p>Setter for orm.</p>
   * @param pOrm reference
   **/
  public final void setOrm(final IOrm pOrm) {
    this.orm = pOrm;
  }

  /**
   * <p>Getter for rdb.</p>
   * @return IRdb<RS>
   **/
  public final IRdb<RS> getRdb() {
    return this.rdb;
  }

  /**
   * <p>Setter for rdb.</p>
   * @param pRdb reference
   **/
  public final void setRdb(final IRdb<RS> pRdb) {
    this.rdb = pRdb;
  }

  /**
   * <p>Getter for i18n.</p>
   * @return II18n
   **/
  public final II18n getI18n() {
    return this.i18n;
  }

  /**
   * <p>Setter for i18n.</p>
   * @param pI18n reference
   **/
  public final void setI18n(final II18n pI18n) {
    this.i18n = pI18n;
  }

  /**
   * <p>Getter for srvClVl.</p>
   * @return SrvClVl
   **/
  public final SrvClVl getSrvClVl() {
    return this.srvClVl;
  }

  /**
   * <p>Setter for srvClVl.</p>
   * @param pSrvClVl reference
   **/
  public final void setSrvClVl(final SrvClVl pSrvClVl) {
    this.srvClVl = pSrvClVl;
  }

  /**
   * <p>Getter for isAndr.</p>
   * @return boolean
   **/
  public final boolean getIsAndr() {
    return this.isAndr;
  }

  /**
   * <p>Setter for isAndr.</p>
   * @param pIsAndr reference
   **/
  public final void setIsAndr(final boolean pIsAndr) {
    this.isAndr = pIsAndr;
  }

  /**
   * <p>Getter for hlTyItSr.</p>
   * @return IHlIntCls
   **/
  public final IHlIntCls getHlTyItSr() {
    return this.hlTyItSr;
  }

  /**
   * <p>Setter for hlTyItSr.</p>
   * @param pHlTyItSr reference
   **/
  public final void setHlTyItSr(final IHlIntCls pHlTyItSr) {
    this.hlTyItSr = pHlTyItSr;
  }

  /**
   * <p>Getter for hlTyEnSr.</p>
   * @return IHlIntCls
   **/
  public final IHlIntCls getHlTyEnSr() {
    return this.hlTyEnSr;
  }

  /**
   * <p>Setter for hlTyEnSr.</p>
   * @param pHlTyEnSr reference
   **/
  public final void setHlTyEnSr(final IHlIntCls pHlTyEnSr) {
    this.hlTyEnSr = pHlTyEnSr;
  }

  /**
   * <p>Getter for srWrhEnr.</p>
   * @return ISrWrhEnr
   **/
  public final ISrWrhEnr getSrWrhEnr() {
    return this.srWrhEnr;
  }

  /**
   * <p>Setter for srWrhEnr.</p>
   * @param pSrWrhEnr reference
   **/
  public final void setSrWrhEnr(final ISrWrhEnr pSrWrhEnr) {
    this.srWrhEnr = pSrWrhEnr;
  }
}
