/*
BSD 2-Clause License

Copyright (c) 2019, Beigesoft™
All rights reserved.

Redistribution and use in source and binary forms, with or without
modification, are permitted provided that the following conditions are met:

* Redistributions of source code must retain the above copyright notice, this
  list of conditions and the following disclaimer.

* Redistributions in binary form must reproduce the above copyright notice,
  this list of conditions and the following disclaimer in the documentation
  and/or other materials provided with the distribution.

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE
FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

package org.beigesoft.ws.srv;

import java.util.List;
import java.util.Arrays;
import java.util.Map;
import java.util.HashMap;
import java.util.Date;

import org.beigesoft.exc.ExcCode;
import org.beigesoft.log.ILog;
import org.beigesoft.mdl.ColVals;
import org.beigesoft.rdb.IOrm;
import org.beigesoft.rdb.IRdb;
import org.beigesoft.rdb.SrvClVl;
import org.beigesoft.ws.mdl.EOrdStat;
import org.beigesoft.ws.mdlp.Buyer;
import org.beigesoft.ws.mdlp.SeItmPlc;
import org.beigesoft.ws.mdlp.ItmPlc;
import org.beigesoft.ws.mdlp.SrvPlc;
import org.beigesoft.ws.mdlp.SeSrvPlc;
import org.beigesoft.ws.mdlp.CuOr;
import org.beigesoft.ws.mdlp.CuOrGdLn;
import org.beigesoft.ws.mdlp.CuOrSrLn;
import org.beigesoft.ws.mdlp.SerBus;
import org.beigesoft.ws.mdlp.SeSerBus;
import org.beigesoft.ws.mdlp.CuOrSe;
import org.beigesoft.ws.mdlp.CuOrSeGdLn;
import org.beigesoft.ws.mdlp.CuOrSeSrLn;
import org.beigesoft.ws.mdlp.Itlist;

/**
 * <p>It cancels all given buyer's orders.
 * E.g. buyer has not paid online after accepting (booking) orders.
 * It changes item's availability and orders status to given NEW or CANCELED.
 * It implements unbooking item available quantity from
 * several places - from the first place.
 * </p>
 *
 * @param <RS> platform dependent RDBMS recordset
 * @author Yury Demidenko
 */
public class CncOrd<RS> implements ICncOrd {

  /**
   * <p>Logger.</p>
   **/
  private ILog log;

  /**
   * <p>ORM service.</p>
   */
  private IOrm orm;

  /**
   * <p>DB service.</p>
   */
  private IRdb<RS> rdb;

  /**
   * <p>Column values service.</p>
   **/
  private SrvClVl srvClVl;

  /**
   * <p>Android configuration, RDB insert returns autogenerated ID,
   * updating with expression like "VER=VER+1" is not possible.</p>
   **/
  private boolean isAndr;

  /**
   * <p>It cancels all given buyer's orders.
   * For example buyer had not paid online after accepting (booking) orders.
   * It changes item's availability and orders status to given NEW or CANCELED.
   * </p>
   * @param pRvs additional request scoped parameters
   * @param pBuyr buyer
   * @param pPurId purchase ID
   * @param pStFr usually BOOKED
   * @param pStTo usually NEW
   * @throws Exception - an exception
   **/
  @Override
  public final void cancel(final Map<String, Object> pRvs,
    final Buyer pBuyr, final Long pPurId,
      final EOrdStat pStFr, final EOrdStat pStTo) throws Exception {
    List<CuOr> ords = null;
    List<CuOrSe> sords = null;
    String wheStBr = "where STAS=" + pStFr.ordinal() + " and BUYR="
      + pBuyr.getIid() + " and PUR=" + pPurId;
    Map<String, Object> vs = new HashMap<String, Object>();
    String[] ndFlNm = new String[] {"nme"};
    vs.put("PicPlcndFds", ndFlNm);
    vs.put("BuyerdpLv", 0);
    ords = this.orm.retLstCnd(pRvs, vs, CuOr.class, wheStBr); vs.clear();
    for (CuOr co : ords) {
      cancel(pRvs, co, pStTo);
    }
    vs.put("SeSelndFds", new String[] {"dbcr"});
    vs.put("DbCrndFds", ndFlNm);
    vs.put("SeSeldpLv", 2);
    vs.put("BuyerdpLv", 0);
    sords = this.orm.retLstCnd(pRvs, vs, CuOrSe.class, wheStBr); vs.clear();
    for (CuOrSe co : sords) {
      cancel(pRvs, co, pStTo);
    }
  }

  /**
   * <p>It cancels given buyer's order.</p>
   * @param pRvs additional request scoped parameters
   * @param pCuOr order
   * @param pStas NEW or CANCELED
   * @throws Exception - an exception
   **/
  @Override
  public final void cancel(final Map<String, Object> pRvs,
    final CuOr pCuOr, final EOrdStat pStas) throws Exception {
    Map<String, Object> vs = new HashMap<String, Object>();
    String[] ndFlGd = new String[] {"ownr", "nme", "good", "uom", "quan", "pri",
      "tot", "toTx"};
    Arrays.sort(ndFlGd);
    String[] ndFlNm = new String[] {"nme"};
    vs.put("CuOrGdLnndFds", ndFlGd);
    vs.put("UomndFds", ndFlNm);
    vs.put("ItmdpLv", 0);
    vs.put("CuOrdpLv", 0);
    List<CuOrGdLn> gds = this.orm.retLstCnd(pRvs, vs, CuOrGdLn.class,
      "where OWNR=" + pCuOr.getIid()); vs.clear();
    ColVals cvVrQu = null;
    String[] verQuan = new String[] {"ver", "quan"};
    Arrays.sort(verQuan);
    if (!this.isAndr) {
      cvVrQu = new ColVals();
      this.srvClVl.putExpr(cvVrQu, "quan");
    }
    for (CuOrGdLn gl : gds) {
      vs.put("ItmPlcndFds", verQuan);
      vs.put("ItmPlcdpLv", 1);
      List<ItmPlc> gps = getOrm().retLstCnd(pRvs, vs, ItmPlc.class,
        "where ALWAY=0 and ITM=" + gl.getGood().getIid()); vs.clear();
      if (gps.size() > 0) {
        if (this.isAndr) {
          vs.put("ndFds", verQuan);
          gps.get(0).setQuan(gps.get(0).getQuan().add(gl.getQuan()));
          getOrm().update(pRvs, vs, gps.get(0)); vs.clear();
          vs.put("ItlistndFds", verQuan);
          Itlist il = getOrm().retEntCnd(pRvs, vs, Itlist.class,
            "TYP=0 and ITID=" + gps.get(0).getItm().getIid());
          il.setQuan(il.getQuan().add(gl.getQuan()));
          vs.put("ndFds", verQuan);
          getOrm().update(pRvs, vs, il); vs.clear();
        } else {
          this.srvClVl.put(cvVrQu, "ver", new Date().getTime());
          this.srvClVl.put(cvVrQu, "quan", "QUAN+" + gl.getQuan());
          this.rdb.update(ItmPlc.class, cvVrQu, "ITM=" + gps.get(0).getItm()
            .getIid() + " and PIPL=" + gps.get(0).getPipl().getIid());
          this.rdb.update(Itlist.class, cvVrQu,
            "TYP=0 and ITID=" + gps.get(0).getItm().getIid());
        }
      }
    }
    String[] ndFlSr = new String[] {"ownr", "nme", "srv", "uom", "quan", "pri",
      "tot", "toTx", "dt1", "dt2"};
    Arrays.sort(ndFlSr);
    vs.put("CuOrSrLnndFds", ndFlSr);
    vs.put("UomndFds", ndFlNm);
    vs.put("SrvdpLv", 0);
    vs.put("CuOrdpLv", 0);
    List<CuOrSrLn> sls = this.orm.retLstCnd(pRvs, vs, CuOrSrLn.class,
      "where OWNR=" + pCuOr.getIid()); vs.clear();
    for (CuOrSrLn sl : sls) {
      if (sl.getDt1() == null) { //non-bookable:
        vs.put("SrvPlcndFds", verQuan);
        vs.put("SrvPlcdpLv", 1);
        List<SrvPlc> sps = getOrm().retLstCnd(pRvs, vs, SrvPlc.class,
          "where ALWAY=0 and ITM=" + sl.getSrv().getIid()); vs.clear();
        if (sps.size() > 0) {
          if (this.isAndr) {
            vs.put("ndFds", verQuan);
            sps.get(0).setQuan(sps.get(0).getQuan().add(sl.getQuan()));
            getOrm().update(pRvs, vs, sps.get(0)); vs.clear();
            vs.put("ItlistndFds", verQuan);
            Itlist il = getOrm().retEntCnd(pRvs, vs, Itlist.class,
              "TYP=1 and ITID=" + sps.get(0).getItm().getIid());
            il.setQuan(il.getQuan().add(sl.getQuan()));
            vs.put("ndFds", verQuan);
            getOrm().update(pRvs, vs, il); vs.clear();
          } else {
            this.srvClVl.put(cvVrQu, "ver", new Date().getTime());
            this.srvClVl.put(cvVrQu, "quan", "QUAN+" + sl.getQuan());
            this.rdb.update(SrvPlc.class, cvVrQu, "ITM=" + sps.get(0).getItm()
              .getIid() + " and PIPL=" + sps.get(0).getPipl().getIid());
            this.rdb.update(Itlist.class, cvVrQu,
              "TYP=1 and ITID=" + sps.get(0).getItm().getIid());
          }
        }
      } else { //bookable:
        List<SerBus> sebs = getOrm().retLstCnd(pRvs, vs, SerBus.class,
          "where FRE=0 and SRV=" + sl.getSrv().getIid() + " and FRTM="
            + sl.getDt1().getTime() + " and TITM=" + sl.getDt1().getTime());
        vs.clear();
        if (sebs.size() == 1) {
          sebs.get(0).setFre(true);
          getOrm().update(pRvs, vs, sebs.get(0));
        } else if (sebs.size() > 1) {
          this.log.error(pRvs, CncOrd.class,
        "Several SERBUS for booked service: " + sl.getSrv().getIid()
      + "/"  + sl.getDt1().getTime() + "/" + sl.getDt1().getTime());
          for (SerBus seb : sebs) {
            seb.setFre(true);
            getOrm().update(pRvs, vs, seb);
          }
        } else {
          this.log.error(pRvs, CncOrd.class,
        "There is no SERBUS for booked service: " + sl.getSrv().getIid()
      + "/"  + sl.getDt1().getTime() + "/" + sl.getDt1().getTime());
        }
      }
    }
    String[] ndFds = new String[] {"ver", "stas", "dscr"};
    Arrays.sort(ndFds);
    vs.put("ndFds", ndFds);
    pCuOr.setStas(pStas);
    getOrm().update(pRvs, vs, pCuOr);
  }

  /**
   * <p>It cancels given buyer's S.E.order.</p>
   * @param pRvs additional request scoped parameters
   * @param pCuOr order
   * @param pStas NEW or CANCELED
   * @throws Exception - an exception
   **/
  @Override
  public final void cancel(final Map<String, Object> pRvs,
    final CuOrSe pCuOr, final EOrdStat pStas) throws Exception {
    Map<String, Object> vs = new HashMap<String, Object>();
    String[] ndFlGd = new String[] {"ownr", "nme", "good", "uom", "quan", "pri",
      "tot", "toTx"};
    Arrays.sort(ndFlGd);
    String[] ndFlNm = new String[] {"nme"};
    vs.put("CuOrSeGdLnndFds", ndFlGd);
    vs.put("UomndFds", ndFlNm);
    vs.put("SeItmdpLv", 0);
    vs.put("CuOrSedpLv", 0);
    List<CuOrSeGdLn> gds = this.orm.retLstCnd(pRvs, vs, CuOrSeGdLn.class,
      "where OWNR=" + pCuOr.getIid()); vs.clear();
    ColVals cvVrQu = null;
    String[] verQuan = new String[] {"ver", "quan"};
    Arrays.sort(verQuan);
    if (!this.isAndr) {
      cvVrQu = new ColVals();
      this.srvClVl.putExpr(cvVrQu, "quan");
    }
    for (CuOrSeGdLn gl : gds) {
      vs.put("SeItmPlcndFds", verQuan);
      vs.put("SeSrvPlcdpLv", 1);
      List<SeItmPlc> gps = getOrm().retLstCnd(pRvs, vs, SeItmPlc.class,
        "where ALWAY=0 and ITM=" + gl.getGood().getIid()); vs.clear();
      if (gps.size() > 0) {
        if (this.isAndr) {
          vs.put("ndFds", verQuan);
          gps.get(0).setQuan(gps.get(0).getQuan().add(gl.getQuan()));
          getOrm().update(pRvs, vs, gps.get(0)); vs.clear();
          vs.put("ItlistndFds", verQuan);
          Itlist il = getOrm().retEntCnd(pRvs, vs, Itlist.class,
            "TYP=2 and ITID=" + gps.get(0).getItm().getIid());
          il.setQuan(il.getQuan().add(gl.getQuan()));
          vs.put("ndFds", verQuan);
          getOrm().update(pRvs, vs, il); vs.clear();
        } else {
          this.srvClVl.put(cvVrQu, "ver", new Date().getTime());
          this.srvClVl.put(cvVrQu, "quan", "QUAN+" + gl.getQuan());
          this.rdb.update(SeItmPlc.class, cvVrQu, "ITM=" + gps.get(0).getItm()
            .getIid() + " and PIPL=" + gps.get(0).getPipl().getIid());
          this.rdb.update(Itlist.class, cvVrQu, "TYP=2 and ITID="
            + gps.get(0).getItm().getIid());
        }
      }
    }
    String[] ndFlSr = new String[] {"ownr", "nme", "srv", "uom", "quan", "pri",
      "tot", "toTx", "dt1", "dt2"};
    Arrays.sort(ndFlSr);
    vs.put("CuOrSeSrLnndFds", ndFlSr);
    vs.put("UomndFds", ndFlNm);
    vs.put("SeSrvdpLv", 0);
    vs.put("CuOrSedpLv", 0);
    List<CuOrSeSrLn> sls = this.orm.retLstCnd(pRvs, vs, CuOrSeSrLn.class,
      "where OWNR=" + pCuOr.getIid()); vs.clear();
    for (CuOrSeSrLn sl : sls) {
      if (sl.getDt1() == null) { //non-bookable:
        vs.put("SeSrvPlcndFds", verQuan);
        vs.put("SeSrvPlcdpLv", 1);
        List<SeSrvPlc> sps = getOrm().retLstCnd(pRvs, vs, SeSrvPlc.class,
          "where ALWAY=0 and ITM=" + sl.getSrv().getIid()); vs.clear();
        if (sps.size() > 0) {
          if (this.isAndr) {
            vs.put("ndFds", verQuan);
            sps.get(0).setQuan(sps.get(0).getQuan().add(sl.getQuan()));
            getOrm().update(pRvs, vs, sps.get(0)); vs.clear();
            vs.put("ItlistndFds", verQuan);
            Itlist il = getOrm().retEntCnd(pRvs, vs, Itlist.class,
              "TYP=3 and ITID=" + sps.get(0).getItm().getIid());
            il.setQuan(il.getQuan().add(sl.getQuan()));
            vs.put("ndFds", verQuan);
            getOrm().update(pRvs, vs, il); vs.clear();
          } else {
            this.srvClVl.put(cvVrQu, "ver", new Date().getTime());
            this.srvClVl.put(cvVrQu, "quan", "QUAN+" + sl.getQuan());
            this.rdb.update(SeSrvPlc.class, cvVrQu, "ITM=" + sps.get(0).getItm()
              .getIid() + " and PIPL=" + sps.get(0).getPipl().getIid());
            this.rdb.update(Itlist.class, cvVrQu,
              "TYP=3 and ITID=" + sps.get(0).getItm().getIid());
          }
        }
      } else { //bookable:
        List<SeSerBus> sebs = getOrm().retLstCnd(pRvs, vs, SeSerBus.class,
          "where FRE=0 and SRV=" + sl.getSrv().getIid() + " and FRTM="
            + sl.getDt1().getTime() + " and TITM=" + sl.getDt1().getTime());
        if (sebs.size() == 1) {
          sebs.get(0).setFre(true); //TODO ndFds
          getOrm().update(pRvs, vs, sebs.get(0));
        } else if (sebs.size() > 1) {
          this.log.error(pRvs, CncOrd.class,
        "Several SESERBUS for booked SeSrv: " + sl.getSrv().getIid()
      + "/"  + sl.getDt1().getTime() + "/" + sl.getDt1().getTime());
          for (SeSerBus seb : sebs) {
            seb.setFre(true);
            getOrm().update(pRvs, vs, seb);
          }
        } else {
          this.log.error(pRvs, CncOrd.class,
      "There is no SESERBUS for booked SeSrv: " + sl.getSrv().getIid()
    + "/"  + sl.getDt1().getTime() + "/" + sl.getDt1().getTime());
          throw new ExcCode(ExcCode.WRPR, "serv_no_avlb");
        }
      }
    }
    pCuOr.setStas(pStas);
    String[] ndFds = new String[] {"ver", "stas", "dscr"};
    Arrays.sort(ndFds);
    vs.put("ndFds", ndFds);
    getOrm().update(pRvs, vs, pCuOr);
  }

  //Simple getters and setters:
  /**
   * <p>Geter for log.</p>
   * @return ILog
   **/
  public final ILog getLog() {
    return this.log;
  }

  /**
   * <p>Setter for log.</p>
   * @param pLogger reference
   **/
  public final void setLog(final ILog pLogger) {
    this.log = pLogger;
  }

  /**
   * <p>Getter for orm.</p>
   * @return IOrm
   **/
  public final IOrm getOrm() {
    return this.orm;
  }

  /**
   * <p>Setter for orm.</p>
   * @param pOrm reference
   **/
  public final void setOrm(final IOrm pOrm) {
    this.orm = pOrm;
  }

  /**
   * <p>Getter for rdb.</p>
   * @return IRdb<RS>
   **/
  public final IRdb<RS> getRdb() {
    return this.rdb;
  }

  /**
   * <p>Setter for rdb.</p>
   * @param pRdb reference
   **/
  public final void setRdb(final IRdb<RS> pRdb) {
    this.rdb = pRdb;
  }

  /**
   * <p>Getter for isAndr.</p>
   * @return boolean
   **/
  public final boolean getIsAndr() {
    return this.isAndr;
  }

  /**
   * <p>Setter for isAndr.</p>
   * @param pIsAndr reference
   **/
  public final void setIsAndr(final boolean pIsAndr) {
    this.isAndr = pIsAndr;
  }

  /**
   * <p>Getter for srvClVl.</p>
   * @return SrvClVl
   **/
  public final SrvClVl getSrvClVl() {
    return this.srvClVl;
  }

  /**
   * <p>Setter for srvClVl.</p>
   * @param pSrvClVl reference
   **/
  public final void setSrvClVl(final SrvClVl pSrvClVl) {
    this.srvClVl = pSrvClVl;
  }
}
