//  Copyright © 2014-present BetterJapanese
//  
//  Licensed under the Apache License, Version 2.0 (the "License");
//  you may not use this file except in compliance with the License.
//  You may obtain a copy of the License at
//
//      http://www.apache.org/licenses/LICENSE-2.0
//
//  Unless required by applicable law or agreed to in writing, software
//  distributed under the License is distributed on an "AS IS" BASIS,
//  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
//  See the License for the specific language governing permissions and
//  limitations under the License.

package org.betterjapanese.spock.extension

import org.spockframework.runtime.extension.AbstractAnnotationDrivenExtension
import org.spockframework.runtime.model.FeatureInfo
import org.spockframework.runtime.model.MethodInfo

/**
 * Adds an interceptor to the feature method annotated with {@link SetupFeature}.
 *
 * No interceptor is added to the feature method if:
 *
 * {@link SetupFeatures} already specifies an interceptor with the same
 * invokee, unless {@link SetupFeatures} is ignored.
 */
class SetupFeatureExtension extends
      AbstractAnnotationDrivenExtension<SetupFeature> {

    @Override
    void visitFeatureAnnotation(SetupFeature annotation, FeatureInfo feature) {
        
        def invokee = annotation.value()
        
        boolean ignore
        boolean intercepted

        MethodInfo method = feature.featureMethod
        ignore = method.reflection.isAnnotationPresent(IgnoreSetupFeatures)

        if (!ignore) {
            feature.interceptors.each { interceptor ->
                if (interceptor instanceof SetupFeatureInterceptor)
                    if (interceptor.invokee == invokee)
                        intercepted = true
            }
        }

        if (!intercepted) {
            def interceptor = new SetupFeatureInterceptor(invokee: invokee)
            feature.addInterceptor(interceptor)
        }
    }
}
