//  Copyright © 2014-present BetterJapanese
//  
//  Licensed under the Apache License, Version 2.0 (the "License");
//  you may not use this file except in compliance with the License.
//  You may obtain a copy of the License at
//
//      http://www.apache.org/licenses/LICENSE-2.0
//
//  Unless required by applicable law or agreed to in writing, software
//  distributed under the License is distributed on an "AS IS" BASIS,
//  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
//  See the License for the specific language governing permissions and
//  limitations under the License.

package org.betterjapanese.spock.extension

import org.betterjapanese.spock.extension.IgnoreSetupFeatures

import org.spockframework.runtime.extension.AbstractAnnotationDrivenExtension
import org.spockframework.runtime.model.SpecInfo
import org.spockframework.runtime.model.MethodInfo

/**
 * Adds an interceptor to each feature method in specifications annotated with
 * {@link SetupFeatures}.
 *
 * No interceptor is added to a feature method if:
 *
 * The feature method is annotated with {@link IgnoreSetupFeatures}.
 */
class SetupFeaturesExtension extends
      AbstractAnnotationDrivenExtension<SetupFeatures> {

    @Override
    void visitSpecAnnotation(SetupFeatures annotation, SpecInfo spec) {

        def invokee = annotation.value()
        def interceptor = new SetupFeatureInterceptor(invokee: invokee)
        
        spec.getFeatures().each { feature ->
            boolean ignored
            MethodInfo method = feature.featureMethod
            ignored = method.reflection.isAnnotationPresent(IgnoreSetupFeatures)
            if (!ignored) feature.addInterceptor(interceptor)
        }
    }
}
