package com.github.markusbernhardt.proxy.selector.pac;

import java.io.File;
import java.net.MalformedURLException;
import java.util.Calendar;

import org.assertj.core.api.Assertions;
import org.junit.jupiter.api.AfterAll;
import org.junit.jupiter.api.BeforeAll;
import org.junit.jupiter.api.Test;

import com.github.markusbernhardt.proxy.TestUtil;
import com.github.markusbernhardt.proxy.util.ProxyException;

/*****************************************************************************
 * Tests for the javax.script PAC script parser.
 * 
 * @author Markus Bernhardt, Copyright 2016
 * @author Bernd Rosstauscher, Copyright 2009
 ****************************************************************************/

public class JavaxPacScriptParserTest {

    /*************************************************************************
     * Set calendar for date and time base tests. Current date for all tests is: 15. December 1994 12:00.00 its a
     * Thursday
     ************************************************************************/
    @BeforeAll
    public static void setup() {
        Calendar cal = Calendar.getInstance();
        cal.set(Calendar.YEAR, 1994);
        cal.set(Calendar.MONTH, Calendar.DECEMBER);
        cal.set(Calendar.DAY_OF_MONTH, 15);
        cal.set(Calendar.HOUR_OF_DAY, 12);
        cal.set(Calendar.MINUTE, 00);
        cal.set(Calendar.SECOND, 00);
        cal.set(Calendar.MILLISECOND, 00);

        // TODO Rossi 26.08.2010 need to fake time
        // JavaxPacScriptParser.setCurrentTime(cal);
    }

    /*************************************************************************
     * Cleanup after the tests.
     ************************************************************************/
    @AfterAll
    public static void teadDown() {
        // JavaxPacScriptParser.setCurrentTime(null);
    }

    /*************************************************************************
     * Test method
     * 
     * @throws ProxyException
     *             on proxy detection error.
     * @throws MalformedURLException
     *             on URL erros
     ************************************************************************/
    @Test
    public void testScriptExecution() throws ProxyException, MalformedURLException {
        PacScriptParser p = new JavaxPacScriptParser(new UrlPacScriptSource(toUrl("test1.pac")));
        p.evaluate(TestUtil.HTTP_TEST_URI.toString(), "host1.unit-test.invalid");
    }
    
     /*************************************************************************
     * Tests that a pac script file that has the BOM character at the beginning is
     * read and does not throw an exception.
     * 
     * @throws ProxyException
     *             on proxy detection error.
     * @throws MalformedURLException
     *             on URL erros
     ************************************************************************/
    @Test
    public void testBomExecution() throws ProxyException, MalformedURLException {
        PacScriptParser p = new JavaxPacScriptParser(new UrlPacScriptSource(toUrl("pacWithBom.pac")));
        String result = p.evaluate(TestUtil.HTTP_TEST_URI.toString(), "host1.test");
        Assertions.assertThat(result).contains("DIRECT");
    }

    /*************************************************************************
     * Test method
     * 
     * @throws ProxyException
     *             on proxy detection error.
     * @throws MalformedURLException
     *             on URL erros
     ************************************************************************/
    @Test
    public void testCommentsInScript() throws ProxyException, MalformedURLException {
        PacScriptParser p = new JavaxPacScriptParser(new UrlPacScriptSource(toUrl("test2.pac")));
        p.evaluate(TestUtil.HTTP_TEST_URI.toString(), "host1.unit-test.invalid");
    }

    /*************************************************************************
     * Test that a space in an IP address can still be parsed.
     * 
     * @throws ProxyException
     *             on proxy detection error.
     * @throws MalformedURLException
     *             on URL errors
     ************************************************************************/
    @Test
    public void testSpaceInScript() throws ProxyException, MalformedURLException {
        PacScriptParser p = new JavaxPacScriptParser(new UrlPacScriptSource(toUrl("test3.pac")));
        String result = p.evaluate(TestUtil.HTTP_TEST_URI.toString(), "host1.unit-test.invalid");
        Assertions.assertThat(result).isNull();
    }

    /*************************************************************************
     * Test method
     * 
     * @throws ProxyException
     *             on proxy detection error.
     * @throws MalformedURLException
     *             on URL erros
     ************************************************************************/
    @Test
    public void testScriptWeekDayScript() throws ProxyException, MalformedURLException {
        PacScriptParser p = new JavaxPacScriptParser(new UrlPacScriptSource(toUrl("testWeekDay.pac")));
        p.evaluate(TestUtil.HTTP_TEST_URI.toString(), "host1.unit-test.invalid");
    }

    /*************************************************************************
     * Test method
     * 
     * @throws ProxyException
     *             on proxy detection error.
     * @throws MalformedURLException
     *             on URL erros
     ************************************************************************/
    @Test
    public void testDateRangeScript() throws ProxyException, MalformedURLException {
        PacScriptParser p = new JavaxPacScriptParser(new UrlPacScriptSource(toUrl("testDateRange.pac")));
        p.evaluate(TestUtil.HTTP_TEST_URI.toString(), "host1.unit-test.invalid");
    }

    /*************************************************************************
     * Test method
     * 
     * @throws ProxyException
     *             on proxy detection error.
     * @throws MalformedURLException
     *             on URL erros
     ************************************************************************/
    @Test
    public void testTimeRangeScript() throws ProxyException, MalformedURLException {
        PacScriptParser p = new JavaxPacScriptParser(new UrlPacScriptSource(toUrl("testTimeRange.pac")));
        p.evaluate(TestUtil.HTTP_TEST_URI.toString(), "host1.unit-test.invalid");
    }

    /*************************************************************************
     * Test method
     * 
     * @throws ProxyException
     *             on proxy detection error.
     * @throws MalformedURLException
     *             on URL erros
     ************************************************************************/
    @Test
    public void methodsShouldReturnJsStrings() throws ProxyException, MalformedURLException {
        PacScriptParser p = new JavaxPacScriptParser(new UrlPacScriptSource(toUrl("testReturnTypes.pac")));
        String actual = p.evaluate(TestUtil.HTTP_TEST_URI.toString(), "host1.unit-test.invalid");
        Assertions.assertThat(actual).isEqualTo("number boolean string");
    }

    /*************************************************************************
     * Test method
     * 
     * @throws ProxyException
     *             on proxy detection error.
     * @throws MalformedURLException
     *             on URL erros
     ************************************************************************/
    @Test
    public void shouldNotExecuteCodeInPac() throws ProxyException, MalformedURLException {
        PacScriptParser p = new JavaxPacScriptParser(new UrlPacScriptSource(toUrl("testRemoteCodeExecution.pac")));
        Assertions.assertThatThrownBy(()-> p.evaluate(TestUtil.HTTP_TEST_URI.toString(), "host.does.not.matter")).isInstanceOf(Exception.class);
    }

    /*************************************************************************
     * Helper method to build the url to the given test file
     * 
     * @param testFile
     *            the name of the test file.
     * @return the URL.
     * @throws MalformedURLException
     ************************************************************************/

    private String toUrl(String testFile) throws MalformedURLException {
        return new File(TestUtil.TEST_DATA_FOLDER + "pac", testFile).toURI().toURL().toString();
    }

}
