/*
 * Copyright (c) 2015 SureLogic, Inc.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package com.surelogic;

import java.lang.annotation.Documented;
import java.lang.annotation.ElementType;
import java.lang.annotation.Retention;
import java.lang.annotation.RetentionPolicy;
import java.lang.annotation.Target;

/**
 * Declares a promise that applies to multiple declarations within the scope of
 * code that the annotation appears on. The general form is
 * <p>
 * <tt>&#064;Promise("</tt><i>promise</i> <tt>for</tt> <i>target</i><tt>")</tt>
 * <p>
 * Where the <i>promise</i> is placed on all declarations matched by
 * <i>target</i>. It is possible to omit the <code>for</code> clause which
 * indicates that <i>promise</i> should be placed on all meaningful declarations
 * within the type or package. For example,
 * <code>&#064;Promise("&#064;Borrowed(this)")</code> on a type places a
 * <code>&#064;Borrowed("this")</code> promise on all the method and
 * constructors declared within the type.
 * <p>
 * The syntax for <i>promise</i> is the same as if the promise was written in
 * the code except that all the quotation marks (&quot;) are removed (e.g.,
 * <code>&#064;Borrowed("this")</code> becomes <code>&#064;Borrowed(this)</code>
 * ).
 * <p>
 * One handy feature of this annotation is that it can be used to place promises
 * on the implicit no-argument constructor that is generated by the compiler if
 * a class contains no explicit constructor. The snippet below places a
 * <code>&#064;Borrowed("this")</code> on the implicit (or explicit) no-argument
 * constructor for {@code Implicit}.
 * 
 * <pre>
 * &#064;Promise(&quot;&#064;Borrowed(this) for new()&quot;)
 * class Implicit { ... }
 * </pre>
 * 
 * To declare more than one scoped promise for an entity use the
 * {@link Promises} annotation. It is a modeling error for an entity to have
 * both a {@link Promises} and a {@link Promise} annotation.
 * <p>
 * For specific annotations, <code>&#064;Promise</code> can be used to <i>change
 * the default</i> in the sense that it can set a default annotation on each
 * targeted declaration that the programmer can override with a more specific
 * annotation on the actual declaration. The specifics of this behavior are
 * listed in the <i>Semantics</i> section below.
 * 
 * <h3>Semantics:</h3>
 * 
 * This annotation creates one or more "virtual" promises within the scope of
 * the declaration to which the annotation is applied. From the point of view of
 * the verifying analyses, it is as if these virtual promises were annotated by
 * the programmer&mdash;and they are to be verified. The results reported from
 * tools, such as SureLogic JSure, should "link" each virtual promise to its
 * associated {@link Promise} annotation to make it clear to the programmer the
 * source of each virtual promise.
 * <p>
 * In the case of specific annotations this promise can <i>change the
 * default</i> in the sense that it can set a default annotation on each
 * declaration that the programmer can override with a more specific annotation.
 * This is the case for the annotations listed below.
 * <dl>
 * <dt><code>&#064;NonNull</code>, <code>&#064;Nullable</code>, and
 * <code>&#064;Raw</code>
 * <dd>An explicit null value annotation on a declaration replaces any virtual
 * annotation placed on the same declaration by a <code>&#064;Promise</code>.
 * For example, <code>&#064;Promise("&#064;NonNull")</code> in a
 * <tt>package-info.java</tt> file can be used change the default for a whole
 * package. Within that package <code>&#064;Nullable</code> and
 * <code>&#064;Raw</code> annotations made on particular declarations replace
 * the <code>&#064;NonNull</code> intent from the <code>&#064;Promise</code>.
 * 
 * <dt><code>&#064;RegionEffects</code></dt>
 * <dd>An explicit effects annotation on a declaration replaces one placed on
 * the same declaration by a <code>&#064;Promise</code>.
 * </dl>
 * 
 * <h3>Examples:</h3>
 * 
 * Consider the following example {@link Promise} annotations on a class
 * {@code C}:
 *
 * <dl>
 * <dt>&#064;Promise("&#064;Borrowed(this)")</dt>
 * <dd>Places the promise on all constructors and methods in {@code C} because
 * <code>&#064;Borrowed("this")</code> only applies to constructors and methods.
 * </dd>
 * 
 * <dt>&#064;Promise("&#064;Borrowed(this) for new(**)")</dt>
 * <dd>Applies to any constructor in {@code C}. The '<b>**</b>' pattern
 * indicates any number of parameters including zero.</dd>
 * 
 * <dt>&#064;Promise("&#064;Borrowed(this) for !static **(**)")</dt>
 * <dd>Applies to any instance method or constructor declared in {@code C}. The
 * '<b>**(**)</b>' pattern matches both methods and constructors. The
 * '<b>*(**)</b>' pattern only matches methods.</dd>
 * 
 * <dt>&#064;Promise("&#064;Borrowed(this) for !static *(**) | new(**)")</dt>
 * <dd>Equivalent to the promise above but without using the '<b>**(**)</b>'
 * shorthand.</dd>
 * 
 * <dt>
 * &#064;Promise(
 * "&#064;RequiresLock(MyLock) for some*(**) &amp; !(someoneElse())")</dt>
 * <dd>Applies to any method in {@code C} with a name starting with
 * <code>some</code>, except for a no-argument method named
 * <code>someoneElse</code>.</dd>
 * 
 * <dt>&#064;Promise("&#064;InRegion(MyRegion) for * mutable* | int *")</dt>
 * <dd>Applies to fields in {@code C} declared of any type with names starting
 * with <code>mutable</code>, or of type {@code int} with any name.<br>
 * </dl>
 * 
 * The scoped promise {@link Promise} can be used if the constructor is
 * implicit. It has the ability to place promises on implicit and explicit
 * constructors. The below example places the annotaiton on the implicit no-arg
 * constructor:
 * 
 * <pre>
 * &#064;RegionLock(&quot;Lock is this protects Instance&quot;)
 * &#064;Promise(&quot;&#064;Borrowed(this) for new(**)&quot;)
 * public class Example {
 *   int x = 1;
 *   int y = 1;
 *   ...
 * }
 * </pre>
 * 
 * Here is an example of <i>changing the default</i> of all references in a
 * package to <code>&#064;NonNull</code>. In the file
 * <tt>com/stuff/package-info.java</tt>
 * 
 * <pre>
 * &#064;com.surelogic.Promise(&quot;@NonNull&quot;)
 * package com.stuff;
 * </pre>
 * 
 * This makes all declarations in the package <code>&#064;NonNull</code>. You
 * can specifically annotate a particular declaration to make it
 * <code>&#064;Nullable</code> or <code>&#064;Raw</code> such as is done on the
 * <code>getDestinationNameOrNull</code> method in the listing below.
 * 
 * <pre>
 * package com.stuff;
 * 
 * public class Flight {
 *   ...
 *   &#064;Nullable
 *   public String getDestinationNameOrNull() { ... }
 * }
 * </pre>
 * 
 * A second <code>&#064;Promise</code> annotation can be used to alter the
 * default for a particular type. In the <code>LegacyOldCode</code> class listed
 * below the default is changed back to <code>&#064;Nullable</code>
 * &mdash;overriding the <code>&#064;Promise</code> annotation
 * <code>package-info.java</code> file.
 * 
 * <pre>
 * package com.stuff;
 * 
 * &#064;Promise("&#064;Nullable")
 * public class LegacyOldCode {
 *   ...
 *   &#064;NonNull
 *   public String getLegacyValue() { ... }
 * }
 * </pre>
 * 
 * Here is an example of <em>changing the default</em> effects of all
 * constructors in a package to <code>&#064;RegionEffects("none")</code>. In the
 * file <tt>com/stuff/package-info.java</tt>
 * 
 * <pre>
 * &#064;com.surelogic.Promise(&quot;@RegionEffects(none) for new(**)&quot;)
 * package com.stuff;
 * </pre>
 * 
 * Any particular constructor within the package can override this default with
 * a specific annotation. For example the constructor for <code>Flight</code> in
 * the listing below declares <i>reads All</i> as its effects because it perhaps
 * reads several static variables in the codebase.
 * 
 * <pre>
 * package com.stuff;
 * 
 * public class Flight {
 *   ...
 *   &#064;RegionEffects("reads All")
 *   public Flight(int number) { ... }
 * }
 * </pre>
 * 
 * @see Promises
 */
@Documented
@Retention(RetentionPolicy.RUNTIME)
@Target({ ElementType.PACKAGE, ElementType.TYPE })
public @interface Promise {
  /**
   * The value of this attribute must conform to the following grammar (in <a
   * href="http://www.ietf.org/rfc/rfc4234.txt">Augmented Backus&ndash;Naur
   * Form</a>):
   * 
   * <pre>
   * value = Payload [ &quot;for&quot; Target ]
   * 
   * Payload = A promise with &quot;'s removed, e.g., &#064;Borrowed("this") becomes &#064;Borrowed(this)
   * 
   * Target =
   *   TypeDecPat / FieldDecPat / MethodDecPat/ ConstructorDecPat /
   *   Target &quot;|&quot; Target /
   *   Target &quot;&amp;&quot; Target /
   *   [ &quot;!&quot; ] &quot;(&quot; Target &quot;)&quot;
   *   
   * TypeDecPat = [ Visibility ] ( QualifiedName / IdentifierPat [ InPackagePat ] )
   * FieldDecPat = [ Visibility ] [ Static ] TypeSigPat IdentifierPat [ InTypePat ]
   * MethodDecPat = [ Visibility ] [ Static ] IdentifierPat &quot;(&quot; [ ParameterSigPat ] &quot;)&quot; [ InTypePat ]
   * ConstructorDecPat = [ Visibility ] &quot;new&quot; &quot;(&quot; [ ParameterSigPat ] &quot;)&quot; [ InTypePat ]
   * 
   * TypeSigPat =
   *   &quot;*&quot; /
   *   ( &quot;boolean&quot; / &quot;char&quot; / &quot;byte&quot; / &quot;short&quot; /
   *     &quot;int&quot; / &quot;long&quot; / &quot;float&quot; / &quot;double&quot; / &quot;void&quot; /
   *     IdentifierPat / QualifiedName ) *( &quot;[]&quot; )
   * ParameterSigPat = ( TypeSigPat / &quot;**&quot; ) *(&quot;,&quot; ( TypeSigPat / &quot;**&quot; ) )
   * 
   * InNameExp =
   *   QualifiedNamePat /
   *   InNameExp &quot;|&quot; InNameExp /
   *   InNameExp &quot;&amp;&quot; InNameExp /
   *   [ &quot;!&quot; ] &quot;(&quot; InNameExp &quot;)&quot;
   * InNamePat =
   *   QualifiedNamePat /
   *   &quot;(&quot; InNameExp &quot;)&quot;
   * InPackagePat = &quot;in&quot; InNamePat
   * InTypePat    = &quot;in&quot; InNamePat [ InPackagePat ]
   * 
   * Visibility = &quot;public&quot; / &quot;protected&quot; / &quot;private&quot;
   * Static = [ &quot;!&quot; ] &quot;static&quot;
   * IdentifierPat = [ &quot;*&quot; ] ( Identifier *(&quot;*&quot; Identifier) ) [ &quot;*&quot; ]
   * QualifiedNamePat = ( IdentifierPat / &quot;*&quot; / &quot;**&quot; ) *(&quot;.&quot; ( IdentifierPat / &quot;*&quot; / &quot;**&quot; ) )
   * QualifiedName = Identifier *(&quot;.&quot; Identifier)
   * 
   * IDENTIFIER = Legal Java Identifier
   * </pre>
   * 
   * @return a value following the syntax described above.
   */
  String value();
}
