package org.bidib.wizard.camel.stream;

import java.util.HashMap;
import java.util.Map;

import javax.annotation.PostConstruct;

import org.apache.camel.CamelContext;
import org.apache.camel.builder.RouteBuilder;
import org.bidib.api.json.types.ConnectionRegistryEvent;
import org.bidib.wizard.api.model.connection.BidibConnection;
import org.bidib.wizard.camel.config.MessageStreamConfiguration;
import org.bidib.wizard.camel.stream.processor.BidibMessageProcessor;
import org.bidib.wizard.core.model.connection.ConnectionRegistry;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.boot.context.properties.EnableConfigurationProperties;
import org.springframework.stereotype.Service;

import io.reactivex.rxjava3.core.Observer;
import io.reactivex.rxjava3.disposables.Disposable;
import io.reactivex.rxjava3.subjects.BehaviorSubject;

@Service
@EnableConfigurationProperties({ MessageStreamConfiguration.class })
public class ConnectionObserver {

    private static final Logger LOGGER = LoggerFactory.getLogger(ConnectionObserver.class);

    @Autowired
    private MessageStreamConfiguration messageStreamConfiguration;

    @Autowired
    private ConnectionRegistry connectionRegistry;

    @Autowired
    private BidibMessageStreamFactory bidibMessageStreamFactory;

    private Map<String, BidibMessageStreamFunction> messageStreamRegistry = new HashMap<>();

    @Autowired
    private CamelContext context;

    @Autowired
    private BidibMessageProcessor bidibMessageProcessor;

    @PostConstruct
    public void initialize() {
        LOGGER.info("Initialize the ConnectionObserver.");

        context.start();

        final BehaviorSubject<ConnectionRegistryEvent> connectionRegistrySubject =
            connectionRegistry.getConnectionRegistrySubject();
        connectionRegistrySubject.subscribe(new Observer<ConnectionRegistryEvent>() {

            @Override
            public void onSubscribe(Disposable d) {
                LOGGER.info("Subscribed to connection registry changes, disposed: {}", d.isDisposed());
            }

            @Override
            public void onNext(ConnectionRegistryEvent evt) {

                LOGGER.info("New connection event received from connection registry: {}", evt);

                // TODO use a scheduler ... async

                final String connectionId = evt.getConnectionId();
                switch (evt.getConnectionRegistryState()) {
                    case ADD:

                        if (messageStreamConfiguration.isStreamsEnabled()) {
                            createStream(connectionId);
                        }
                        else {
                            LOGGER.info("No streams enabled, skip add connection.");
                        }
                        break;
                    default:

                        synchronized (messageStreamRegistry) {
                            LOGGER.info("Remove messageStream from registry, connectionId: {}", connectionId);
                            BidibMessageStreamFunction messageStreamSafe = messageStreamRegistry.remove(connectionId);
                            if (messageStreamSafe != null) {
                                messageStreamSafe.shutdown();
                            }
                        }

                        break;
                }

            }

            @Override
            public void onError(Throwable e) {

                LOGGER.info("Subscription to connection registry changes has thrown an error: {}", e);
            }

            @Override
            public void onComplete() {

                LOGGER.info("The subscription to connection registry changes has finished.");
            }
        });

    }

    private void createStream(String connectionId) {

        LOGGER.info("Create new messageStream for connectionId: {}", connectionId);

        BidibConnection connection = connectionRegistry.getConnection(connectionId);

        BidibMessageStreamFunction messageStream = bidibMessageStreamFactory.createMessageStream(connection);

        synchronized (messageStreamRegistry) {
            messageStreamRegistry.put(connectionId, messageStream);
        }

        LOGGER.info("Created new messageStream: {}", messageStream);

        try {

            context.addRoutes(new RouteBuilder() {

                @Override
                public void configure() throws Exception {

                    from(EndpointDefines.DIRECT_BIDIB_MESSAGES)
                        // .log(LoggingLevel.INFO, "bidibMessage", "Received event: " + body())
                        .process(bidibMessageProcessor);

                }

            });

            LOGGER.info("Execute the stream passed.");
        }
        catch (Exception ex) {
            LOGGER.warn("Execute the stream failed.", ex);
        }

    }
}
