package org.bidib.wizard.mvc.error.view;

import java.util.concurrent.atomic.AtomicBoolean;
import java.util.function.Supplier;

import javax.swing.SwingUtilities;

import org.bidib.wizard.dialog.ExceptionDialog;
import org.bidib.wizard.mvc.error.model.ErrorModel;
import org.bidib.wizard.mvc.error.model.listener.ErrorListener;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public abstract class ErrorView {

    private static final Logger LOGGER = LoggerFactory.getLogger(ErrorView.class);

    private final AtomicBoolean dialogVisible = new AtomicBoolean();

    public ErrorView(final ErrorModel model, final Supplier<Boolean> startupPassed) {

        model.addErrorListener(new ErrorListener() {
            @Override
            public void errorChanged(final String errorMessage, final String stackTrace) {
                LOGGER.warn("Show the error message: {}", errorMessage);

                if (dialogVisible.getAndSet(true)) {
                    LOGGER.warn("The dialog is visible already. Only show single instance.");
                    return;
                }

                if (!SwingUtilities.isEventDispatchThread()) {
                    try {
                        SwingUtilities.invokeLater(() -> {
                            try {
                                showErrorDialog(errorMessage, stackTrace);

                                model.resetStackTrace();

                                // TODO call windwClosed again?
                                if (!startupPassed.get().booleanValue()) {
                                    windowClosed();
                                }
                            }
                            catch (Exception ex) {
                                LOGGER.warn("Show error dialog failed", ex);
                            }
                        });
                    }
                    catch (Exception ex) {
                        LOGGER.warn("Show error dialog failed", ex);
                    }
                }
                else {
                    try {
                        showErrorDialog(errorMessage, stackTrace);

                        model.resetStackTrace();

                        // TODO call windwClosed again?
                        if (!startupPassed.get().booleanValue()) {
                            windowClosed();
                        }
                    }
                    catch (Exception ex) {
                        LOGGER.warn("Show error dialog failed", ex);
                    }
                }
            }
        });

    }

    private void showErrorDialog(final String errorMessage, final String stackTrace) {

        final ExceptionDialog dialog = new ExceptionDialog() {

            private static final long serialVersionUID = 1L;

            @Override
            protected Supplier<String> getExceptionMessageSupplier() {
                return () -> stackTrace;
            }

            @Override
            protected Supplier<String> getErrorMessageSupplier() {
                return () -> errorMessage;
            }
        };

        dialog.pack();
        dialog.setLocationRelativeTo(null);
        dialog.setVisible(true);

        dialogVisible.set(false);
    }

    public abstract void windowClosed();
}
