package org.bidib.wizard.mvc.main.model;

import java.util.function.IntConsumer;

import org.bidib.jbidibc.messages.enums.PortConfigKeys;
import org.bidib.wizard.api.locale.Resources;
import org.bidib.wizard.client.common.model.SimpleHierarchicalPortTableModel;
import org.bidib.wizard.model.ports.BacklightPort;
import org.bidib.wizard.model.status.BacklightPortStatus;
import org.bidib.wizard.mvc.main.model.listener.BacklightPortModelListener;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class BacklightPortTableModel
    extends SimpleHierarchicalPortTableModel<BacklightPortStatus, BacklightPort, BacklightPortModelListener> {
    private static final long serialVersionUID = 1L;

    private static final Logger LOGGER = LoggerFactory.getLogger(BacklightPortTableModel.class);

    public static final int COLUMN_LABEL = 0;

    public static final int COLUMN_DIM_SLOPE_UP = 1;

    public static final int COLUMN_DIM_SLOPE_DOWN = 2;

    public static final int COLUMN_DMX_MAPPING = 3;

    public static final int COLUMN_VALUE = 4;

    public static final int COLUMN_PORT_INSTANCE = 5;

    public BacklightPortTableModel() {
    }

    @Override
    public int getColumnPortInstance() {
        return COLUMN_PORT_INSTANCE;
    }

    @Override
    protected void initialize() {
        super.setPortValuesProcessing(true);
        columnNames =
            new String[] { Resources.getString(BacklightPortTableModel.class, "label"),
                Resources.getString(BacklightPortTableModel.class, "dimSlopeDown"),
                Resources.getString(BacklightPortTableModel.class, "dimSlopeUp"),
                Resources.getString(BacklightPortTableModel.class, "dmxMapping"),
                Resources.getString(BacklightPortTableModel.class, "test"), null };
    }

    @Override
    public void addRow(BacklightPort port) {
        if (port != null) {
            Object[] rowData = new Object[columnNames.length];

            rowData[COLUMN_LABEL] = port.toString();
            rowData[COLUMN_DIM_SLOPE_DOWN] = port.getDimSlopeDown();
            rowData[COLUMN_DIM_SLOPE_UP] = port.getDimSlopeUp();
            rowData[COLUMN_DMX_MAPPING] = port.getDmxMapping();
            rowData[COLUMN_VALUE] = port.getValue();
            rowData[COLUMN_PORT_INSTANCE] = port;

            addRow(rowData);
        }
    }

    @Override
    public boolean isCellEditable(int row, int column) {
        boolean isEditable = false;
        // BacklightPort lightPort = (BacklightPort) getValueAt(row, COLUMN_PORT_INSTANCE);
        // switch (column) {
        // case COLUMN_VALUE:
        // // the test can be changed.
        // // isEditable = true;
        // if (Port.getConfiguredPortType(lightPort) == LcOutputType.BACKLIGHTPORT) {
        // isEditable = true;
        // }
        // break;
        // default:
        // break;
        // }
        return isEditable;
    }

    @Override
    public Class<?> getColumnClass(int column) {
        switch (column) {
            case COLUMN_LABEL:
                return String.class;
            case COLUMN_PORT_INSTANCE:
                return Object.class;
            default:
                return Integer.class;
        }
    }

    @Override
    public Object getValueAt(int row, int column) {
        Object result = null;
        BacklightPort port = (BacklightPort) super.getValueAt(row, BacklightPortTableModel.COLUMN_PORT_INSTANCE);

        switch (column) {
            case COLUMN_LABEL:
                column = COLUMN_PORT_INSTANCE;
                break;
            case COLUMN_VALUE:
                if (port != null) {
                    // return the value for the brightness
                    result = port.getRelativeValue();
                    LOGGER.trace("Return relative brightness value: {}", result);
                    return result;
                }
                break;
            case COLUMN_DMX_MAPPING:
                return port.getDmxMapping();

            case COLUMN_PORT_INSTANCE:
                return port;
            default:
                // column = COLUMN_PORT_INSTANCE;
                break;
        }
        return super.getValueAt(row, column);
    }

    @Override
    public void setValueAt(Object value, int row, int column) {
        final BacklightPort port = (BacklightPort) super.getValueAt(row, BacklightPortTableModel.COLUMN_PORT_INSTANCE);
        LOGGER.trace("Set value, row: {}, column: {}, value: {}, port: {}", row, column, value, port);

        switch (column) {
            case COLUMN_LABEL:
                port.setLabel((String) value);
                super.setValueAt(port.toString(), row, column);
                fireLabelChanged(port, port.getLabel());
                break;
            case COLUMN_DIM_SLOPE_DOWN:
                port.setDimSlopeDown((Integer) value);
                super.setValueAt(value, row, column);
                fireConfigValuesChanged(port, PortConfigKeys.BIDIB_PCFG_DIMM_DOWN_8_8);
                break;
            case COLUMN_DIM_SLOPE_UP:
                port.setDimSlopeUp((Integer) value);
                super.setValueAt(value, row, column);
                fireConfigValuesChanged(port, PortConfigKeys.BIDIB_PCFG_DIMM_UP_8_8);
                break;
            case COLUMN_DMX_MAPPING:
                try {
                    if (value instanceof Integer) {
                        port.setDmxMapping((Integer) value);
                    }
                    else {
                        port.setDmxMapping(Integer.parseInt(value.toString()));
                    }
                    super.setValueAt(value, row, column);
                    fireConfigValuesChanged(port, PortConfigKeys.BIDIB_PCFG_OUTPUT_MAP);
                }
                catch (NumberFormatException e) {
                }
                break;
            case COLUMN_VALUE:
                try {
                    // set the value for the brightness
                    // int intValue = 0;
                    // if (value instanceof Integer) {
                    // intValue = BacklightPort.getAbsoluteValue(((Integer) value).intValue());
                    // }
                    // else {
                    // intValue = BacklightPort.getAbsoluteValue(Integer.parseInt(value.toString()));
                    // }
                    // LOGGER.info("Set the current value: {}, absolute value: {}", value, intValue);

                    // port.setValue(intValue, true);
                    // super.setValueAt(intValue, row, column);

                    // set the value on the temporary port
                    final BacklightPort testBacklightPort = new BacklightPort();
                    testBacklightPort.setId(port.getId());
                    // testBacklightPort.setValue(intValue, true);
                    // testBacklightPort.setAdjusting(false);
                    testBacklightPort.setStatus(BacklightPortStatus.START);
                    toggleTest(testBacklightPort);
                }
                catch (NumberFormatException ex) {
                    LOGGER.warn("Parse backlight value failed.", ex);
                }
                break;
        }
    }

    @Override
    public void notifyPortStatusChanged(final BacklightPort port) {
        LOGGER.debug("The port status has changed, port: {}, port.status: {}", port, port.getStatus());

        for (int row = 0; row < getRowCount(); row++) {
            if (port.equals(getValueAt(row, BacklightPortTableModel.COLUMN_PORT_INSTANCE))) {

                BacklightPort backlightPort = port;

                LOGGER.trace("The port value has changed: {}", backlightPort.getValue());
                super.setValueAt(port.getValue(), row, BacklightPortTableModel.COLUMN_VALUE);
                break;
            }
        }
    }

    @Override
    public void notifyPortConfigChanged(BacklightPort port) {
        LOGGER.info("The port config was changed for port: {}", port.getDebugString());

        for (int row = 0; row < getRowCount(); row++) {
            if (port.equals(getValueAt(row, BacklightPortTableModel.COLUMN_PORT_INSTANCE))) {
                BacklightPort backlightPort = port;
                super.setValueAt(backlightPort.getDmxMapping(), row, BacklightPortTableModel.COLUMN_DMX_MAPPING);
                super.setValueAt(backlightPort.getDimSlopeDown(), row, BacklightPortTableModel.COLUMN_DIM_SLOPE_DOWN);
                super.setValueAt(backlightPort.getDimSlopeUp(), row, BacklightPortTableModel.COLUMN_DIM_SLOPE_UP);

                break;
            }
        }
    }

    @Override
    public void notifyPortLabelChanged(final BacklightPort port) {
        LOGGER.info("The port label was changed for port: {}", port.getDebugString());

        for (int row = 0; row < getRowCount(); row++) {
            if (port.equals(getValueAt(row, BacklightPortTableModel.COLUMN_PORT_INSTANCE))) {
                super.setValueAt(port.toString(), row, BacklightPortTableModel.COLUMN_LABEL);

                break;
            }
        }
    }

    @Override
    public void refreshRow(final BacklightPort port, final IntConsumer rowConsumer) {
        int row = findRow(port);
        rowConsumer.accept(row);
    }

    private int findRow(final BacklightPort port) {
        for (int row = 0; row < getRowCount(); row++) {
            BacklightPort current = (BacklightPort) getValueAt(row, BacklightPortTableModel.COLUMN_PORT_INSTANCE);

            if (port.equals(current)) {
                return row;
            }
        }
        return -1;
    }

    /**
     * Set the new port config values of the port.
     * 
     * @param port
     *            the port with the new port config values
     */

    // TODO rename to storePortConfig ??

    public void updatePortConfig(final BacklightPort port) {

        LOGGER.debug("The config of the port has changed: {}", port.getDebugString());

        fireConfigValuesChanged(port, /*
                                       * PortConfigKeys.BIDIB_PCFG_LEVEL_PORT_ON,
                                       * PortConfigKeys.BIDIB_PCFG_LEVEL_PORT_OFF,
                                       */
            PortConfigKeys.BIDIB_PCFG_DIMM_DOWN_8_8, PortConfigKeys.BIDIB_PCFG_DIMM_UP_8_8,
            PortConfigKeys.BIDIB_PCFG_OUTPUT_MAP);

        int row = findRow(port);
        if (row > -1) {
            // set the label
            setValueAt(port.getLabel(), row, 0);
        }

    }

    private void fireConfigValuesChanged(BacklightPort port, final PortConfigKeys... portConfigKeys) {
        portListener.configChanged(port, portConfigKeys);
    }

    public void fireTestButtonPressed(BacklightPort port) {
        LOGGER.info("Slider value has changed for port: {}", port);

        portListener.testButtonPressed(port, port.getValue());
    }

    private void toggleTest(final BacklightPort port) {
        LOGGER.info("Toggle the test function for port: {}", port);

        // portListener.toggleTest(port);
    }

    @Override
    protected BacklightPort getRow(int rowIndex) {
        BacklightPort servoPort = (BacklightPort) getValueAt(rowIndex, COLUMN_PORT_INSTANCE);
        return servoPort;
    }

    public void setValue(final BacklightPort port) {
        LOGGER.info("Set the value: {}", port.getDebugString());

        fireTestButtonPressed(port);
    }

    public void updateSliderPosition(int row, int column) {
        fireTableCellUpdated(row, column);
    }

}
