package org.bidib.wizard.mvc.main.view.panel;

import java.awt.Color;
import java.awt.Component;
import java.awt.Dimension;
import java.awt.Insets;
import java.awt.Point;
import java.awt.event.MouseEvent;
import java.util.LinkedHashMap;
import java.util.Map;

import javax.swing.Box;
import javax.swing.BoxLayout;
import javax.swing.Icon;
import javax.swing.ImageIcon;
import javax.swing.JLabel;
import javax.swing.JPanel;
import javax.swing.JTable;
import javax.swing.UIManager;
import javax.swing.border.EmptyBorder;
import javax.swing.plaf.UIResource;
import javax.swing.table.TableCellRenderer;

import org.apache.commons.lang3.StringUtils;
import org.bidib.jbidibc.core.schema.bidiblabels.NodeLabels;
import org.bidib.jbidibc.messages.ProtocolVersion;
import org.bidib.jbidibc.messages.enums.BoosterControl;
import org.bidib.jbidibc.messages.enums.DetachedState;
import org.bidib.jbidibc.messages.enums.IdentifyState;
import org.bidib.jbidibc.messages.utils.NodeUtils;
import org.bidib.jbidibc.messages.utils.ProductUtils;
import org.bidib.wizard.api.locale.Resources;
import org.bidib.wizard.api.model.NodeInterface;
import org.bidib.wizard.client.common.view.BidibNodeNameUtils;
import org.bidib.wizard.common.labels.WizardLabelWrapper;
import org.bidib.wizard.common.utils.ImageUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class BidibNodeRenderer extends JPanel implements TableCellRenderer {
    private static final long serialVersionUID = 1L;

    private static final Logger LOGGER = LoggerFactory.getLogger(BidibNodeRenderer.class);

    private final WizardLabelWrapper wizardLabelWrapper;

    private JLabel mainIconLabel, additionalIconLabel, valueLabel;

    private final Icon bidibLeafIcon;

    private final Icon bidibIdentifyIcon;

    private final Icon bidibNodeIcon;

    private final Icon bidibErrorIcon;

    private final Icon bidibRestartIcon;

    private final Icon bidibLeafWarnIcon;

    private final Icon bidibNodeWarnIcon;

    private final Icon bidibDetachedIcon;

    private final Icon bidibNodeConfigPendingIcon;

    private boolean alwaysShowProductNameInTree;

    private String messageUnsupportedProtocol;

    // Colors
    /** Color to use for the foreground for selected nodes. */
    protected Color textSelectionColor;

    /** Color to use for the foreground for non-selected nodes. */
    protected Color textNonSelectionColor;

    /** Color to use for the background when a node is selected. */
    protected Color backgroundSelectionColor;

    /** Color to use for the background when the node isn't selected. */
    protected Color backgroundNonSelectionColor;

    /** Color to use for the focus indicator when the node has focus. */
    protected Color borderSelectionColor;

    private Map<BoosterControl, ImageIcon> boosterControlIcons;

    private final Icon multiBaseMasterIcon;

    private final Icon multiBaseSlaveIcon;

    private final String multiBaseMasterTooltip;

    private final String multiBaseSlaveTooltip;

    /**
     * Set to true after the constructor has run.
     */
    private boolean inited;

    private Map<BoosterControl, String> boosterControlToolTips;

    public BidibNodeRenderer(final WizardLabelWrapper wizardLabelWrapper, Icon leafIcon, Icon nodeIcon,
        Icon identifyIcon, Icon errorIcon, Icon restartPendingIcon, Icon leafWarnIcon, Icon nodeWarnIcon,
        final Icon detachedIcon, Icon nodeConfigPendingIcon, String messageUnsupportedProtocol) {
        this.wizardLabelWrapper = wizardLabelWrapper;
        this.bidibLeafIcon = leafIcon;
        this.bidibNodeIcon = nodeIcon;
        this.bidibIdentifyIcon = identifyIcon;
        this.bidibErrorIcon = errorIcon;
        this.bidibRestartIcon = restartPendingIcon;
        this.bidibLeafWarnIcon = leafWarnIcon;
        this.bidibNodeWarnIcon = nodeWarnIcon;
        this.bidibDetachedIcon = detachedIcon;
        this.bidibNodeConfigPendingIcon = nodeConfigPendingIcon;
        this.messageUnsupportedProtocol = messageUnsupportedProtocol;

        boosterControlIcons = new LinkedHashMap<>();
        boosterControlIcons
            .put(BoosterControl.CONNECT,
                ImageUtils
                    .createImageIcon(BidibNodeRenderer.class, UIManager.getString("BoosterControl.connect.icon.name"),
                        16, 16));
        boosterControlIcons
            .put(BoosterControl.LOCAL,
                ImageUtils
                    .createImageIcon(BidibNodeRenderer.class, UIManager.getString("BoosterControl.local.icon.name"), 16,
                        16));
        boosterControlIcons
            .put(BoosterControl.REMOTE,
                ImageUtils
                    .createImageIcon(BidibNodeRenderer.class, UIManager.getString("BoosterControl.remote.icon.name"),
                        16, 16));

        boosterControlToolTips = new LinkedHashMap<>();
        boosterControlToolTips
            .put(BoosterControl.CONNECT, Resources.getString(BoosterControl.class, "connect.tooltip"));
        boosterControlToolTips.put(BoosterControl.LOCAL, Resources.getString(BoosterControl.class, "local.tooltip"));
        boosterControlToolTips.put(BoosterControl.REMOTE, Resources.getString(BoosterControl.class, "remote.tooltip"));

        multiBaseMasterIcon =
            ImageUtils.createImageIcon(BidibNodeRenderer.class, "/icons/rfbase/speedgauge_16x16.png", 16, 16);
        multiBaseSlaveIcon =
            ImageUtils.createImageIcon(BidibNodeRenderer.class, "/icons/rfbase/rfbase_16x16.png", 16, 16);

        multiBaseMasterTooltip = Resources.getString(BidibNodeRenderer.class, "rfbasis.tooltip.master");
        multiBaseSlaveTooltip = Resources.getString(BidibNodeRenderer.class, "rfbasis.tooltip.slave");

        setLayout(new BoxLayout(this, BoxLayout.LINE_AXIS));
        setOpaque(false);
        mainIconLabel = new JLabel(new ImageIcon());
        additionalIconLabel = new JLabel(new ImageIcon());
        valueLabel = new JLabel();
        valueLabel.setOpaque(true);
        valueLabel.setBorder(new EmptyBorder(1, 2, 1, 2));
        add(mainIconLabel);
        add(additionalIconLabel);
        add(Box.createRigidArea(new Dimension(4, 0)));
        add(valueLabel);

        inited = true;
    }

    @Override
    public void updateUI() {
        super.updateUI();

        Insets margins = UIManager.getInsets("Tree.rendererMargins");
        if (margins != null) {
            setBorder(new EmptyBorder(margins.top, margins.left, margins.bottom, margins.right));
        }

        if (!inited || (getTextSelectionColor() instanceof UIResource)) {
            setTextSelectionColor(UIManager.getColor("Tree.selectionForeground"));
        }
        if (!inited || (getTextNonSelectionColor() instanceof UIResource)) {
            setTextNonSelectionColor(UIManager.getColor("Tree.textForeground"));
        }

        if (!inited || (getBackgroundSelectionColor() instanceof UIResource)) {
            setBackgroundSelectionColor(UIManager.getColor("Tree.selectionBackground"));
        }
        if (!inited || (getBackgroundNonSelectionColor() instanceof UIResource)) {
            setBackgroundNonSelectionColor(UIManager.getColor("Tree.textBackground"));
        }
        if (!inited || (getBorderSelectionColor() instanceof UIResource)) {
            setBorderSelectionColor(UIManager.getColor("Tree.selectionBorderColor"));
        }
    }

    @Override
    public Component getTableCellRendererComponent(
        JTable table, Object value, boolean isSelected, boolean hasFocus, int row, int column) {

        final NodeInterface node = (NodeInterface) value;

        boolean leaf = !NodeUtils.hasSubNodesFunctions(node.getUniqueId());
        String tooltipText = null;

        if (isDetachedNode(node)) {
            mainIconLabel.setIcon(bidibDetachedIcon);
            tooltipText = "Node is detached";
        }
        else if (isNodeHasError(node)) {
            if (isNodeHasRestartPendingError(node)) {
                mainIconLabel.setIcon(bidibRestartIcon);
            }
            else {
                mainIconLabel.setIcon(bidibErrorIcon);
            }
        }
        else if (isIdentifyNode(node)) {
            mainIconLabel.setIcon(bidibIdentifyIcon);
        }
        else if (isNodeConfigPending(node)) {
            mainIconLabel.setIcon(bidibNodeConfigPendingIcon);
        }
        else if (isUnsupportedProtocolVersionNode(node)) {
            if (leaf) {
                mainIconLabel.setIcon(bidibLeafWarnIcon);
            }
            else {
                mainIconLabel.setIcon(bidibNodeWarnIcon);
            }
        }
        else if (isUpdateableNode(node)) {
            if (leaf) {
                mainIconLabel.setIcon(bidibLeafIcon);
            }
            else {
                mainIconLabel.setIcon(bidibNodeIcon);
            }
        }
        else {
            if (leaf) {
                mainIconLabel.setIcon(bidibLeafIcon);
            }
            else {
                mainIconLabel.setIcon(bidibNodeIcon);
            }
        }
        prepareLabel(node);

        if (isSelected) {
            valueLabel.setBackground(getBackgroundSelectionColor());
            valueLabel.setForeground(getTextSelectionColor());
        }
        else {
            valueLabel.setBackground(getBackgroundNonSelectionColor());
            valueLabel.setForeground(getTextNonSelectionColor());
        }

        additionalIconLabel.setIcon(null);
        additionalIconLabel.setToolTipText(tooltipText);

        if (node != null) {
            if (NodeUtils.hasBoosterFunctions(node.getUniqueId())
                && NodeUtils.hasAccessoryFunctions(node.getUniqueId())) {
                // the node is a booster and provides accessories

                try {
                    BoosterControl boosterControl = node.getBoosterNode().getBoosterControl();

                    Icon boosterControlIcon = boosterControlIcons.get(boosterControl);
                    additionalIconLabel.setIcon(boosterControlIcon);

                    additionalIconLabel.setToolTipText(boosterControlToolTips.get(boosterControl));

                }
                catch (Exception ex) {
                    LOGGER.warn("Set the booster source control icon failed.", ex);
                }
            }
            else if (ProductUtils.isRFBasisNode(node.getUniqueId()) && node.getBaseNumber() > 0) {
                // multi base configuration
                if (node.getBaseNumber() == 1) {
                    additionalIconLabel.setIcon(multiBaseMasterIcon);
                    additionalIconLabel.setToolTipText(multiBaseMasterTooltip);
                }
                else {
                    additionalIconLabel.setIcon(multiBaseSlaveIcon);
                    additionalIconLabel.setToolTipText(multiBaseSlaveTooltip);
                }

            }
        }

        return this;
    }

    private boolean isNodeConfigPending(NodeInterface node) {
        if (node != null) {
            return !node.isInitialLoadFinished();
        }
        return false;
    }

    protected void prepareLabel(final NodeInterface node) {

        final NodeLabels nodeLabels = wizardLabelWrapper.loadLabels(node.getUniqueId());

        BidibNodeNameUtils.NodeLabelData labelData =
            BidibNodeNameUtils.prepareLabel(node, nodeLabels, alwaysShowProductNameInTree, true);

        valueLabel.setText(labelData.getNodeLabel());

        if (isUnsupportedProtocolVersionNode(node)) {
            setToolTipText(messageUnsupportedProtocol);
        }
        else if (isNodeHasError(node)) {
            String errorReason = BidibNodeNameUtils.getErrorReason(node);
            setToolTipText(errorReason);
        }
        else {
            setToolTipText(labelData.getNodeToolTipText());
        }
    }

    protected boolean isNodeHasError(final NodeInterface node) {
        if (node != null) {
            return node.isNodeHasError();
        }
        return false;
    }

    protected boolean isNodeHasRestartPendingError(final NodeInterface node) {
        if (node != null) {
            return node.isNodeHasRestartPendingError();
        }
        return false;
    }

    protected boolean isUpdateableNode(final NodeInterface node) {
        if (node != null) {
            return node.isUpdatable();
        }
        return false;
    }

    protected boolean isIdentifyNode(final NodeInterface node) {
        if (node != null && IdentifyState.START.equals(node.getIdentifyState())) {
            return true;
        }
        return false;
    }

    protected boolean isDetachedNode(final NodeInterface node) {
        if (node != null && node.getDetachedState() == DetachedState.DETACHED) {
            return true;
        }
        return false;
    }

    /**
     * Check if the node has an unsupported protocol version.
     * 
     * @param node
     *            the node
     * @return {@code true}: the prototcol version of the node is unsupported, {@code false} protocol version is
     *         supported
     */
    protected boolean isUnsupportedProtocolVersionNode(final NodeInterface node) {
        if (node != null && ProtocolVersion.isUnsupportedProtocolVersion(node.getNode().getProtocolVersion())) {
            return true;
        }
        return false;
    }

    /**
     * @param alwaysShowProductNameInTree
     *            the alwaysShowProductNameInTree to set
     */
    public void setAlwaysShowProductNameInTree(boolean alwaysShowProductNameInTree) {
        this.alwaysShowProductNameInTree = alwaysShowProductNameInTree;
    }

    /**
     * Sets the color the text is drawn with when the node is selected.
     */
    public void setTextSelectionColor(Color newColor) {
        textSelectionColor = newColor;
    }

    /**
     * Returns the color the text is drawn with when the node is selected.
     */
    public Color getTextSelectionColor() {
        return textSelectionColor;
    }

    /**
     * Sets the color the text is drawn with when the node isn't selected.
     */
    public void setTextNonSelectionColor(Color newColor) {
        textNonSelectionColor = newColor;
    }

    /**
     * Returns the color the text is drawn with when the node isn't selected.
     */
    public Color getTextNonSelectionColor() {
        return textNonSelectionColor;
    }

    /**
     * Sets the color to use for the background if node is selected.
     */
    public void setBackgroundSelectionColor(Color newColor) {
        backgroundSelectionColor = newColor;
    }

    /**
     * Returns the color to use for the background if node is selected.
     */
    public Color getBackgroundSelectionColor() {
        return backgroundSelectionColor;
    }

    /**
     * Sets the background color to be used for non selected nodes.
     */
    public void setBackgroundNonSelectionColor(Color newColor) {
        backgroundNonSelectionColor = newColor;
    }

    /**
     * Returns the background color to be used for non selected nodes.
     */
    public Color getBackgroundNonSelectionColor() {
        return backgroundNonSelectionColor;
    }

    /**
     * Sets the color to use for the border.
     */
    public void setBorderSelectionColor(Color newColor) {
        borderSelectionColor = newColor;
    }

    /**
     * Returns the color the border is drawn.
     */
    public Color getBorderSelectionColor() {
        return borderSelectionColor;
    }

    @Override
    public String getToolTipText(MouseEvent newEvent) {
        LOGGER.trace("Get the tooltip text, p: {}", newEvent.getPoint());

        Point p = newEvent.getPoint();
        if (p.x > 16 && p.x < 33) {
            String tip = additionalIconLabel.getToolTipText();
            if (StringUtils.isNotBlank(tip)) {
                return tip;
            }
        }

        return super.getToolTipText(newEvent);
    }
}
