package org.bidib.wizard.mvc.main.view.panel;

import java.awt.BorderLayout;
import java.awt.Point;
import java.awt.event.MouseAdapter;
import java.awt.event.MouseEvent;
import java.util.LinkedList;
import java.util.List;

import javax.swing.JPanel;
import javax.swing.JScrollPane;
import javax.swing.ListSelectionModel;
import javax.swing.border.EmptyBorder;
import javax.swing.event.ListSelectionEvent;
import javax.swing.event.ListSelectionListener;
import javax.swing.event.TableModelEvent;
import javax.swing.event.TableModelListener;
import javax.swing.table.JTableHeader;

import org.bidib.wizard.api.locale.Resources;
import org.bidib.wizard.api.model.listener.PortListListener;
import org.bidib.wizard.api.model.listener.PortListener;
import org.bidib.wizard.api.model.listener.PortListenerProvider;
import org.bidib.wizard.api.utils.PortListUtils;
import org.bidib.wizard.client.common.controller.NodeSelectionProvider;
import org.bidib.wizard.client.common.model.SimpleHierarchicalPortTableModel;
import org.bidib.wizard.client.common.model.SimplePortTableModel;
import org.bidib.wizard.client.common.model.listener.PortModelListener;
import org.bidib.wizard.client.common.table.AbstractHierarchicalEmptyTable.PackLastColumnEnum;
import org.bidib.wizard.client.common.table.AbstractPortEditorPanel;
import org.bidib.wizard.client.common.table.AbstractPortHierarchicalTable;
import org.bidib.wizard.common.script.node.types.TargetType;
import org.bidib.wizard.model.ports.Port;
import org.bidib.wizard.model.ports.event.PortConfigChangeEvent;
import org.bidib.wizard.model.status.BidibStatus;
import org.bidib.wizard.nodescript.script.node.ChangeLabelSupport;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.jidesoft.grid.CellStyleTableHeader;
import com.jidesoft.swing.DefaultOverlayable;
import com.jidesoft.swing.JideSwingUtilities;
import com.jidesoft.swing.StyledLabelBuilder;

import io.reactivex.rxjava3.subjects.PublishSubject;

public abstract class SimpleHierarchicalPortListPanel<TM extends SimpleHierarchicalPortTableModel<S, P, M>, S extends BidibStatus, P extends Port<S>, L extends PortListener<P>, M extends PortModelListener<P>>
    extends JPanel
    implements PortListListener, ChangeLabelSupport, PortListenerProvider<P> {
    private static final long serialVersionUID = 1L;

    protected final Logger LOGGER = LoggerFactory.getLogger(getClass());

    protected TM tableModel;

    protected AbstractPortHierarchicalTable<P> table;

    protected L portListener;

    public SimpleHierarchicalPortListPanel(final TM tableModel, String emptyTableText,
        final PublishSubject<PortConfigChangeEvent> portConfigChangeEventSubject, final NodeSelectionProvider nodeSelectionProvider) {
        this.tableModel = tableModel;

        setLayout(new BorderLayout());

        setBorder(new EmptyBorder(5, 5, 5, 5));

        createTable(tableModel, emptyTableText, portConfigChangeEventSubject, nodeSelectionProvider);

        final DefaultOverlayable overlayTable = new DefaultOverlayable(new JScrollPane(table));
        table.getModel().addTableModelListener(new TableModelListener() {
            @Override
            public void tableChanged(TableModelEvent e) {
                overlayTable.setOverlayVisible(table.getModel().getRowCount() == 0);
            }
        });

        overlayTable
            .addOverlayComponent(StyledLabelBuilder.createStyledLabel("{" + table.getEmptyTableText() + ":f:gray}"));

        add(overlayTable, BorderLayout.CENTER);
    }

    protected SimplePortTableModel<S, P, M> getTableModel() {
        return tableModel;
    }

    protected void createTable(
        final TM tableModel, String emptyTableText,
        final PublishSubject<PortConfigChangeEvent> portConfigChangeEventSubject, final NodeSelectionProvider nodeSelectionProvider) {

        table = createPortTable(tableModel, emptyTableText);
        table.createComponentFactory(portConfigChangeEventSubject, nodeSelectionProvider);

        table.setSortingEnabled(false);
        // sort the table automatically
        table.setAutoResort(true);

        table.setRestoreSelectionAndRowHeightAutomatically(true);

        table.setSelectionMode(ListSelectionModel.SINGLE_SELECTION);
        table.getSelectionModel().addListSelectionListener(new ListSelectionListener() {
            @Override
            public void valueChanged(ListSelectionEvent e) {
                int row = table.getSelectedRow();
                if (row != -1) {
                    table.expandRow(row);
                }
            }
        });

        table.adjustRowHeight();

        // install the renderers
        table.prepareTableColumns();

        JTableHeader header = this.table.getTableHeader();

        JideSwingUtilities.insertMouseListener(header, new MouseAdapter() {
            @Override
            public void mouseClicked(MouseEvent e) {
                if (e.getSource() instanceof CellStyleTableHeader) {
                    CellStyleTableHeader _header = (CellStyleTableHeader) e.getSource();
                    Point p = e.getPoint();
                    int index = _header.originalColumnAtPoint(p);
                    if (_header.getTable() != null && index == 0 && p.x < 20) {
                        LOGGER.info("Collapse all items.");

                        boolean isCollapsed = tableModel.isCollapsed();
                        tableModel.setCollapsed(!isCollapsed);
                        if (isCollapsed) {
                            table.expandAllRows();
                        }
                        else {
                            table.collapseAllRows();
                        }
                    }
                }
            }
        }, 0);
    }

    protected AbstractPortHierarchicalTable<P> createPortTable(final TM tableModel, String emptyTableText) {

        return new AbstractPortHierarchicalTable<P>(tableModel, emptyTableText) {
            private static final long serialVersionUID = 1L;

            @Override
            public void clearTable() {
            }

            @Override
            protected AbstractPortEditorPanel<P> createPortEditorPanel(
                P port, int row, final PublishSubject<PortConfigChangeEvent> portConfigChangeEventSubject, final NodeSelectionProvider nodeSelectionProvider) {
                return null;
            }

            @Override
            public void prepareTableColumns() {

            }
        };
    }

    public void setPortListener(final L listener) {
        portListener = listener;

        // TODO check if this is required
        // tableModel.setPortListener(listener);
    }

    @Override
    public String getName() {
        return Resources.getString(getClass(), "name");
    }

    @Override
    public void listChanged() {
        if (tableModel == null) {
            return;
        }

        tableModel.setRowCount(0);

        List<P> ports = new LinkedList<>();
        ports.addAll(getPorts());
        synchronized (ports) {
            for (P port : ports) {
                LOGGER.debug("Adding row for port: {}", port);
                tableModel.addRow(port);
            }
        }

        // pack last column
        if (PackLastColumnEnum.NONE != isPackLastColumn()) {
            packColumn();
        }

        table.expandAllRows();
        scrollToTop();

        LOGGER.debug("The port list has changed has finished in SimplePortListPanel.");
    }

    protected void packColumn() {
        if (table.getColumnCount() > 1) {
            table.packColumn(table.getColumnCount() - 1, 2, isPackLastColumn());
        }
    }

    /**
     * @return the last column must be packed after all rows are added.
     */
    protected PackLastColumnEnum isPackLastColumn() {
        return PackLastColumnEnum.FIXED;
    }

    @Override
    public void changeLabel(TargetType portType) {
        int portNum = portType.getPortNum();
        String label = portType.getLabel();

        P port = null;
        List<P> ports = getPorts();
        synchronized (ports) {
            port = PortListUtils.findPortByPortNumber(ports, portNum);
        }

        if (port != null) {
            LOGGER.info("Set the port label: {}", label);
            port.setLabel(label);

            try {
                portListener.labelChanged(port, label);
            }
            catch (Exception ex) {
                LOGGER.warn("Change port label failed.", ex);
            }
        }
    }

    /**
     * Get the list of ports.
     * 
     * @return the list of ports
     */
    protected abstract List<P> getPorts();

    @Override
    public PortListener<P> getPortListener() {
        return portListener;
    }

    private void scrollToTop() {
        table.scrollToTop();
        // if (table.getParent() instanceof JViewport) {
        // JViewport scrollPane = (JViewport) table.getParent();
        // scrollPane.setViewPosition(new Point(0, 0));
        // }
    }
}
