package org.bidib.wizard.mvc.main.view.panel.nodetree;

import java.awt.Point;
import java.awt.event.ActionEvent;
import java.beans.PropertyChangeEvent;
import java.beans.PropertyChangeListener;
import java.beans.PropertyVetoException;
import java.util.Arrays;
import java.util.LinkedHashSet;
import java.util.LinkedList;
import java.util.List;
import java.util.Locale;
import java.util.Set;

import javax.swing.AbstractAction;
import javax.swing.ImageIcon;
import javax.swing.JCheckBoxMenuItem;
import javax.swing.JMenuItem;
import javax.swing.JPopupMenu;
import javax.swing.JSeparator;
import javax.swing.ListSelectionModel;
import javax.swing.SwingUtilities;
import javax.swing.event.ListSelectionEvent;
import javax.swing.event.ListSelectionListener;
import javax.swing.table.JTableHeader;

import org.apache.commons.collections4.CollectionUtils;
import org.bidib.jbidibc.messages.utils.NodeUtils;
import org.bidib.wizard.api.locale.Resources;
import org.bidib.wizard.api.model.BoosterNodeInterface;
import org.bidib.wizard.api.model.NodeInterface;
import org.bidib.wizard.common.labels.WizardLabelWrapper;
import org.bidib.wizard.common.utils.ImageUtils;
import org.bidib.wizard.core.model.settings.WizardSettings;
import org.bidib.wizard.core.service.SettingsService;
import org.bidib.wizard.core.utils.AopUtils;
import org.bidib.wizard.mvc.main.view.component.LabeledDisplayItems;
import org.bidib.wizard.mvc.main.view.panel.BidibNodeRenderer;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.jidesoft.grid.AutoResizePopupMenuCustomizer;
import com.jidesoft.grid.CellRendererManager;
import com.jidesoft.grid.DefaultExpandableRow;
import com.jidesoft.grid.GridResource;
import com.jidesoft.grid.SortableTreeTableModel;
import com.jidesoft.grid.TableColumnChooser;
import com.jidesoft.grid.TableColumnChooserPopupMenuCustomizer;
import com.jidesoft.grid.TableHeaderPopupMenuInstaller;
import com.jidesoft.grid.TreeTable;

public class JideNodeTree extends TreeTable implements LabeledDisplayItems<NodeInterface> {
    private static final long serialVersionUID = 1L;

    private static final Logger LOGGER = LoggerFactory.getLogger(JideNodeTree.class);

    public static final String PROPERTYNAME_SELECTED_NODE = "selectedNode";

    private int prevSelectedIndex = -1;

    private final SettingsService settingsService;

    private final WizardLabelWrapper wizardLabelWrapper;

    private final PropertyChangeListener pclNode;

    private final Set<ListSelectionListener> listSelectionListeners = new LinkedHashSet<ListSelectionListener>();

    public JideNodeTree(final SettingsService settingsService, final WizardLabelWrapper wizardLabelWrapper) {

        this.settingsService = settingsService;
        this.wizardLabelWrapper = wizardLabelWrapper;

        setSelectionMode(ListSelectionModel.SINGLE_SELECTION);
        getTableHeader().setReorderingAllowed(false);

        setDoubleClickEnabled(false);

        // set the model
        setModel(new JideNodeTreeTableModel());

        // initialize the renderer
        initialize(this.settingsService, this.wizardLabelWrapper);

        final TableHeaderPopupMenuInstaller installer = new TableHeaderPopupMenuInstaller(this);
        installer.addTableHeaderPopupMenuCustomizer(new AutoResizePopupMenuCustomizer());
        final TableColumnChooserPopupMenuCustomizer tableColumnChooserPopupMenuCustomizer =
            new TableColumnChooserPopupMenuCustomizer() {

                @Override
                public void customizePopupMenu(JTableHeader header, JPopupMenu popup, int clickingColumn) {

                    super.customizePopupMenu(header, popup, clickingColumn);

                    JideNodeTree.this.customizePopupMenu(header, popup, clickingColumn);
                }
            };
        tableColumnChooserPopupMenuCustomizer.setFixedColumns(new int[] { 0 });
        installer.addTableHeaderPopupMenuCustomizer(tableColumnChooserPopupMenuCustomizer);

        setShowHorizontalLines(false);
        hideOptionalColumns();

        getSortableTableModel().sortColumn(JideNodeTreeTableModel.COLUMN_NAME, true, true);

        setRowHeight(getRowHeight() + 6);

        pclNode = new PropertyChangeListener() {

            @Override
            public void propertyChange(PropertyChangeEvent evt) {

                switch (evt.getPropertyName()) {
                    case NodeInterface.PROPERTY_INITIAL_LOAD_FINISHED:
                        LOGGER
                            .info("A property of the node has changed, repaint the tree, propertyName: {}, node: {}",
                                evt.getPropertyName(), evt.getSource());

                        // refresh the tree
                        SwingUtilities.invokeLater(() -> repaint());
                        break;
                    case NodeInterface.PROPERTY_IDENTIFY_STATE:
                    case NodeInterface.PROPERTY_DETACHED:
                    case NodeInterface.PROPERTY_ERROR_STATE:
                    case NodeInterface.PROPERTY_REASON_DATA:
                    case BoosterNodeInterface.PROPERTY_BOOSTER_CONTROL:

                        LOGGER
                            .info("A property of the node has changed, repaint the tree, propertyName: {}, node: {}",
                                evt.getPropertyName(), evt.getSource());

                        // refresh the tree
                        SwingUtilities.invokeLater(() -> repaint());
                        break;
                    case NodeInterface.PROPERTY_NODE_PREFIX + org.bidib.jbidibc.messages.Node.PROPERTY_USERNAME:
                    case NodeInterface.PROPERTY_LABEL:
                        try {
                            final NodeInterface node = (NodeInterface) evt.getSource();
                            labelChanged(node);
                        }
                        catch (Exception ex) {
                            LOGGER.warn("Handle update of node label failed.", ex);
                        }
                        break;
                    default:
                        break;
                }

            }
        };
    }

    /**
     * Remove the reset to default menu item
     * 
     * @see <a href=
     *      "http://www.jidesoft.com/forum/viewtopic.php?f=11&t=16533&p=81468&hilit=TableColumnChooserPopupMenuCustomizer#p81464">Jide
     *      Forum</a>
     */
    private void customizePopupMenu(JTableHeader header, JPopupMenu popup, int clickingColumn) {

        for (int i = 0; i < popup.getComponentCount(); i++) {
            LOGGER.info("name: {}", popup.getComponent(i).getName());
            if (TableColumnChooserPopupMenuCustomizer.CONTEXT_MENU_RESET_COLUMNS
                .equals(popup.getComponent(i).getName())) {

                JMenuItem item = (JMenuItem) popup.getComponent(i);

                // change action of reset columns
                item
                    .setAction(new AbstractAction(JideNodeTree.this
                        .getResourceString(TableColumnChooserPopupMenuCustomizer.CONTEXT_MENU_RESET_COLUMNS)) {

                        private static final long serialVersionUID = 1L;

                        @Override
                        public void actionPerformed(ActionEvent e) {
                            hideOptionalColumns();
                        }
                    });
                break;
            }
        }
        // add menu item for 'show product name in node tree'
        JSeparator separator = new JPopupMenu.Separator();
        popup.add(separator);
        boolean isAlwaysShowProductNameInTree = settingsService.getWizardSettings().isAlwaysShowProductNameInTree();
        final JCheckBoxMenuItem item =
            new JCheckBoxMenuItem(Resources.getString(JideNodeTree.class, "columnChooserPopup.showProductName"));
        item.setSelected(isAlwaysShowProductNameInTree);
        item.addActionListener(evt -> {
            settingsService.getWizardSettings().setAlwaysShowProductNameInTree(!isAlwaysShowProductNameInTree);
        });
        popup.add(item);
    }

    private String getResourceString(String key) {

        return GridResource.getResourceBundle(Locale.getDefault()).getString(key);
    }

    private void hideOptionalColumns() {
        TableColumnChooser
            .hideColumns(this,
                new int[] { JideNodeTreeTableModel.COLUMN_UNIQUE, JideNodeTreeTableModel.COLUMN_ADDRESS });
    }

    /**
     * Initialize the component
     */
    protected void initialize(final SettingsService settingsService, final WizardLabelWrapper wizardLabelWrapper) {
        // Set the icon for leaf nodes.
        ImageIcon bidibLeafIcon = ImageUtils.createImageIcon(JideNodeTree.class, "/icons/green-leaf.png", 16, 16);
        ImageIcon bidibNodeIcon = ImageUtils.createImageIcon(JideNodeTree.class, "/icons/green-node.png", 16, 16);
        ImageIcon bidibIdentifyIcon = ImageUtils.createImageIcon(JideNodeTree.class, "/icons/red-leaf.png", 16, 16);
        ImageIcon bidibErrorIcon = ImageUtils.createImageIcon(JideNodeTree.class, "/icons/error-leaf.png", 16, 16);
        ImageIcon bidibRestartIcon =
            ImageUtils.createImageIcon(JideNodeTree.class, "/icons/restart-pending.png", 16, 16);
        ImageIcon bidibLeafWarnIcon =
            ImageUtils.createImageIcon(JideNodeTree.class, "/icons/green-leaf-warn.png", 16, 16);
        ImageIcon bidibNodeWarnIcon =
            ImageUtils.createImageIcon(JideNodeTree.class, "/icons/green-node-warn.png", 16, 16);
        ImageIcon bidibNodeDetachedIcon =
            ImageUtils.createImageIcon(JideNodeTree.class, "/icons/green-node-detached.png", 16, 16);

        ImageIcon bidibNodeConfigPendingIcon =
            ImageUtils.createImageIcon(JideNodeTree.class, "/icons/orange-node.png", 16, 16);

        String messageUnsupportedProtocol = Resources.getString(JideNodeTree.class, "unsupported-protocol-version");

        if (bidibNodeIcon != null) {

            final BidibNodeRenderer renderer =
                new BidibNodeRenderer(wizardLabelWrapper, bidibLeafIcon, bidibNodeIcon, bidibIdentifyIcon,
                    bidibErrorIcon, bidibRestartIcon, bidibLeafWarnIcon, bidibNodeWarnIcon, bidibNodeDetachedIcon,
                    bidibNodeConfigPendingIcon, messageUnsupportedProtocol);

            boolean alwaysShowProductNameInTree = settingsService.getWizardSettings().isAlwaysShowProductNameInTree();
            LOGGER.info("alwaysShowProductNameInTree: {}", alwaysShowProductNameInTree);

            renderer.setAlwaysShowProductNameInTree(alwaysShowProductNameInTree);

            try {
                final WizardSettings ws = AopUtils.getTargetObject(settingsService.getWizardSettings());

                ws
                    .addPropertyChangeListener(WizardSettings.PROPERTY_ALWAYS_SHOW_PRODUCTNAME_IN_TREE,
                            evt -> {

                                Object newValue = evt.getNewValue();
                                if (newValue instanceof Boolean) {
                                    boolean alwaysShowProductNameInTree1 = (Boolean) newValue;

                                    renderer.setAlwaysShowProductNameInTree(alwaysShowProductNameInTree1);

                                    LOGGER
                                        .info("Refresh the tree because alwaysShowProductNameInTree has changed: {}",
                                                alwaysShowProductNameInTree1);

                                    // force refresh of table
                                    final SortableTreeTableModel<DefaultExpandableRow> sortableTreeModel =
                                        (SortableTreeTableModel<DefaultExpandableRow>) getModel();
                                    JideNodeTreeTableModel model =
                                        (JideNodeTreeTableModel) sortableTreeModel.getActualModel();
                                    model.fireTableDataChanged();
                                }
                            });
            }
            catch (Exception ex) {
                LOGGER.warn("Add property change listeners failed.", ex);
            }

            // register the renderer
            CellRendererManager.registerRenderer(JideNodeTreeNode.class, renderer);
        }
        else {
            LOGGER.warn("BidibNode icon missing; using default.");
        }
    }

    @Override
    public void addListSelectionListener(ListSelectionListener listener) {

        listSelectionListeners.add(listener);
    }

    @Override
    public int getIndex(Point point) {
        return rowAtPoint(point);
    }

    @Override
    public int getItemSize() {
        return getRowCount();
    }

    @Override
    public int getRowForLocation(int x, int y) {
        return rowAtPoint(new Point(x, y));
    }

    @Override
    public int getSelectedIndex() {
        return getSelectedRow();
    }

    @Override
    public NodeInterface getSelectedItem() {

        int index = getSelectedRow();
        if (index > -1) {
            final JideNodeTreeNode row = (JideNodeTreeNode) getRowAt(index);
            return row.getNode();
        }
        return null;
    }

    @Override
    public Point indexToLocation(int index) {
        return getCellRect(index, 0, false).getLocation();
    }

    @Override
    public void refreshView() {

    }

    @Override
    public NodeInterface selectElement(Point point) {
        int row = rowAtPoint(point);
        return (NodeInterface) getModel().getValueAt(row, 0);
    }

    @Override
    public void setItems(NodeInterface[] items) {
        LOGGER.info("Set the new items: {}", new Object[] { items });

        final SortableTreeTableModel<DefaultExpandableRow> sortableTreeModel =
            (SortableTreeTableModel<DefaultExpandableRow>) getModel();
        JideNodeTreeTableModel model = (JideNodeTreeTableModel) sortableTreeModel.getActualModel();

        for (DefaultExpandableRow treeNode : model.getOriginalRows()) {
            removeListenerFromNode(treeNode);
        }

        final List<NodeInterface> newNodeItems = new LinkedList<>();

        if (items != null && items.length > 0) {
            newNodeItems.addAll(Arrays.asList(items));
        }
        else {
            LOGGER.info("No tree items to set available.");
        }

        final List<JideNodeTreeNode> originalRows = new LinkedList<>();

        // add the nodes to the tree
        JideNodeTreeNode interfaceNode = null;
        // search the interface node
        for (NodeInterface itemNode : newNodeItems) {
            if (NodeUtils.convertAddress(itemNode.getNode().getAddr()) == NodeUtils.INTERFACE_ADDRESS) {
                LOGGER.debug("Adding interface node: {}", itemNode);
                // this is the interface node
                interfaceNode = new JideNodeTreeNode(itemNode);

                if (interfaceNode.getNode() instanceof NodeInterface) {
                    LOGGER.info("Add pclNode to node: {}", itemNode);
                    interfaceNode.getNode().addPropertyChangeListener(pclNode);
                } else {
                    LOGGER.warn("The current interface node is not of type Node. No PropertyChangeListener added!");
                }

                originalRows.add(interfaceNode);
                newNodeItems.remove(itemNode);
                break;
            }
        }

        model.setOriginalRows(originalRows);

        if (interfaceNode != null && CollectionUtils.isNotEmpty(newNodeItems)) {
            // add the nodes
            addNodesOfLevel(interfaceNode, newNodeItems);
        }

        model.refresh();

        expandAll();

    }

    private void addNodesOfLevel(JideNodeTreeNode parentNode, List<NodeInterface> newNodeItems) {
        LOGGER.debug("Add children to node: {}", parentNode);

        List<NodeInterface> toRemove = new LinkedList<>();

        for (NodeInterface itemNode : newNodeItems) {

            // check if the hierarchy level is correct
            if (isChildOfNode(parentNode, itemNode)) {
                LOGGER.debug("Adding new node: {}, address: {}", itemNode, itemNode.getNode().getAddr());

                JideNodeTreeNode nodeTreeNode = new JideNodeTreeNode(itemNode);

                // add the property change listener
                if (nodeTreeNode.getNode() instanceof NodeInterface) {
                    LOGGER.info("Add pclNode to node: {}", itemNode);
                    nodeTreeNode.getNode().addPropertyChangeListener(pclNode);
                }
                else {
                    LOGGER.warn("The current node is not of type Node. No PropertyChangeListener added!");
                }

                parentNode.addChild(nodeTreeNode);

                toRemove.add(itemNode);
            }
        }

        LOGGER.trace("Remove inserted nodes from original list of nodeItems: {}", toRemove);
        newNodeItems.removeAll(toRemove);

        if (newNodeItems.size() == 0) {
            LOGGER.trace("No more node items to add.");
            return;
        }

        List<JideNodeTreeNode> children = (List<JideNodeTreeNode>) parentNode.getChildren();
        if (children != null) {
            for (JideNodeTreeNode childNode : children) {
                addNodesOfLevel(childNode, newNodeItems);
            }
        }
    }

    protected boolean isChildOfNode(JideNodeTreeNode parentTreeNode, NodeInterface node) {
        // get the parent node from the tree node
        final NodeInterface parentNode = parentTreeNode.getNode();

        // compare the address parts
        byte[] parentAddr = parentNode.getNode().getAddr();
        byte[] nodeAddr = node.getNode().getAddr();

        // get the length of the parent addr
        int parentLen = 0;
        for (int index = 0; index < parentAddr.length; index++) {
            if (parentAddr[index] != 0) {
                parentLen++;
            }
        }
        LOGGER.debug("Length of parent address: {}", parentLen);

        // get the length of the node addr
        int nodeLen = 0;
        for (int index = 0; index < nodeAddr.length; index++) {
            if (nodeAddr[index] != 0) {
                nodeLen++;
            }
        }
        LOGGER.debug("Length of node address: {}", nodeLen);

        if (nodeLen != (parentLen + 1)) {
            LOGGER.debug("Node is not a child of the provided parent because length of address does not match.");
            return false;
        }

        for (int index = 0; index < (nodeLen - 1); index++) {
            LOGGER
                .debug("Comparing at index: {}, nodeAddr: {}, parentAddr: {}", index, nodeAddr[index],
                    parentAddr[index]);
            if (nodeAddr[index] != parentAddr[index]) {
                return false;
            }
        }
        return true;
    }

    private void removeListenerFromNode(DefaultExpandableRow treeNode) {
        // remove all nodes from the tree
        for (int i = treeNode.getChildrenCount(); i > 0; i--) {
            LOGGER.trace("remove child at index: {}", i - 1);
            try {
                Object child = treeNode.getChildAt(i - 1);
                if (child instanceof JideNodeTreeNode) {
                    JideNodeTreeNode childNode = (JideNodeTreeNode) child;

                    if (childNode.getNode() instanceof NodeInterface) {
                        NodeInterface node = childNode.getNode();
                        LOGGER.info("Remove pclNode from node: {}", node);
                        node.removePropertyChangeListener(pclNode);
                    }
                    else {
                        LOGGER.warn("The current child node is not of type Node. No PropertyChangeListener removed!");
                    }

                    removeListenerFromNode(childNode);
                }
                treeNode.removeChild(child);
            }
            catch (Exception ex) {
                LOGGER.warn("Remove node failed, index: " + (i - 1), ex);
            }
        }
    }

    @Override
    public void setSelectedItem(NodeInterface node) {

        LOGGER.info("Select the node in the tree: {}", node);

        int currentlySelectedIndex = prevSelectedIndex;

        int index = getSelectedIndex();

        try {
            fireVetoableChange(PROPERTYNAME_SELECTED_NODE, currentlySelectedIndex, index);

            LOGGER.info("The change of the selected node was not vetoed, perform the change.");

            for (ListSelectionListener listSelListener : listSelectionListeners) {
                listSelListener.valueChanged(new ListSelectionEvent(JideNodeTree.this, index, index, false));
            }

            prevSelectedIndex = index;
        }
        catch (PropertyVetoException ex) {
            LOGGER.warn("The selectedNode change was vetoed.", ex);
        }

    }

    @Override
    public void selectedValueChanged(int index) {

    }

    private void labelChanged(final NodeInterface node) {
        LOGGER.info("The label was changed on node: {}", node);

        final SortableTreeTableModel<DefaultExpandableRow> sortableTreeModel =
            (SortableTreeTableModel<DefaultExpandableRow>) getModel();
        final JideNodeTreeTableModel model = (JideNodeTreeTableModel) sortableTreeModel.getActualModel();

        if (SwingUtilities.isEventDispatchThread()) {
            // resort();
            model.refresh();
        }
        else {
            SwingUtilities.invokeLater(() -> {
                // resort();
                model.refresh();
            });
        }
    }
}
