package org.bidib.wizard.mvc.netdebug.controller;

import java.io.ByteArrayOutputStream;

import javax.swing.JOptionPane;
import javax.swing.SwingUtilities;

import org.apache.commons.lang3.StringUtils;
import org.bidib.jbidibc.debug.DebugMessageListener;
import org.bidib.jbidibc.debug.DebugMessageReceiver;
import org.bidib.jbidibc.messages.ConnectionListener;
import org.bidib.jbidibc.messages.exception.PortNotOpenedException;
import org.bidib.jbidibc.messages.helpers.Context;
import org.bidib.jbidibc.messages.utils.ByteUtils;
import org.bidib.jbidibc.netbidib.debug.NetBidibDebugClient;
import org.bidib.jbidibc.netbidib.debug.NetBidibDebugClientInterface;
import org.bidib.wizard.client.common.view.DockKeys;
import org.bidib.wizard.client.common.view.DockUtils;
import org.bidib.wizard.common.model.settings.DebugToolsSettingsInterface;
import org.bidib.wizard.core.service.SettingsService;
import org.bidib.wizard.mvc.netdebug.controller.listener.NetDebugControllerListener;
import org.bidib.wizard.mvc.netdebug.model.NetDebugModel;
import org.bidib.wizard.mvc.netdebug.view.NetDebugView;
import org.bidib.wizard.mvc.netdebug.view.listener.NetDebugViewListener;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;

import com.vlsolutions.swing.docking.Dockable;
import com.vlsolutions.swing.docking.DockableState;
import com.vlsolutions.swing.docking.DockingConstants;
import com.vlsolutions.swing.docking.DockingDesktop;
import com.vlsolutions.swing.docking.DockingUtilities;
import com.vlsolutions.swing.docking.RelativeDockablePosition;
import com.vlsolutions.swing.docking.TabbedDockableContainer;

public class NetDebugController implements NetDebugControllerListener {

    private static final Logger LOGGER = LoggerFactory.getLogger(NetDebugController.class);

    private final DockingDesktop desktop;

    private NetDebugModel netDebugModel;

    private NetDebugView netDebugView;

    private NetBidibDebugClientInterface debugClient;

    private DebugMessageListener messageListener;

    private DebugMessageReceiver messageReceiver;

    @Autowired
    private SettingsService settingsService;

    public NetDebugController(final DockingDesktop desktop) {
        this.desktop = desktop;
    }

    public void start() {
        // check if the debug interface view is already opened
        String searchKey = DockKeys.NET_DEBUG_VIEW;
        LOGGER.info("Search for view with key: {}", searchKey);
        Dockable view = desktop.getContext().getDockableByKey(searchKey);
        if (view != null) {
            LOGGER.info("Select the existing debug interface view instead of open a new one.");
            DockUtils.selectWindow(view);
            return;
        }

        LOGGER.info("Create new NetDebugView.");
        netDebugModel = new NetDebugModel();

        // get the host and port from settings
        final DebugToolsSettingsInterface debugToolsSettings = settingsService.getDebugToolsSettings();
        netDebugModel.setSelectedHost(debugToolsSettings.getNetBidibHost());
        netDebugModel.setSelectedPort(debugToolsSettings.getNetBidibPort());

        netDebugView = new NetDebugView(desktop, this, netDebugModel, settingsService);

        DockableState[] dockables = desktop.getDockables();
        LOGGER.info("Current dockables: {}", new Object[] { dockables });
        if (dockables.length > 1) {

            DockableState tabPanelNodeDetails = null;
            // search the node details tab panel
            for (DockableState dockable : dockables) {

                if (DockKeys.DOCKKEY_TAB_PANEL.equals(dockable.getDockable().getDockKey())) {
                    LOGGER.info("Found the tab panel dockable.");
                    tabPanelNodeDetails = dockable;

                    break;
                }
            }

            Dockable dock = desktop.getDockables()[1].getDockable();
            if (tabPanelNodeDetails != null) {
                LOGGER.info("Add the debug interface view next to the node details panel.");
                dock = tabPanelNodeDetails.getDockable();

                TabbedDockableContainer container = DockingUtilities.findTabbedDockableContainer(dock);
                int order = 0;
                if (container != null) {
                    order = container.getTabCount();
                }
                LOGGER.info("Add new debugInterfaceView at order: {}", order);

                desktop.createTab(dock, netDebugView, order, true);
            }
            else {
                desktop.split(dock, netDebugView, DockingConstants.SPLIT_RIGHT);
            }
        }
        else {
            desktop.addDockable(netDebugView, RelativeDockablePosition.RIGHT);
        }

        netDebugView.addNetDebugViewListener(new NetDebugViewListener() {

            @Override
            public void openConnection() {

                final String host = netDebugModel.getSelectedHost();
                final int port = netDebugModel.getSelectedPort();

                LOGGER.info("Open the connection, host: {}, port: {}.", host, port);

                if (debugClient == null) {

                    LOGGER.info("Create new instance of debug reader.");

                    try {
                        messageListener = new DebugMessageListener() {

                            @Override
                            public void debugMessage(final String message) {
                                LOGGER.info("debug message received size: {}", message.length());
                                netDebugView.addLog(message);
                            }
                        };
                        messageReceiver = new DebugMessageReceiver() {
                            @Override
                            public void processMessages(ByteArrayOutputStream output) {

                                // TODO convert the message to a string

                                StringBuilder message = new StringBuilder(ByteUtils.bytesToHex(output));
                                message.append("\n");
                                LOGGER.info("processMessages, received message: {}", message);

                                // clear the output
                                output.reset();

                                publishMessage(message.toString());
                            };
                        };
                        messageReceiver.addMessageListener(messageListener);

                        debugClient = new NetBidibDebugClient(messageReceiver);
                        LOGGER.info("Created debugClient: {}", debugClient);
                    }
                    catch (Exception ex) {
                        LOGGER.warn("Open debug port failed.", ex);
                    }
                }

                try {
                    String portName = host + ":" + port;
                    debugClient.open(portName, new ConnectionListener() {

                        @Override
                        public void opened(String port) {
                            LOGGER.info("Port opened: {}", port);
                            SwingUtilities.invokeLater(new Runnable() {
                                @Override
                                public void run() {
                                    netDebugModel.setConnected(true);
                                }
                            });
                        }

                        @Override
                        public void closed(String port) {
                            LOGGER.info("Port closed: {}", port);
                            SwingUtilities.invokeLater(new Runnable() {
                                @Override
                                public void run() {
                                    netDebugModel.setConnected(false);
                                }
                            });
                        }

                        @Override
                        public void status(String messageKey, final Context context) {
                            // no implementation
                        }

                        @Override
                        public void stall(boolean stall) {
                            // no implementation
                        };

                    });
                }
                catch (

                PortNotOpenedException ex) {
                    LOGGER.warn("Open debug port failed.", ex);

                    JOptionPane
                        .showMessageDialog(netDebugView.getComponent(),
                            "Open debug Port failed. Reason: " + ex.getReason(), "NetDebug Interface",
                            JOptionPane.ERROR_MESSAGE);
                }
                catch (

                Exception ex) {
                    LOGGER.warn("Open debug port failed.", ex);
                }

            }

            @Override
            public void closeConnection() {
                LOGGER.info("Close the debug connection.");

                fireCloseConnection();

            }

            @Override
            public void transmit() {
                String sendText = netDebugModel.getSendText();

                if (StringUtils.isEmpty(sendText)) {
                    LOGGER.info("No data to send!");
                    return;
                }

                LOGGER.info("Send text to debugReader: {}", sendText);

                // TODO prepare the message
                try {
                    byte[] message = ByteUtils.parseHexBinary(sendText.replaceAll(" ", ""));

                    if (debugClient != null) {
                        debugClient.send(message);
                    }
                }
                catch (Exception ex) {
                    LOGGER.warn("Parse provided message or send failed. Current message: {}", sendText, ex);
                }
            }

        });

        netDebugModel.addPropertyChangeListener(NetDebugModel.PROPERTY_SELECTED_HOST, evt -> {
            LOGGER.info("Update the host in the settings.");
            debugToolsSettings.setNetBidibHost(netDebugModel.getSelectedHost());
        });
        netDebugModel.addPropertyChangeListener(NetDebugModel.PROPERTY_SELECTED_PORT, evt -> {
            LOGGER.info("Update the port in the settings.");
            debugToolsSettings.setNetBidibPort(netDebugModel.getSelectedPort());
        });
    }

    private void fireCloseConnection() {
        if (debugClient != null) {
            LOGGER.info("Close the debug client.");
            debugClient.close();

            netDebugModel.setConnected(false);

            messageReceiver.removeMessageListener(messageListener);
            messageListener = null;
            messageReceiver = null;

            debugClient = null;
        }
        else {
            LOGGER.info("debug client not available.");
        }
    }

    @Override
    public void viewClosed() {
        LOGGER.info("The view is closed.");

        fireCloseConnection();

        // LOGGER.info("Update the host and port in the settings.");
        // final DebugToolsSettingsInterface debugToolsSettings = settingsService.getDebugToolsSettings();
        // debugToolsSettings.setNetBidibHost(netDebugModel.getSelectedHost());
        // debugToolsSettings.setNetBidibPort(netDebugModel.getSelectedPort());

    }
}
