package org.bidib.wizard.mvc.nodedebug.view;

import java.awt.BorderLayout;
import java.awt.Component;
import java.awt.Insets;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.beans.IndexedPropertyChangeEvent;
import java.beans.PropertyChangeEvent;
import java.beans.PropertyChangeListener;
import java.text.SimpleDateFormat;
import java.util.Date;
import java.util.LinkedList;
import java.util.List;

import javax.swing.JButton;
import javax.swing.JComponent;
import javax.swing.JMenuItem;
import javax.swing.JPanel;
import javax.swing.JPopupMenu;
import javax.swing.JScrollPane;
import javax.swing.JTextField;
import javax.swing.SwingConstants;
import javax.swing.text.Document;

import org.bidib.wizard.api.locale.Resources;
import org.bidib.wizard.client.common.component.ColorPane;
import org.bidib.wizard.client.common.view.BasicPopupMenu;
import org.bidib.wizard.mvc.common.view.text.ChangeDocumentListener;
import org.bidib.wizard.mvc.common.view.text.CopyAllAction;
import org.bidib.wizard.mvc.common.view.text.HistoryModel;
import org.bidib.wizard.mvc.common.view.text.HistoryTextField;
import org.bidib.wizard.mvc.common.view.text.JTextFieldLimitDocument;
import org.bidib.wizard.mvc.nodedebug.model.DebugConsoleModel;
import org.bidib.wizard.mvc.nodedebug.model.DebugConsoleModel.ConsoleLine;
import org.bidib.wizard.mvc.nodedebug.view.listener.DebugConsoleViewListener;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.jgoodies.binding.adapter.Bindings;
import com.jgoodies.binding.beans.PropertyAdapter;
import com.jgoodies.binding.beans.PropertyConnector;
import com.jgoodies.binding.value.ValueModel;
import com.jgoodies.forms.builder.FormBuilder;
import com.jgoodies.forms.debug.FormDebugPanel;
import com.jgoodies.forms.factories.Paddings;
import com.jidesoft.swing.DefaultOverlayable;
import com.jidesoft.swing.JideScrollPane;
import com.jidesoft.swing.StyledLabelBuilder;
import com.vlsolutions.swing.docking.DockKey;
import com.vlsolutions.swing.docking.Dockable;

public class DebugConsoleView implements Dockable {

    private static final Logger LOGGER = LoggerFactory.getLogger(DebugConsoleView.class);

    private static final String ENCODED_DIALOG_COLUMN_SPECS =
        "pref, 3dlu, max(100dlu;pref), 3dlu, 20dlu, 3dlu, pref, 3dlu, pref, 3dlu, pref, 3dlu, pref, 3dlu, fill:50dlu:grow, 3dlu, pref, 3dlu, pref";

    private static final String ENCODED_DIALOG_ROW_SPECS = "pref, 3dlu, fill:80dlu:grow";

    private final DockKey DOCKKEY = new DockKey("DebugConsoleView");

    private final DebugConsoleModel debugConsoleModel;

    private final JComponent contentPanel;

    private final ColorPane coloredTextPane = new ColorPane();

    private final SimpleDateFormat dateFormat = new SimpleDateFormat("HH:mm:ss.SSS");

    private ValueModel sendTextValueModel;

    private JTextField sendText;

    private final JButton transmitButton = new JButton(Resources.getString(getClass(), "transmit"));

    private final List<DebugConsoleViewListener> listeners = new LinkedList<>();

    public DebugConsoleView(final DebugConsoleModel debugConsoleModel) {

        this.debugConsoleModel = debugConsoleModel;

        DOCKKEY.setName(Resources.getString(getClass(), "title"));
        DOCKKEY.setFloatEnabled(true);
        DOCKKEY.setAutoHideEnabled(false);

        LOGGER.info("Create new DebugConsoleView");

        // create form builder
        FormBuilder dialogBuilder = null;
        boolean debugDialog = false;
        if (debugDialog) {
            JPanel panel = new FormDebugPanel();
            dialogBuilder =
                FormBuilder.create().columns(ENCODED_DIALOG_COLUMN_SPECS).rows(ENCODED_DIALOG_ROW_SPECS).panel(panel);
        }
        else {
            JPanel panel = new JPanel(new BorderLayout());
            dialogBuilder =
                FormBuilder.create().columns(ENCODED_DIALOG_COLUMN_SPECS).rows(ENCODED_DIALOG_ROW_SPECS).panel(panel);
        }
        dialogBuilder.border(Paddings.DIALOG);

        // create the textfield for send message to debug
        sendTextValueModel =
            new PropertyAdapter<DebugConsoleModel>(debugConsoleModel, DebugConsoleModel.PROPERTY_SEND_TEXT, true);

        // set default 20 items in history
        HistoryModel.setMax(20);
        sendText = new HistoryTextField("sendText", false, true);
        final Document textDoc = new JTextFieldLimitDocument(32);
        sendText.setDocument(textDoc);
        Bindings.bind(sendText, sendTextValueModel, false);

        sendText.addActionListener(new ActionListener() {

            @Override
            public void actionPerformed(ActionEvent e) {
                fireTransmit();
            }
        });

        final DefaultOverlayable sendTextOverlayable = new DefaultOverlayable(sendText);
        sendTextOverlayable
            .addOverlayComponent(
                StyledLabelBuilder
                    .createStyledLabel("{" + Resources.getString(getClass(), "transmitText.prompt") + ":f:gray}"),
                SwingConstants.WEST);
        sendTextOverlayable.setOverlayLocationInsets(new Insets(0, -5, 0, 5));

        textDoc
            .addDocumentListener(
                new ChangeDocumentListener(doc -> sendTextOverlayable.setOverlayVisible(doc.getLength() < 1)));
        sendTextOverlayable.setOverlayVisible(textDoc.getLength() < 1);

        dialogBuilder.add(Resources.getString(getClass(), "transmitText")).xy(1, 1);
        dialogBuilder.add(sendTextOverlayable).xyw(3, 1, 15);

        dialogBuilder.add(transmitButton).xy(19, 1);
        transmitButton.addActionListener(new ActionListener() {

            @Override
            public void actionPerformed(ActionEvent e) {
                fireTransmit();
            }
        });
        transmitButton.setEnabled(false);

        // add the log area
        coloredTextPane.setEditable(false);

        JScrollPane scrollPaneConsole = new JScrollPane(coloredTextPane);
        scrollPaneConsole.setAutoscrolls(true);
        dialogBuilder.add(scrollPaneConsole).xyw(1, 3, 19);

        // add bindings for enable/disable the send button
        PropertyConnector
            .connect(debugConsoleModel, DebugConsoleModel.PROPERTY_TRANSMIT_ENABLED, transmitButton, "enabled");

        // set the initial state
        transmitButton.setEnabled(debugConsoleModel.isTransmitEnabled());

        // build the panel
        JPanel contentPanelTemp = dialogBuilder.build();
        contentPanelTemp.setOpaque(false);

        JideScrollPane scrollPane = new JideScrollPane(contentPanelTemp);
        contentPanel = scrollPane;

        debugConsoleModel
            .addPropertyChangeListener(DebugConsoleModel.PROPERTY_CONSOLE_CONTENT, new PropertyChangeListener() {

                @Override
                public void propertyChange(PropertyChangeEvent evt) {
                    LOGGER.info("Property was changed: {}", evt);
                    if (evt instanceof IndexedPropertyChangeEvent) {
                        IndexedPropertyChangeEvent ipce = (IndexedPropertyChangeEvent) evt;

                        ConsoleLine newLine = (ConsoleLine) ipce.getNewValue();

                        if (newLine != null) {
                            String now = dateFormat.format(new Date());
                            coloredTextPane.append(newLine.getColor(), now + " - " + newLine.getMessage() + "\n");
                        }
                    }
                }
            });

        debugConsoleModel
            .addPropertyChangeListener(DebugConsoleModel.PROPERTY_CONSOLE_CONTENT_SIZE, new PropertyChangeListener() {
                @Override
                public void propertyChange(PropertyChangeEvent evt) {
                    LOGGER.info("Property was changed: {}", evt);

                    if (evt.getNewValue() != null) {
                        Integer value = (Integer) evt.getNewValue();
                        if (value == 0) {
                            // clear the console
                            coloredTextPane.clear();
                        }
                    }
                }
            });

        // create the popup menu
        JPopupMenu popupMenu = new BasicPopupMenu();
        JMenuItem clearConsole = new JMenuItem(Resources.getString(getClass(), "clear_console"));
        clearConsole.addActionListener(new ActionListener() {
            @Override
            public void actionPerformed(ActionEvent e) {
                fireClearConsole();
            }
        });
        popupMenu.add(clearConsole);

        JMenuItem copyAllToClipboard = new JMenuItem(Resources.getString(getClass(), "copyAllToClipboard"));
        copyAllToClipboard.addActionListener(new ActionListener() {
            @Override
            public void actionPerformed(ActionEvent e) {
                fireCopyAllToClipboard();
            }
        });
        popupMenu.add(copyAllToClipboard);
        coloredTextPane.setComponentPopupMenu(popupMenu);
    }

    @Override
    public Component getComponent() {
        return contentPanel;
    }

    @Override
    public DockKey getDockKey() {
        return DOCKKEY;
    }

    public void addDebugConsoleViewListener(DebugConsoleViewListener l) {
        listeners.add(l);
    }

    public void close() {
    }

    private void fireClearConsole() {
        LOGGER.info("clear the console.");

        debugConsoleModel.clear();
        coloredTextPane.clear();
    }

    private void fireCopyAllToClipboard() {
        LOGGER.info("Copy all content to clipboard.");

        ActionEvent copyAll = new ActionEvent(coloredTextPane, 0, CopyAllAction.copyAllAction);
        CopyAllAction action = new CopyAllAction();
        action.actionPerformed(copyAll);
    }

    private void fireTransmit() {

        for (DebugConsoleViewListener listener : listeners) {
            listener.transmit();
        }
        // clear the text field
        sendText.setText(null);
    }
}
