package org.bidib.wizard.mvc.position.view;

import java.awt.Component;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.awt.event.MouseEvent;
import java.lang.reflect.InvocationTargetException;
import java.util.ArrayList;
import java.util.Collections;
import java.util.List;
import java.util.concurrent.Executors;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.TimeUnit;

import javax.swing.JMenuItem;
import javax.swing.JTable;
import javax.swing.SwingUtilities;

import org.apache.commons.collections4.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.bidib.jbidibc.messages.utils.ThreadFactoryBuilder;
import org.bidib.wizard.api.locale.Resources;
import org.bidib.wizard.api.model.PositionAddressData;
import org.bidib.wizard.api.model.PositionFeedbackPort;
import org.bidib.wizard.api.model.listener.PortListener;
import org.bidib.wizard.client.common.dialog.LabelDialog;
import org.bidib.wizard.model.ports.FeedbackPort;
import org.bidib.wizard.model.ports.Port;
import org.bidib.wizard.model.status.FeedbackPortStatus;
import org.bidib.wizard.client.common.model.SimplePortTableModel;
import org.bidib.wizard.client.common.model.listener.PortModelListener;
import org.bidib.wizard.client.common.view.menu.PortListMenu;
import org.bidib.wizard.client.common.view.menu.listener.PortListMenuListener;
import org.bidib.wizard.mvc.main.view.panel.SimplePortListPanel;
import org.bidib.wizard.client.common.table.DefaultPortListMenuListener;
import org.bidib.wizard.mvc.main.view.table.FlagEditor;
import org.bidib.wizard.client.common.table.PortTable;
import org.bidib.wizard.mvc.position.controller.FeedbackPositionController;
import org.bidib.wizard.mvc.position.model.FeedbackPositionModel;
import org.bidib.wizard.mvc.position.model.FeedbackPositionTableModel;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class FeedbackPositionListPanel
    extends
    SimplePortListPanel<FeedbackPortStatus, PositionFeedbackPort, PortListener<PositionFeedbackPort>, PortModelListener<PositionFeedbackPort>> {

    private static final long serialVersionUID = 1L;

    private static final Logger LOGGER = LoggerFactory.getLogger(FeedbackPositionListPanel.class);

    private final FeedbackPositionController controller;

    private final FeedbackPositionModel feedbackPositionModel;

    private final FeedbackPositionTableModel feedbackPositionTableModel;

    protected final ScheduledExecutorService outdatedPositionWorker =
        Executors
            .newScheduledThreadPool(1,
                new ThreadFactoryBuilder().setNameFormat("outdatedPositionsWorkers-thread-%d").build());

    private final int outdatedTimeout = 2000;

    public FeedbackPositionListPanel(final FeedbackPositionController controller,
        final FeedbackPositionModel feedbackPositionModel) {
        super(new FeedbackPositionTableModel(), Resources.getString(FeedbackPositionListPanel.class, "empty_table"));

        this.controller = controller;
        this.feedbackPositionModel = feedbackPositionModel;

        LOGGER.debug("Create new FeedbackPositionListPanel.");

        FeedbackPositionTableCellRenderer renderer = new FeedbackPositionTableCellRenderer();
        renderer.setTimeout(outdatedTimeout);
        table.setDefaultCellRenderer(renderer);
        table.setDefaultEditor(PositionFeedbackPort.class, new FlagEditor() {
            private static final long serialVersionUID = 1L;

            @Override
            public Component getTableCellEditorComponent(
                JTable table, Object value, boolean isSelected, int row, int column) {
                if (value instanceof PositionFeedbackPort) {
                    textField.setText(value != null ? ((PositionFeedbackPort) value).getLabel() : null);
                    textField.setToolTipText(null);
                }
                else {
                    textField.setText(value != null ? value.toString() : null);
                    textField.setToolTipText(null);
                }
                return textField;
            }
        });

        table.setRowHeight(60);
        table.setTableHeader(null);

        feedbackPositionTableModel = (FeedbackPositionTableModel) tableModel;

        this.feedbackPositionModel.addPortListListener(this);

        // get notified of changes of label
        // addPortListener(feedbackPositionListener);

        int interval = 2000;
        outdatedPositionWorker.scheduleAtFixedRate(() -> {
            LOGGER.debug("Trigger check outdated positions.");
            try {
                triggerValidatePositions();
            }
            catch (Exception ex) {
                LOGGER.warn("Trigger the position failed.", ex);
            }
            LOGGER.debug("Trigger position has finished.");
        }, 500, interval, TimeUnit.MILLISECONDS);
    }

    @Override
    protected PortTable createPortTable(
        final SimplePortTableModel<FeedbackPortStatus, PositionFeedbackPort, PortModelListener<PositionFeedbackPort>> tableModel,
        String emptyTableText) {
        return new PortTable(tableModel, emptyTableText) {
            private static final long serialVersionUID = 1L;

            @Override
            public void adjustRowHeight() {
                // we don't need to adjust the row height
            }

            @Override
            protected void prepareTableStyleProvider() {
                // we don't need the table style provider
            }

            @Override
            public void clearTable() {
            }

            @Override
            protected PortListMenu createMenu() {
                return new FeedbackPositionListMenu(popupEvent, "No port selected.");
            }

            @Override
            protected PortListMenuListener createMenuListener() {

                // for feedback ports the port mapping is not visible
                setPortMappingVisible(false);

                // create the port list menu
                LOGGER.info("Create the menu listener.");
                return new FeedbackPositionListMenuListener() {
                    @Override
                    public void editLabel(final MouseEvent popupEvent) {
                        final int row = getRow(popupEvent.getPoint());
                        final int column = getColumn(popupEvent.getPoint());
                        LOGGER.info("Edit label on row: {}, column: {}", row, column);
                        if (row > -1) {
                            Object val = getValueAt(row, column);
                            if (val instanceof Port<?>) {
                                // val = ((Port<?>) val).toString();
                                val = ((Port<?>) val).getLabel();
                            }

                            // show the port name editor
                            new LabelDialog((String) val, popupEvent.getXOnScreen(), popupEvent.getYOnScreen()) {
                                @Override
                                public void labelChanged(String label) {
                                    LOGGER
                                        .info("Set the new label for row: {}, column: {}, label: {}", row, column,
                                            label);
                                    setValueAt(label, row, column);
                                }
                            };
                        }
                        else {
                            LOGGER.warn("The row is not available!");
                        }
                    }

                    @Override
                    public void clearAddressesAndPortStatus() {
                        LOGGER.info("Clear all addresses and set the status to free.");

                        feedbackPositionModel.clearAddressesAndPortStatus();

                    }

                    @Override
                    public void openLocoDialog() {
                        LOGGER.info("Open the loco controller.");

                        final int row = getRow(popupEvent.getPoint());
                        final int column = getColumn(popupEvent.getPoint());
                        if (row > -1) {
                            Object val = getValueAt(row, column);
                            if (val instanceof PositionFeedbackPort) {

                                PositionFeedbackPort feedbackPort = (PositionFeedbackPort) val;
                                if (CollectionUtils.isNotEmpty(feedbackPort.getAddresses())) {
                                    PositionAddressData addressData =
                                        feedbackPort.getAddresses().stream().findFirst().get();
                                    // fireOpenLocoDialog(addressData);
                                }
                                else {
                                    LOGGER.info("No address available.");
                                    // fireOpenLocoDialog(null);
                                }
                            }
                        }
                        else {
                            LOGGER.warn("The row is not available!");
                        }
                    }

                    @Override
                    public void openPomDialog() {
                        LOGGER.info("Open the POM controller.");

                        final int row = getRow(popupEvent.getPoint());
                        final int column = getColumn(popupEvent.getPoint());
                        if (row > -1) {
                            Object val = getValueAt(row, column);
                            if (val instanceof PositionFeedbackPort) {

                                PositionFeedbackPort feedbackPort = (PositionFeedbackPort) val;
                                if (CollectionUtils.isNotEmpty(feedbackPort.getAddresses())) {
                                    PositionAddressData addressData =
                                        feedbackPort.getAddresses().stream().findFirst().get();
                                    // fireOpenPomDialog(addressData);
                                }
                                else {
                                    LOGGER.info("No address available.");
                                    // fireOpenPomDialog(null);
                                }
                            }
                        }
                        else {
                            LOGGER.warn("The row is not available!");
                        }
                    }
                };
            }

            @Override
            protected void showPortListMenu(MouseEvent e, PortListMenu portListMenu, int row, int column) {

                Object value = getValueAt(row, column);
                if (row >= 0 && column >= 0 && (value instanceof Port<?> || value instanceof String)) {
                    if (row >= 0 && getSelectedRowCount() == 0) {
                        setRowSelectionInterval(row, row);
                    }

                    if (value instanceof FeedbackPort) {
                        final FeedbackPort feedbackPort = (FeedbackPort) value;
                        // if the port has mapping support enabled the activate the menu
                        if (!feedbackPort.isEnabled()) {
                            portListMenu.setMapPortEnabled(feedbackPort.isRemappingEnabled(), isPortMappingVisible());
                        }
                        else {
                            portListMenu.setMapPortEnabled(false, isPortMappingVisible());
                        }

                        // prepare the port label
                        String label = null;

                        if (StringUtils.isNotBlank(feedbackPort.getLabel())) {
                            label = String.format("%1$02d : %2$s", feedbackPort.getId(), feedbackPort.getLabel());
                        }
                        else if (feedbackPort.getId() > -1) {
                            label = String.format("%1$02d", feedbackPort.getId());
                        }
                        else {
                            label = " ";
                        }

                        ((FeedbackPositionListMenu) portListMenu).setLabel("Port: " + label);

                    }
                    else {
                        portListMenu.setMapPortEnabled(false, false);
                        ((FeedbackPositionListMenu) portListMenu).setLabel("No port selected.");
                    }
                    portListMenu.setInsertPortsEnabled(false, false);

                    portListMenu.setPopupEvent(popupEvent);

                    grabFocus();
                    portListMenu.show(e.getComponent(), e.getX(), e.getY());
                }
            }

        };
    }

    @Override
    public void listChanged() {
        LOGGER.debug("List has changed, remove all rows and add rows again.");
        tableModel.setRowCount(0);

        List<PositionFeedbackPort> ports = getPorts();

        ((FeedbackPositionTableModel) tableModel).addRows(ports);
    }

    @Override
    protected List<PositionFeedbackPort> getPorts() {
        if (feedbackPositionModel != null) {
            return feedbackPositionModel.getPortList();
        }
        return Collections.emptyList();
    }

    @Override
    public Class<?> getPortClass() {
        return PositionFeedbackPort.class;
    }

    private void triggerValidatePositions() {
        if (feedbackPositionModel != null) {

            final List<PositionFeedbackPort> outdatedPorts = new ArrayList<>();
            feedbackPositionModel.validatePositions(outdatedPorts, outdatedTimeout);

            if (CollectionUtils.isNotEmpty(outdatedPorts)) {
                LOGGER.debug("Update outdated ports.");
                try {
                    SwingUtilities.invokeAndWait(() -> feedbackPositionTableModel.validatePositions(outdatedPorts));
                }
                catch (InvocationTargetException | InterruptedException ex) {
                    LOGGER.warn("Validate outdated positions failed.", ex);
                }
            }
        }
    }

    private class FeedbackPositionListMenu extends PortListMenu {

        private static final long serialVersionUID = 1L;

        private JMenuItem clearAddressesAndPortStatus;

        private JMenuItem openCarDialog;

        private JMenuItem openPomDialog;

        public FeedbackPositionListMenu(final MouseEvent popupEvent, String label) {
            super(popupEvent, label);

            clearAddressesAndPortStatus =
                new JMenuItem(
                    Resources.getString(FeedbackPositionListMenu.class, "clearAddressesAndPortStatus") + " ...");

            clearAddressesAndPortStatus.addActionListener(new ActionListener() {
                @Override
                public void actionPerformed(ActionEvent e) {
                    fireClearAddressesAndPortStatus();
                }

            });
            add(clearAddressesAndPortStatus);

            openCarDialog =
                new JMenuItem(Resources.getString(FeedbackPositionListMenu.class, "openCarDialog") + " ...");

            openCarDialog.setEnabled(false);

            openCarDialog.addActionListener(new ActionListener() {
                @Override
                public void actionPerformed(ActionEvent e) {
                    fireOpenLocoDialog();
                }

            });
            add(openCarDialog);

            openPomDialog =
                new JMenuItem(Resources.getString(FeedbackPositionListMenu.class, "openPomDialog") + " ...");

            openPomDialog.setEnabled(false);

            openPomDialog.addActionListener(new ActionListener() {
                @Override
                public void actionPerformed(ActionEvent e) {
                    fireOpenPomDialog();
                }

            });
            add(openPomDialog);
        }

        private void fireClearAddressesAndPortStatus() {
            for (PortListMenuListener l : menuListeners) {
                if (l instanceof FeedbackPositionListMenuListener) {
                    ((FeedbackPositionListMenuListener) l).clearAddressesAndPortStatus();
                }
            }
        }

        private void fireOpenLocoDialog() {
            for (PortListMenuListener l : menuListeners) {
                if (l instanceof FeedbackPositionListMenuListener) {
                    ((FeedbackPositionListMenuListener) l).openLocoDialog();
                }
            }
        }

        private void fireOpenPomDialog() {
            for (PortListMenuListener l : menuListeners) {
                if (l instanceof FeedbackPositionListMenuListener) {
                    ((FeedbackPositionListMenuListener) l).openPomDialog();
                }
            }
        }
    }

    private abstract class FeedbackPositionListMenuListener extends DefaultPortListMenuListener {

        public abstract void clearAddressesAndPortStatus();

        public abstract void openLocoDialog();

        public abstract void openPomDialog();
    }

    public void cleanup() {
        outdatedPositionWorker.shutdown();
    }

}
