package org.bidib.wizard.mvc.loco.model;

import java.util.BitSet;
import java.util.Collection;
import java.util.LinkedList;
import java.util.concurrent.atomic.AtomicInteger;

import javax.swing.SwingUtilities;

import org.bidib.wizard.model.status.DirectionStatus;
import org.bidib.wizard.model.status.SpeedSteps;
import org.bidib.wizard.mvc.loco.model.listener.LocoModelListener;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.jgoodies.common.bean.Bean;

public class LocoModel extends Bean {
    private static final long serialVersionUID = 1L;

    private static final Logger LOGGER = LoggerFactory.getLogger(LocoModel.class);

    private final Collection<LocoModelListener> listeners = new LinkedList<LocoModelListener>();

    public static final String PROPERTYNAME_SPEED = "speed";

    public static final String PROPERTYNAME_ADDRESS = "address";

    public static final String PROPERTYNAME_ACTIVE_BASE = "activeBase";

    public static final String PROPERTYNAME_REPORTED_CELLNUMBER = "reportedCellNumber";

    public static final String PROPERTYNAME_REPORTED_SPEED = "reportedSpeed";

    public static final String PROPERTYNAME_CARCONTROLENABLED = "carControlEnabled";

    public static final String PROPERTYNAME_DEV_BINSTATE_NUMBER = "devBinStateNumber";

    public static final String PROPERTYNAME_DEV_BINSTATE_VALUE = "devBinStateValue";

    public static final String PROPERTYNAME_COUNTER_CS_DRIVE = "counterCsDrive";

    public static final String PROPERTYNAME_COUNTER_CS_DRIVE_ACK = "counterCsDriveAck";

    public static final String PROPERTYNAME_COUNTER_CS_BIN_STATE = "counterCsBinState";

    private Integer address;

    private Integer reportedSpeed;

    private int dynStateEnergy;

    // default direction is forward
    private DirectionStatus direction = DirectionStatus.FORWARD;

    private Integer speed;

    private SpeedSteps speedSteps = SpeedSteps.DCC128;

    private BitSet functions = new BitSet(29);

    private boolean carControlEnabled;

    private RfBasisMode activeBase = RfBasisMode.SINGLE;

    private RfBasisMode prevActiveBase;

    private Integer reportedCellNumber;

    private Integer devBinStateNumber;

    private BinStateValue devBinStateValue = BinStateValue.ON;

    private AtomicInteger counterCsDrive = new AtomicInteger();

    private AtomicInteger counterCsDriveAck = new AtomicInteger();

    private AtomicInteger counterCsBinState = new AtomicInteger();

    public void addLocoModelListener(LocoModelListener l) {
        listeners.add(l);
    }

    public void removeLocoModelListener(LocoModelListener l) {
        listeners.remove(l);
    }

    public Integer getAddress() {
        return address;
    }

    public void setAddress(Integer address) {
        Integer oldValue = this.address;

        this.address = address;

        firePropertyChange(PROPERTYNAME_ADDRESS, oldValue, address);
    }

    public Integer getReportedSpeed() {
        return reportedSpeed;
    }

    public void setReportedSpeed(Integer reportedSpeed) {
        LOGGER.info("Update the reported speed: {}", reportedSpeed);
        Integer oldValue = this.reportedSpeed;

        this.reportedSpeed = reportedSpeed;

        firePropertyChange(PROPERTYNAME_REPORTED_SPEED, oldValue, reportedSpeed);
    }

    public DirectionStatus getDirection() {
        return direction;
    }

    public void setDirection(DirectionStatus direction) {
        if (this.direction != direction) {
            this.direction = direction;
            fireDirectionChanged(direction);
        }
    }

    public boolean getFunction(int index) {
        return functions.get(index);
    }

    public BitSet getFunctions() {
        return (BitSet) functions.clone();
    }

    public void setFunction(int index, boolean value) {
        if (getFunction(index) != value) {
            functions.set(index, value);
            fireFunctionChanged(index, value);
        }
    }

    public int getDynStateEnergy() {
        return dynStateEnergy;
    }

    public void setDynStateEnergy(int dynStateEnergy) {
        if (this.dynStateEnergy != dynStateEnergy || dynStateEnergy == 0 || dynStateEnergy == 1) {
            this.dynStateEnergy = dynStateEnergy;

            fireDynStateEnergyChanged(dynStateEnergy);
        }
    }

    public SpeedSteps getSpeedSteps() {
        return speedSteps;
    }

    public void setSpeedSteps(SpeedSteps speedSteps) {
        if (this.speedSteps != speedSteps) {
            this.speedSteps = speedSteps;
            fireSpeedStepsChanged(speedSteps);
        }
    }

    public Integer getSpeed() {
        return speed;
    }

    public void setSpeed(Integer speed) {
        Integer oldValue = this.speed;

        LOGGER.info("Set the new speed: {}, oldValue: {}", speed, oldValue);

        this.speed = speed;
        // force signal 0 or 1 to listeners every time
        if (speed != null && (speed.intValue() == 0 || speed.intValue() == 1)) {
            oldValue = null;
        }

        firePropertyChange(PROPERTYNAME_SPEED, oldValue, speed);
    }

    private void fireDirectionChanged(DirectionStatus direction) {
        for (LocoModelListener l : listeners) {
            l.directionChanged(direction);
        }
    }

    private void fireFunctionChanged(int index, boolean value) {
        for (LocoModelListener l : listeners) {
            l.functionChanged(index, value);
        }
    }

    private void fireSpeedStepsChanged(SpeedSteps speedSteps) {
        for (LocoModelListener l : listeners) {
            l.speedStepsChanged(speedSteps);
        }
    }

    private void fireDynStateEnergyChanged(int dynStateEnergy) {
        for (LocoModelListener l : listeners) {
            l.dynStateEnergyChanged(dynStateEnergy);
        }
    }

    private void fireBinaryStateChanged(int state, boolean value) {
        for (LocoModelListener l : listeners) {
            l.binaryStateChanged(state, value);
        }
    }

    public void setBinaryState(int state, boolean value) {
        fireBinaryStateChanged(state, value);
    }

    /**
     * @return the carControlEnabled
     */
    public boolean isCarControlEnabled() {
        return carControlEnabled;
    }

    /**
     * @param carControlEnabled
     *            the carControlEnabled to set
     */
    public void setCarControlEnabled(boolean carControlEnabled) {
        boolean oldValue = this.carControlEnabled;

        this.carControlEnabled = carControlEnabled;

        firePropertyChange(PROPERTYNAME_CARCONTROLENABLED, oldValue, carControlEnabled);
    }

    /**
     * @return the activeBase
     */
    public RfBasisMode getActiveBase() {
        return activeBase;
    }

    /**
     * @param activeBase
     *            the activeBase to set
     */
    public void setActiveBase(RfBasisMode activeBase) {
        LOGGER.info("Set the new active rf base: {}", activeBase);
        RfBasisMode oldValue = this.activeBase;

        // keep the previous active base
        if (this.activeBase != activeBase) {
            LOGGER.info("The new active rf base has changed. Update the prev active base: {}", this.activeBase);
            this.prevActiveBase = this.activeBase;
        }

        this.activeBase = activeBase;

        firePropertyChange(PROPERTYNAME_ACTIVE_BASE, oldValue, activeBase);
    }

    /**
     * @return the previous activeBase
     */
    public RfBasisMode getPrevActiveBase() {
        return prevActiveBase;
    }

    /**
     * Reset the prev active base.
     */
    public void resetPrevActiveBase() {
        prevActiveBase = null;
    }

    /**
     * @return the reportedCellNumber
     */
    public Integer getReportedCellNumber() {
        return reportedCellNumber;
    }

    /**
     * @param reportedCellNumber
     *            the reportedCellNumber to set
     */
    public void setReportedCellNumber(Integer reportedCellNumber) {
        Integer oldValue = this.reportedCellNumber;
        this.reportedCellNumber = reportedCellNumber;
        firePropertyChange(PROPERTYNAME_REPORTED_CELLNUMBER, oldValue, this.reportedCellNumber);
    }

    /**
     * @return the devBinStateNumber
     */
    public Integer getDevBinStateNumber() {
        return devBinStateNumber;
    }

    /**
     * @param devBinStateNumber
     *            the devBinStateNumber to set
     */
    public void setDevBinStateNumber(Integer devBinStateNumber) {
        Integer oldValue = this.devBinStateNumber;
        this.devBinStateNumber = devBinStateNumber;

        firePropertyChange(PROPERTYNAME_DEV_BINSTATE_NUMBER, oldValue, devBinStateNumber);
    }

    /**
     * @return the devBinStateValue
     */
    public BinStateValue getDevBinStateValue() {
        return devBinStateValue;
    }

    /**
     * @param devBinStateValue
     *            the devBinStateValue to set
     */
    public void setDevBinStateValue(BinStateValue devBinStateValue) {
        BinStateValue oldValue = this.devBinStateValue;
        this.devBinStateValue = devBinStateValue;

        firePropertyChange(PROPERTYNAME_DEV_BINSTATE_VALUE, oldValue, devBinStateValue);
    }

    /**
     * @return the counterCsDrive
     */
    public int getCounterCsDrive() {
        return counterCsDrive.get();
    }

    /**
     * Increment the CS drive counter by 1.
     */
    public int incCounterCsDrive() {
        int oldValue = this.counterCsDrive.get();
        int newValue = this.counterCsDrive.addAndGet(1);

        firePropertyChange(PROPERTYNAME_COUNTER_CS_DRIVE, oldValue, newValue);

        return newValue;
    }

    /**
     * Reset the CS drive counter.
     */
    public void resetCounterCsDrive() {
        int oldValue = this.counterCsDrive.get();
        this.counterCsDrive.set(0);

        firePropertyChange(PROPERTYNAME_COUNTER_CS_DRIVE, oldValue, 0);
    }

    /**
     * @return the counterCsDriveAck
     */
    public int getCounterCsDriveAck() {
        return counterCsDriveAck.get();
    }

    /**
     * Increment the CS drive ack counter by 1.
     */
    public int incCounterCsDriveAck() {
        final int oldValue = this.counterCsDriveAck.get();
        final int newValue = this.counterCsDriveAck.addAndGet(1);

        SwingUtilities.invokeLater(() -> firePropertyChange(PROPERTYNAME_COUNTER_CS_DRIVE_ACK, oldValue, newValue));

        return newValue;
    }

    /**
     * Reset the CS drive ack counter.
     */
    public void resetCounterCsAckDrive() {
        int oldValue = this.counterCsDriveAck.get();
        this.counterCsDriveAck.set(0);

        firePropertyChange(PROPERTYNAME_COUNTER_CS_DRIVE_ACK, oldValue, 0);
    }

    /**
     * @return the counterCsBinState
     */
    public int getCounterCsBinState() {
        return counterCsBinState.get();
    }

    /**
     * Increment the CS binState counter by 1.
     */
    public int incCounterCsBinState() {
        int oldValue = this.counterCsBinState.get();
        int newValue = this.counterCsBinState.addAndGet(1);

        firePropertyChange(PROPERTYNAME_COUNTER_CS_BIN_STATE, oldValue, newValue);

        return newValue;
    }

    /**
     * Reset the CS binState counter.
     */
    public void resetCounterCsBinState() {
        int oldValue = this.counterCsBinState.get();
        this.counterCsBinState.set(0);

        firePropertyChange(PROPERTYNAME_COUNTER_CS_BIN_STATE, oldValue, 0);
    }

}
