package org.bidib.wizard.mvc.main.controller;

import java.awt.Point;
import java.util.Collection;

import javax.swing.JFrame;
import javax.swing.JOptionPane;
import javax.swing.SwingUtilities;

import org.bidib.jbidibc.core.schema.bidiblabels.NodeLabels;
import org.bidib.jbidibc.messages.AddressData;
import org.bidib.jbidibc.messages.FeedbackAddressData;
import org.bidib.jbidibc.messages.FeedbackConfidenceData;
import org.bidib.jbidibc.messages.FeedbackDynStateData;
import org.bidib.jbidibc.messages.PomAddressData;
import org.bidib.jbidibc.messages.enums.AddressTypeEnum;
import org.bidib.jbidibc.messages.enums.CommandStationPom;
import org.bidib.jbidibc.messages.enums.EnrailmentDirectionEnum;
import org.bidib.jbidibc.messages.enums.PomAcknowledge;
import org.bidib.jbidibc.messages.enums.PortConfigKeys;
import org.bidib.wizard.api.context.ApplicationContext;
import org.bidib.wizard.api.locale.Resources;
import org.bidib.wizard.api.model.CommandStationNodeInterface;
import org.bidib.wizard.api.model.NodeInterface;
import org.bidib.wizard.api.model.listener.DefaultFeedbackPortListener;
import org.bidib.wizard.api.model.listener.DefaultNodeListListener;
import org.bidib.wizard.api.model.listener.NodeListListener;
import org.bidib.wizard.api.service.node.CommandStationService;
import org.bidib.wizard.api.utils.NodeUtils;
import org.bidib.wizard.common.context.DefaultApplicationContext;
import org.bidib.wizard.common.labels.LabelsChangedEvent;
import org.bidib.wizard.common.labels.WizardLabelFactory;
import org.bidib.wizard.common.labels.WizardLabelWrapper;
import org.bidib.wizard.config.LocoControllerFactory;
import org.bidib.wizard.config.PomProgrammerControllerFactory;
import org.bidib.wizard.core.labels.BidibLabelUtils;
import org.bidib.wizard.core.model.connection.ConnectionRegistry;
import org.bidib.wizard.model.ports.FeedbackPort;
import org.bidib.wizard.mvc.loco.controller.LocoController;
import org.bidib.wizard.mvc.main.controller.listener.FeedbackPortPanelListener;
import org.bidib.wizard.mvc.main.model.FeedbackPortModel;
import org.bidib.wizard.mvc.main.model.FeedbackPortTableModel;
import org.bidib.wizard.mvc.main.model.MainModel;
import org.bidib.wizard.client.common.model.listener.PortModelListener;
import org.bidib.wizard.mvc.main.view.exchange.NodeExchangeHelper;
import org.bidib.wizard.mvc.main.view.panel.FeedbackPortListPanel;
import org.bidib.wizard.mvc.main.view.panel.listener.TabVisibilityListener;
import org.bidib.wizard.mvc.pom.controller.PomProgrammerController;
import org.bidib.wizard.mvc.pom.controller.listener.PomProgrammerControllerListener;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.context.event.EventListener;

import com.vlsolutions.swing.docking.DockingDesktop;

public class FeedbackPortPanelController implements FeedbackPortPanelListener {

    private static final Logger LOGGER = LoggerFactory.getLogger(FeedbackPortPanelController.class);

    private final MainModel mainModel;

    private final NodeListListener nodeListListener;

    private FeedbackPortListPanel feedbackPortListPanel;

    private FeedbackPortModel feedbackPortModel;

    @Autowired
    private WizardLabelWrapper wizardLabelWrapper;

    @Autowired
    private CommandStationService commandStationService;

    public FeedbackPortPanelController(final MainModel mainModel, final FeedbackPortModel feedbackPortModel) {
        this.mainModel = mainModel;

        this.feedbackPortModel = feedbackPortModel;

        nodeListListener = new DefaultNodeListListener() {
            @Override
            public void nodeChanged(final NodeInterface node) {
                LOGGER.trace("The node has changed.");

                final NodeInterface selectedNode = mainModel.getSelectedNode();
                // update the selected node
                feedbackPortModel.setSelectedNode(selectedNode);

                SwingUtilities.invokeLater(() -> {
                    LOGGER.trace("Notify that the selected port was changed.");
                    if (feedbackPortListPanel != null) {
                        feedbackPortListPanel.listChanged();
                    }
                    else {
                        LOGGER.warn("The feedbackPortListPanel is not available.");
                    }
                });
            }
        };
        this.mainModel.addNodeListListener(nodeListListener);
    }

    @Override
    public NodeInterface getSelectedNode() {
        return feedbackPortModel.getSelectedNode();
    }

    public FeedbackPortListPanel createPanel(final TabVisibilityListener tabVisibilityListener) {
        LOGGER.info("Create new feedbackPortListPanel.");

        FeedbackPortTableModel tableModel = new FeedbackPortTableModel();

        feedbackPortListPanel =
            new FeedbackPortListPanel(this, tableModel, feedbackPortModel, mainModel, tabVisibilityListener);

        // create the listener for changes on feedback ports
        final DefaultFeedbackPortListener feedbackPortListener = new DefaultFeedbackPortListener() {
            @Override
            public Class<?> getPortClass() {
                return FeedbackPort.class;
            }

            @Override
            public void addressesChanged(FeedbackPort port, Collection<FeedbackAddressData> addresses) {
                SwingUtilities.invokeLater(() -> tableModel.updatePort(port));
            }

            @Override
            public void confidenceChanged(FeedbackPort port, FeedbackConfidenceData confidence) {
                LOGGER.info("Confidence has changed for port: {}, confidence: {}", port, confidence);
                SwingUtilities.invokeLater(() -> tableModel.updatePort(port));
            }

            @Override
            public void labelChanged(final FeedbackPort port, String label) {

                LOGGER.info("The label has changed, port: {}, label: {}", port, label);

                port.setLabel(label);

                // update the stored labels
                try {
                    NodeLabels nodeLabels = getNodeLabels();
                    BidibLabelUtils.replaceFeedbackPortLabel(nodeLabels, port.getId(), port.getLabel());
                    saveLabels();
                }
                catch (Exception ex) {
                    LOGGER.warn("Save feedback labels failed.", ex);

                    String labelPath = ex.getMessage();
                    JOptionPane
                        .showMessageDialog(JOptionPane.getFrameForComponent(null), Resources
                            .getString(NodeExchangeHelper.class, "labelfileerror.message", new Object[] { labelPath }),
                            Resources.getString(NodeExchangeHelper.class, "labelfileerror.title"),
                            JOptionPane.ERROR_MESSAGE);

                    throw new RuntimeException(ex);
                }
            }

            @Override
            public void speedChanged(FeedbackPort port, int address, int speed) {
                SwingUtilities.invokeLater(() -> tableModel.updatePort(port));
            }

            @Override
            public void statusChanged(final NodeInterface node, final FeedbackPort port) {
                SwingUtilities.invokeLater(() -> tableModel.updatePort(port));
            }

            @Override
            public void dynStatesChanged(FeedbackPort port, Collection<FeedbackDynStateData> dynStates) {
                SwingUtilities.invokeLater(() -> tableModel.updatePort(port));
            }
        };

        feedbackPortModel.addFeedbackPortListener(feedbackPortListener);

        feedbackPortListPanel.setPortListener(feedbackPortListener);

        tableModel.setPortListener(new PortModelListener<FeedbackPort>() {

            @Override
            public void labelChanged(FeedbackPort port, String label) {

                LOGGER.info("The label has changed, port: {}, label: {}", port, label);

                // update the stored labels
                try {
                    NodeLabels nodeLabels = getNodeLabels();
                    BidibLabelUtils.replaceFeedbackPortLabel(nodeLabels, port.getId(), port.getLabel());
                    saveLabels();
                }
                catch (Exception ex) {
                    LOGGER.warn("Save feedback labels failed.", ex);

                    String labelPath = ex.getMessage();
                    JOptionPane
                        .showMessageDialog(JOptionPane.getFrameForComponent(null), Resources
                            .getString(NodeExchangeHelper.class, "labelfileerror.message", new Object[] { labelPath }),
                            Resources.getString(NodeExchangeHelper.class, "labelfileerror.title"),
                            JOptionPane.ERROR_MESSAGE);

                    throw new RuntimeException(ex);
                }

            }

            @Override
            public void configChanged(FeedbackPort port, PortConfigKeys... portConfigKeys) {
                // TODO Auto-generated method stub

            }
        });

        return feedbackPortListPanel;
    }

    @Override
    public void openLocoDialog(FeedbackAddressData addressData) {
        LOGGER.info("Open the loco controller dialog, addressData: {}", addressData);

        //
        CommandStationNodeInterface node =
            NodeUtils.findFirstCommandStationNode(mainModel.getNodeProvider().getNodes());
        if (node != null) {

            ApplicationContext applicationContext = DefaultApplicationContext.getInstance();
            org.springframework.context.ApplicationContext ctx =
                applicationContext
                    .get(DefaultApplicationContext.KEY_SPRINGCONTEXT,
                        org.springframework.context.ApplicationContext.class);

            final LocoControllerFactory locoControllerFactory = ctx.getBean(LocoControllerFactory.class);
            final LocoController locoController =
                locoControllerFactory
                    .createLocoController(node, (JFrame) JOptionPane.getFrameForComponent(feedbackPortListPanel),
                        mainModel.getNodeProvider());

            AddressData initialAddress = null;
            if (addressData != null) {
                final EnrailmentDirectionEnum enrailmentDirection = addressData.getType();
                AddressTypeEnum addressType = null;
                switch (enrailmentDirection) {
                    case LOCOMOTIVE_LEFT:
                    case LOCOMOTIVE_RIGHT:
                        addressType = AddressTypeEnum.LOCOMOTIVE_FORWARD;
                        break;
                    case BASIC_ACCESSORY:
                        addressType = AddressTypeEnum.ACCESSORY;
                        break;
                    case EXTENDED_ACCESSORY:
                        addressType = AddressTypeEnum.EXTENDED_ACCESSORY;
                        break;
                    default:
                        break;
                }
                initialAddress = new AddressData(addressData.getAddress(), addressType);
            }
            locoController.start(initialAddress);
        }
    }

    @Override
    public void openPomDialog(final FeedbackAddressData addressData) {

        LOGGER.info("Open the POM dialog, addressData: {}", addressData);

        //
        CommandStationNodeInterface node =
            NodeUtils.findFirstCommandStationNode(mainModel.getNodeProvider().getNodes());
        if (node != null) {

            ApplicationContext applicationContext = DefaultApplicationContext.getInstance();
            org.springframework.context.ApplicationContext ctx =
                applicationContext
                    .get(DefaultApplicationContext.KEY_SPRINGCONTEXT,
                        org.springframework.context.ApplicationContext.class);

            final PomProgrammerControllerFactory pomProgrammerControllerFactory =
                ctx.getBean(PomProgrammerControllerFactory.class);
            final PomProgrammerController pomProgrammerController =
                pomProgrammerControllerFactory
                    .createPomProgrammerController(node,
                        (JFrame) JOptionPane.getFrameForComponent(feedbackPortListPanel), new Point(0, 0));

            pomProgrammerController.addPomProgrammerControllerListener(new PomProgrammerControllerListener() {

                @Override
                public void sendRequest(
                    CommandStationNodeInterface node, PomAddressData locoAddress, CommandStationPom opCode,
                    int cvNumber, int cvValue) {
                    LOGGER.info("Send POM request.");

                    PomAcknowledge pomAck =
                        commandStationService
                            .sendCvPomRequest(ConnectionRegistry.CONNECTION_ID_MAIN, node, locoAddress, opCode,
                                cvNumber, cvValue);
                    LOGGER.info("Received pomAck: {}", pomAck);
                }

                @Override
                public void close() {
                }
            });

            AddressData initialAddress = null;
            if (addressData != null) {
                initialAddress = new AddressData(addressData.getAddress(), AddressTypeEnum.LOCOMOTIVE_FORWARD);
            }

            final DockingDesktop desktop =
                DefaultApplicationContext
                    .getInstance().get(DefaultApplicationContext.KEY_DESKTOP, DockingDesktop.class);
            // open the dialog
            pomProgrammerController.start(desktop, initialAddress);
        }
    }

    private NodeLabels getNodeLabels() {
        final WizardLabelFactory wizardLabelFactory = wizardLabelWrapper.getWizardLabelFactory();

        NodeLabels nodeLabels = wizardLabelFactory.loadLabels(mainModel.getSelectedNode().getUniqueId());
        return nodeLabels;
    }

    private void saveLabels() {
        try {
            long uniqueId = mainModel.getSelectedNode().getUniqueId();
            wizardLabelWrapper.saveNodeLabels(uniqueId);
        }
        catch (Exception e) {
            LOGGER.warn("Save accessory labels failed.", e);
            throw new RuntimeException(e);
        }
    }

    @EventListener(LabelsChangedEvent.class)
    public void labelsChangedEvent(LabelsChangedEvent labelsChangedEvent) {
        LOGGER.info("The labels have changed, node: {}", labelsChangedEvent);

        if (this.feedbackPortListPanel != null) {
            SwingUtilities.invokeLater(() -> this.feedbackPortListPanel.refreshView());
        }
    }
}
