
package org.bidib.wizard.mvc.main.view.panel;

import java.awt.event.MouseEvent;
import java.lang.ref.WeakReference;
import java.util.Collections;
import java.util.LinkedList;
import java.util.List;

import javax.swing.JOptionPane;
import javax.swing.JPanel;
import javax.swing.table.TableModel;

import org.apache.commons.collections4.CollectionUtils;
import org.bidib.jbidibc.messages.enums.LcOutputType;
import org.bidib.wizard.api.locale.Resources;
import org.bidib.wizard.api.model.NodeInterface;
import org.bidib.wizard.api.model.listener.InputPortListener;
import org.bidib.wizard.api.utils.PortListUtils;
import org.bidib.wizard.client.common.controller.NodeSelectionProvider;
import org.bidib.wizard.client.common.dialog.LabelDialog;
import org.bidib.wizard.client.common.model.listener.PortTableLayoutListener;
import org.bidib.wizard.client.common.table.AbstractPortHierarchicalTable;
import org.bidib.wizard.client.common.table.DefaultPortListMenuListener;
import org.bidib.wizard.client.common.view.TabPanelProvider;
import org.bidib.wizard.client.common.view.menu.listener.PortListMenuListener;
import org.bidib.wizard.model.ports.GenericPort;
import org.bidib.wizard.model.ports.InputPort;
import org.bidib.wizard.model.ports.Port;
import org.bidib.wizard.model.ports.event.PortConfigChangeEvent;
import org.bidib.wizard.model.status.InputPortStatus;
import org.bidib.wizard.mvc.main.controller.InputPortPanelController;
import org.bidib.wizard.mvc.main.model.InputPortTableModel;
import org.bidib.wizard.mvc.main.model.MainModel;
import org.bidib.wizard.mvc.main.model.listener.InputPortModelListener;
import org.bidib.wizard.mvc.main.view.panel.listener.TabComponentCreator;
import org.bidib.wizard.mvc.main.view.panel.listener.TabVisibilityListener;
import org.bidib.wizard.mvc.main.view.panel.listener.TabVisibilityProvider;
import org.bidib.wizard.mvc.main.view.table.InputPortTable;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.jidesoft.grid.SortableTableModel;

import io.reactivex.rxjava3.subjects.PublishSubject;

public class InputPortListPanel
    extends
    SimpleHierarchicalPortListPanel<InputPortTableModel, InputPortStatus, InputPort, InputPortListener, InputPortModelListener>
    implements TabVisibilityProvider, PortTableLayoutListener, TabPanelProvider, TabComponentCreator {

    private static final long serialVersionUID = 1L;

    private static final Logger LOGGER = LoggerFactory.getLogger(InputPortListPanel.class);

    private final MainModel mainModel;

    private final TabVisibilityListener tabVisibilityListener;

    public InputPortListPanel(final InputPortPanelController controller, final InputPortTableModel tableModel,
        final MainModel mainModel, final TabVisibilityListener tabVisibilityListener,
        final PublishSubject<PortConfigChangeEvent> portConfigChangeEventSubject) {
        super(tableModel, Resources.getString(InputPortListPanel.class, "emptyTable"), portConfigChangeEventSubject, mainModel);

        this.mainModel = mainModel;
        this.tabVisibilityListener = tabVisibilityListener;
    }

    @Override
    protected AbstractPortHierarchicalTable<InputPort> createPortTable(
        InputPortTableModel tableModel, String emptyTableText) {

        return new InputPortTable(tableModel, emptyTableText);
    }

    @Override
    protected void createTable(
        final InputPortTableModel tableModel, String emptyTableText,
        final PublishSubject<PortConfigChangeEvent> portConfigChangeEventSubject, final NodeSelectionProvider nodeSelectionProvider) {

        super.createTable(tableModel, emptyTableText, portConfigChangeEventSubject, nodeSelectionProvider);

        final PortListMenuListener portListMenuListener = createPortListMenuListener(this.table);
        table.setPortListMenuListener(portListMenuListener);

    }

    private PortListMenuListener createPortListMenuListener(final AbstractPortHierarchicalTable<InputPort> portTable) {
        final DefaultPortListMenuListener listener = new DefaultPortListMenuListener() {
            @Override
            public void editLabel(final MouseEvent popupEvent) {
                final int row = portTable.getRow(popupEvent.getPoint());
                if (row > -1) {
                    Object val = portTable.getValueAt(row, 0);
                    if (val instanceof Port<?>) {
                        val = ((Port<?>) val).toString();
                    }
                    final Object value = val;
                    if (value instanceof String) {
                        // show the port name editor
                        new LabelDialog((String) value, popupEvent.getXOnScreen(), popupEvent.getYOnScreen()) {
                            @Override
                            public void labelChanged(String label) {
                                portTable.setValueAt(label, row, 0);
                            }
                        };
                    }
                }
                else {
                    LOGGER.warn("The row is not available!");
                }
            }

            @Override
            public void mapPort(final MouseEvent popupEvent) {

                final int row = portTable.getRow(popupEvent.getPoint());
                if (row > -1) {
                    Object val = portTable.getValueAt(row, 0);
                    if (val instanceof InputPort) {
                        InputPort inputPort = (InputPort) val;
                        LOGGER.info("Change mapping for port: {}", inputPort);

                        InputPort masterInputPort = null;
                        if (inputPort.getGenericPort() != null
                            && inputPort.getGenericPort().getPairedPortMaster() != null) {

                            LOGGER
                                .info(
                                    "The selected port is currently used as SwitchPairPort. Use the master to switch the port to a InputPort.");
                            WeakReference<GenericPort> masterPort = inputPort.getGenericPort().getPairedPortMaster();
                            if (masterPort != null && masterPort.get() != null) {
                                masterInputPort =
                                    PortListUtils
                                        .findPortByPortNumber(mainModel.getSelectedNode().getInputPorts(),
                                            masterPort.get().getPortNumber());
                                LOGGER.info("Found master inputPort to use: {}", masterInputPort);

                                if (masterInputPort != null && masterInputPort
                                    .getGenericPort().getCurrentPortType() != LcOutputType.SWITCHPAIRPORT) {

                                    // no need to change the master, not sure if this code is reached anyways if
                                    // no switchPair
                                    masterInputPort = null;
                                }
                            }
                        }

                        // confirm switch to input port
                        int result =
                            JOptionPane
                                .showConfirmDialog(JOptionPane.getFrameForComponent(InputPortListPanel.this),
                                    Resources.getString(InputPortListPanel.class, "switch-port-confirm"),
                                    Resources.getString(InputPortListPanel.class, "switch-port-title"),
                                    JOptionPane.OK_CANCEL_OPTION, JOptionPane.QUESTION_MESSAGE);
                        if (result == JOptionPane.OK_OPTION) {

                            TableModel tm = table.getModel();
                            if (tm instanceof SortableTableModel) {
                                SortableTableModel stm = (SortableTableModel) tm;
                                tm = stm.getActualModel();
                            }

                            InputPortTableModel inputPortTableModel = (InputPortTableModel) tm;

                            if (masterInputPort != null) {
                                LOGGER.info("Change the masterInputPort to a input port: {}", masterInputPort);
                                inputPortTableModel.changePortType(LcOutputType.INPUTPORT, masterInputPort);
                            }
                            LOGGER.info("Change the port to a input port: {}", inputPort);
                            inputPortTableModel.changePortType(LcOutputType.INPUTPORT, inputPort);

                        }
                    }
                }
            }
        };
        return listener;
    }

    @Override
    public JPanel getComponent() {
        return this;
    }

    @Override
    public Object getCreator() {
        return this;
    }

    @Override
    public boolean equals(Object other) {
        if (other instanceof TabComponentCreator) {
            TabComponentCreator creator = (TabComponentCreator) other;
            // TODO if more than a single instance is available this must be changed
            if (creator.getCreator() instanceof InputPortListPanel) {
                return true;
            }
        }
        return false;
    }

    @Override
    public int hashCode() {
        return super.hashCode();
    }

    @Override
    public void listChanged() {
        LOGGER.info("The port list has changed.");
        super.listChanged();

        tabVisibilityListener.setTabVisible(this, isTabVisible());

        portTableLayoutChanged();
    }

    @Override
    public void portTableLayoutChanged() {
        boolean hasPortIdentifiers = false;

        List<InputPort> ports = new LinkedList<>();
        ports.addAll(getPorts());

        LOGGER.info("Number of ports: {}", ports.size());

        synchronized (ports) {
            for (InputPort port : ports) {
                if (port.isRemappingEnabled()) {
                    hasPortIdentifiers = true;
                    break;
                }
            }
        }

        // keep the column index of the column to insert in the view
        int viewColumnIndex = InputPortTableModel.COLUMN_IO_BEHAVIOUR;

        final NodeInterface node = mainModel.getSelectedNode();
        if (node != null) {

            LOGGER.info("A node is selected.");
            boolean hasInputPortConfig = false;
            if (node.getNode().isPortFlatModelAvailable() && CollectionUtils.isNotEmpty(node.getEnabledInputPorts())) {
                LOGGER.info("Check if the at least one input port has the input port config available.");
                for (InputPort port : node.getEnabledInputPorts()) {
                    hasInputPortConfig = port.isHasInputPortConfig();
                    if (hasInputPortConfig) {
                        // found one -> show the column
                        break;
                    }
                }
            }
            LOGGER
                .info("List has changed, hasPortIdentifiers: {}, hasInputPortConfig: {}", hasPortIdentifiers,
                    hasInputPortConfig);

            viewColumnIndex =
                table.setColumnVisible(InputPortTableModel.COLUMN_IO_BEHAVIOUR, viewColumnIndex, hasPortIdentifiers);
            viewColumnIndex =
                table.setColumnVisible(InputPortTableModel.COLUMN_SWITCH_OFF_TIME, viewColumnIndex, hasPortIdentifiers);
        }

        // show/hide the port identifiers
        table.setColumnVisible(InputPortTableModel.COLUMN_PORT_IDENTIFIER, viewColumnIndex, hasPortIdentifiers);
    }

    @Override
    public Class<?> getPortClass() {
        return InputPort.class;
    }

    @Override
    public boolean isTabVisible() {
        final NodeInterface node = mainModel.getSelectedNode();
        if (node != null) {
            boolean isTabVisible = node.hasInputPorts();
            LOGGER.debug("Check if tab is visible: {}", isTabVisible);
            return isTabVisible;
        }
        return false;
    }

    @Override
    protected List<InputPort> getPorts() {
        final NodeInterface node = mainModel.getSelectedNode();
        if (node != null) {
            List<InputPort> ports = new LinkedList<>();
            ports.addAll(node.getInputPorts());

            return ports;
        }
        return Collections.emptyList();
    }

    public void refreshView() {
        listChanged();
    }
}
