package org.bidib.wizard.mvc.main.view.panel;

import java.awt.event.MouseEvent;
import java.lang.ref.WeakReference;
import java.util.Collections;
import java.util.LinkedList;
import java.util.List;

import javax.swing.JOptionPane;
import javax.swing.JPanel;
import javax.swing.table.TableModel;

import org.apache.commons.collections4.CollectionUtils;
import org.bidib.jbidibc.messages.BidibLibrary;
import org.bidib.jbidibc.messages.Feature;
import org.bidib.jbidibc.messages.enums.LcOutputType;
import org.bidib.wizard.api.locale.Resources;
import org.bidib.wizard.api.model.NodeInterface;
import org.bidib.wizard.api.model.listener.SwitchPortListener;
import org.bidib.wizard.api.utils.PortListUtils;
import org.bidib.wizard.client.common.controller.NodeSelectionProvider;
import org.bidib.wizard.client.common.dialog.LabelDialog;
import org.bidib.wizard.client.common.table.AbstractPortHierarchicalTable;
import org.bidib.wizard.client.common.table.DefaultPortListMenuListener;
import org.bidib.wizard.client.common.view.TabPanelProvider;
import org.bidib.wizard.client.common.view.menu.listener.PortListMenuListener;
import org.bidib.wizard.model.ports.GenericPort;
import org.bidib.wizard.model.ports.Port;
import org.bidib.wizard.model.ports.SwitchPort;
import org.bidib.wizard.model.ports.event.PortConfigChangeEvent;
import org.bidib.wizard.model.status.SwitchPortStatus;
import org.bidib.wizard.mvc.main.controller.SwitchPortPanelController;
import org.bidib.wizard.mvc.main.model.MainModel;
import org.bidib.wizard.mvc.main.model.SwitchPortTableModel;
import org.bidib.wizard.mvc.main.model.listener.SwitchPortModelListener;
import org.bidib.wizard.mvc.main.view.panel.listener.TabComponentCreator;
import org.bidib.wizard.mvc.main.view.panel.listener.TabVisibilityListener;
import org.bidib.wizard.mvc.main.view.panel.listener.TabVisibilityProvider;
import org.bidib.wizard.mvc.main.view.table.SwitchPortTable;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.jidesoft.grid.SortableTableModel;

import io.reactivex.rxjava3.subjects.PublishSubject;

public class SwitchPortListPanel
    extends
    SimpleHierarchicalPortListPanel<SwitchPortTableModel, SwitchPortStatus, SwitchPort, SwitchPortListener, SwitchPortModelListener>
    implements TabVisibilityProvider, TabPanelProvider, TabComponentCreator {

    private static final long serialVersionUID = 1L;

    private static final Logger LOGGER = LoggerFactory.getLogger(SwitchPortListPanel.class);

    private final MainModel mainModel;

    private final TabVisibilityListener tabVisibilityListener;

    public SwitchPortListPanel(final SwitchPortPanelController controller, final SwitchPortTableModel tableModel,
        final MainModel mainModel, final TabVisibilityListener tabVisibilityListener,
        final PublishSubject<PortConfigChangeEvent> portConfigChangeEventSubject) {
        super(tableModel, Resources.getString(SwitchPortListPanel.class, "emptyTable"), portConfigChangeEventSubject, mainModel);

        this.mainModel = mainModel;
        this.tabVisibilityListener = tabVisibilityListener;
    }

    @Override
    protected AbstractPortHierarchicalTable<SwitchPort> createPortTable(
        SwitchPortTableModel tableModel, String emptyTableText) {

        return new SwitchPortTable(tableModel, emptyTableText);
    }

    @Override
    protected void createTable(
        final SwitchPortTableModel tableModel, String emptyTableText,
        final PublishSubject<PortConfigChangeEvent> portConfigChangeEventSubject, final NodeSelectionProvider nodeSelectionProvider) {

        super.createTable(tableModel, emptyTableText, portConfigChangeEventSubject, nodeSelectionProvider);

        final PortListMenuListener portListMenuListener = createPortListMenuListener(this.table);
        table.setPortListMenuListener(portListMenuListener);

    }

    private PortListMenuListener createPortListMenuListener(final AbstractPortHierarchicalTable<SwitchPort> portTable) {
        final DefaultPortListMenuListener listener = new DefaultPortListMenuListener() {
            @Override
            public void editLabel(final MouseEvent popupEvent) {
                final int row = portTable.getRow(popupEvent.getPoint());
                if (row > -1) {
                    Object val = portTable.getValueAt(row, 0);
                    if (val instanceof Port<?>) {
                        val = ((Port<?>) val).toString();
                    }
                    final Object value = val;
                    if (value instanceof String) {
                        // show the port name editor
                        new LabelDialog((String) value, popupEvent.getXOnScreen(), popupEvent.getYOnScreen()) {
                            @Override
                            public void labelChanged(String label) {
                                portTable.setValueAt(label, row, 0);
                            }
                        };
                    }
                }
                else {
                    LOGGER.warn("The row is not available!");
                }
            }

            @Override
            public void mapPort(final MouseEvent popupEvent) {

                final int row = portTable.getRow(popupEvent.getPoint());
                if (row > -1) {
                    Object val = portTable.getValueAt(row, 0);
                    if (val instanceof SwitchPort) {
                        SwitchPort switchPort = (SwitchPort) val;
                        LOGGER.info("Change mapping for port: {}", switchPort);

                        SwitchPort masterSwitchPort = null;
                        if (switchPort.getGenericPort() != null
                            && switchPort.getGenericPort().getPairedPortMaster() != null) {

                            LOGGER
                                .info(
                                    "The selected port is currently used as SwitchPairPort. Use the master to switch the port to a SwitchPort.");
                            WeakReference<GenericPort> masterPort = switchPort.getGenericPort().getPairedPortMaster();
                            if (masterPort != null && masterPort.get() != null) {
                                masterSwitchPort =
                                    PortListUtils
                                        .findPortByPortNumber(mainModel.getSelectedNode().getSwitchPorts(),
                                            masterPort.get().getPortNumber());
                                LOGGER.info("Found master switchPort to use: {}", masterSwitchPort);

                                if (masterSwitchPort != null && masterSwitchPort
                                    .getGenericPort().getCurrentPortType() != LcOutputType.SWITCHPAIRPORT) {

                                    // no need to change the master, not sure if this code is reached anyways if
                                    // no switchPair
                                    masterSwitchPort = null;
                                }
                            }
                        }

                        // confirm switch to switch port
                        int result =
                            JOptionPane
                                .showConfirmDialog(JOptionPane.getFrameForComponent(SwitchPortListPanel.this),
                                    Resources.getString(SwitchPortListPanel.class, "switch-port-confirm"),
                                    Resources.getString(SwitchPortListPanel.class, "switch-port-title"),
                                    JOptionPane.OK_CANCEL_OPTION, JOptionPane.QUESTION_MESSAGE);
                        if (result == JOptionPane.OK_OPTION) {

                            TableModel tm = table.getModel();
                            if (tm instanceof SortableTableModel) {
                                SortableTableModel stm = (SortableTableModel) tm;
                                tm = stm.getActualModel();
                            }
                            SwitchPortTableModel switchPortTableModel = (SwitchPortTableModel) tm;

                            if (masterSwitchPort != null) {
                                LOGGER.info("Change the masterSwitchPort to a switch port: {}", masterSwitchPort);
                                switchPortTableModel.changePortType(LcOutputType.SWITCHPORT, masterSwitchPort);
                            }
                            LOGGER.info("Change the port to a switch port: {}", switchPort);
                            switchPortTableModel.changePortType(LcOutputType.SWITCHPORT, switchPort);
                        }
                    }
                }
            }
        };
        return listener;
    }

    @Override
    public JPanel getComponent() {
        return this;
    }

    @Override
    public Object getCreator() {
        return this;
    }

    @Override
    public boolean equals(Object other) {
        if (other instanceof TabComponentCreator) {
            TabComponentCreator creator = (TabComponentCreator) other;
            // TODO if more than a single instance is available this must be changed
            if (creator.getCreator() instanceof SwitchPortListPanel) {
                return true;
            }
        }
        return false;
    }

    @Override
    public int hashCode() {
        return super.hashCode();
    }

    @Override
    public void listChanged() {
        LOGGER.info("The port list has changed.");

        super.listChanged();

        boolean hasPortIdentifiers = false;

        List<SwitchPort> ports = new LinkedList<>();
        ports.addAll(getPorts());

        synchronized (ports) {
            for (SwitchPort port : ports) {
                if (port.isRemappingEnabled()) {
                    hasPortIdentifiers = true;
                    break;
                }
            }
        }

        final NodeInterface node = mainModel.getSelectedNode();
        if (node != null) {
            LOGGER.info("A node is selected.");
            boolean hasSwitchPortConfigIo = false;
            boolean hasSwitchPortConfigTicks = false;
            boolean hasSwitchPortConfigLoadType = false;

            if (node.getNode().isPortFlatModelAvailable() && CollectionUtils.isNotEmpty(node.getSwitchPorts())) {
                LOGGER.info("Check if at least one switch port has the switch port config available.");
                for (SwitchPort port : node.getSwitchPorts()) {
                    if (!hasSwitchPortConfigIo) {
                        hasSwitchPortConfigIo =
                            port.isPortConfigKeySupported(Byte.valueOf(BidibLibrary.BIDIB_PCFG_SWITCH_CTRL));
                    }
                    if (!hasSwitchPortConfigTicks) {
                        hasSwitchPortConfigTicks =
                            port.isPortConfigKeySupported(Byte.valueOf(BidibLibrary.BIDIB_PCFG_TICKS));
                    }
                    if (!hasSwitchPortConfigLoadType) {
                        hasSwitchPortConfigLoadType =
                            port.isPortConfigKeySupported(Byte.valueOf(BidibLibrary.BIDIB_PCFG_LOAD_TYPE));
                    }

                    if (hasSwitchPortConfigIo && hasSwitchPortConfigTicks && hasSwitchPortConfigLoadType) {
                        // found one -> show the column
                        break;
                    }
                }
            }
            else {
                Feature switchPortConfigAvailable =
                    Feature.findFeature(node.getNode().getFeatures(), BidibLibrary.FEATURE_SWITCH_CONFIG_AVAILABLE);
                if (switchPortConfigAvailable != null) {
                    hasSwitchPortConfigIo = (switchPortConfigAvailable.getValue() > 0);
                    hasSwitchPortConfigTicks = hasSwitchPortConfigIo;
                }

                for (SwitchPort port : node.getSwitchPorts()) {
                    if (!hasSwitchPortConfigIo) {
                        hasSwitchPortConfigIo =
                            port.isPortConfigKeySupported(Byte.valueOf(BidibLibrary.BIDIB_PCFG_SWITCH_CTRL));
                    }
                    if (!hasSwitchPortConfigTicks) {
                        hasSwitchPortConfigTicks =
                            port.isPortConfigKeySupported(Byte.valueOf(BidibLibrary.BIDIB_PCFG_TICKS));
                    }
                    if (!hasSwitchPortConfigLoadType) {
                        hasSwitchPortConfigLoadType =
                            port.isPortConfigKeySupported(Byte.valueOf(BidibLibrary.BIDIB_PCFG_LOAD_TYPE));
                    }

                    if (hasSwitchPortConfigIo && hasSwitchPortConfigTicks && hasSwitchPortConfigLoadType) {
                        // found one -> show the column
                        break;
                    }
                }
            }
            LOGGER
                .info(
                    "List has changed, hasPortIdentifiers: {}, hasSwitchPortConfigIo: {}, hasSwitchPortConfigTicks: {}",
                    hasPortIdentifiers, hasSwitchPortConfigIo, hasSwitchPortConfigTicks);

            // keep the column index of the column to insert in the view
            int viewColumnIndex = SwitchPortTableModel.COLUMN_IO_BEHAVIOUR;

            // show/hide the IO behaviour column
            viewColumnIndex =
                table
                    .setColumnVisible(SwitchPortTableModel.COLUMN_IO_BEHAVIOUR, viewColumnIndex, hasSwitchPortConfigIo);

            // show/hide the ticks column
            viewColumnIndex =
                table
                    .setColumnVisible(SwitchPortTableModel.COLUMN_SWITCH_OFF_TIME, viewColumnIndex,
                        hasSwitchPortConfigTicks);

            // show/hide the load type column
            viewColumnIndex =
                table
                    .setColumnVisible(SwitchPortTableModel.COLUMN_LOAD_TYPE, viewColumnIndex,
                        hasSwitchPortConfigLoadType);

            // show/hide the port identifier column
            viewColumnIndex =
                table
                    .setColumnVisible(SwitchPortTableModel.COLUMN_PORT_IDENTIFIER, viewColumnIndex, hasPortIdentifiers);
        }

        tabVisibilityListener.setTabVisible(this, isTabVisible());
    }

    @Override
    public Class<?> getPortClass() {
        return SwitchPort.class;
    }

    @Override
    public boolean isTabVisible() {
        final NodeInterface node = mainModel.getSelectedNode();
        if (node != null) {
            boolean isTabVisible = node.hasSwitchPorts();
            LOGGER.debug("Check if tab is visible: {}", isTabVisible);
            return isTabVisible;
        }
        return false;
    }

    @Override
    protected List<SwitchPort> getPorts() {
        final NodeInterface node = mainModel.getSelectedNode();
        if (node != null) {
            List<SwitchPort> ports = new LinkedList<>();
            ports.addAll(node.getSwitchPorts());

            return ports;
        }
        return Collections.emptyList();
    }

    public void refreshView() {
        listChanged();
    }
}
