package org.bidib.wizard.mvc.main.model;

import java.io.File;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.util.ArrayList;
import java.util.List;

import org.apache.commons.io.FileUtils;
import org.assertj.core.api.Assertions;
import org.bidib.wizard.api.model.Accessory;
import org.bidib.wizard.api.model.Flag;
import org.bidib.wizard.api.model.Macro;
import org.bidib.wizard.api.model.PortsProvider;
import org.bidib.wizard.api.model.function.AnalogPortAction;
import org.bidib.wizard.api.model.function.FlagFunction;
import org.bidib.wizard.api.model.function.Function;
import org.bidib.wizard.api.model.function.InputFunction;
import org.bidib.wizard.api.model.function.LightPortAction;
import org.bidib.wizard.api.model.function.MotorPortAction;
import org.bidib.wizard.api.model.function.ServoPortAction;
import org.bidib.wizard.api.model.function.SoundPortAction;
import org.bidib.wizard.api.model.function.SwitchPortAction;
import org.bidib.wizard.model.ports.AnalogPort;
import org.bidib.wizard.model.ports.BacklightPort;
import org.bidib.wizard.model.ports.FeedbackPort;
import org.bidib.wizard.model.ports.GenericPort;
import org.bidib.wizard.model.ports.InputPort;
import org.bidib.wizard.model.ports.LightPort;
import org.bidib.wizard.model.ports.MotorPort;
import org.bidib.wizard.model.ports.Port;
import org.bidib.wizard.model.ports.ServoPort;
import org.bidib.wizard.model.ports.SoundPort;
import org.bidib.wizard.model.ports.SwitchPairPort;
import org.bidib.wizard.model.ports.SwitchPort;
import org.bidib.wizard.model.status.AnalogPortStatus;
import org.bidib.wizard.model.status.FlagStatus;
import org.bidib.wizard.model.status.InputStatus;
import org.bidib.wizard.model.status.MotorPortStatus;
import org.bidib.wizard.model.status.SoundPortStatus;
import org.bidib.wizard.model.status.SwitchPortStatus;
import org.bidib.wizard.mvc.main.model.MacroFactory.ExportFormat;
import org.junit.jupiter.api.BeforeAll;
import org.junit.jupiter.api.Test;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class MacroFactoryTest {
    private static final Logger LOGGER = LoggerFactory.getLogger(MacroFactoryTest.class);

    private static final String EXPORTED_MACRO_TARGET_DIR = "target/exported";

    private static PortsProvider portsProvider;

    @BeforeAll
    static void prepare() {
        LOGGER.info("Create report directory: {}", EXPORTED_MACRO_TARGET_DIR);

        try {
            FileUtils.forceMkdir(new File(EXPORTED_MACRO_TARGET_DIR));
        }
        catch (IOException e) {
            LOGGER.warn("Create report directory failed: " + EXPORTED_MACRO_TARGET_DIR, e);
        }

        portsProvider = new PortsProvider() {

            private List<SwitchPort> switchPorts;

            private List<SwitchPairPort> switchPairPorts;

            private List<SoundPort> soundPorts;

            private List<ServoPort> servoPorts;

            private List<MotorPort> motorPorts;

            private List<LightPort> lightPorts;

            private List<BacklightPort> backlightPorts;

            private List<InputPort> inputPorts;

            private List<Flag> flags;

            private List<FeedbackPort> feedbackPorts;

            private List<AnalogPort> analogPorts;

            private List<Macro> macros;

            private List<Accessory> accessories;

            @Override
            public List<SwitchPort> getEnabledSwitchPorts() {
                if (switchPorts == null) {
                    switchPorts = new ArrayList<>();
                    for (int portNumber = 0; portNumber < 4; portNumber++) {
                        SwitchPort switchPort = new SwitchPort();
                        switchPort.setId(portNumber);
                        switchPort.setLabel("SwitchPort" + portNumber);
                        switchPorts.add(switchPort);
                    }
                }
                return switchPorts;
            }

            @Override
            public List<SwitchPairPort> getSwitchPairPorts() {
                return getEnabledSwitchPairPorts();
            }

            @Override
            public List<SwitchPairPort> getEnabledSwitchPairPorts() {
                if (switchPairPorts == null) {
                    switchPairPorts = new ArrayList<>();
                    for (int portNumber = 0; portNumber < 4; portNumber++) {
                        SwitchPairPort switchPairPort = new SwitchPairPort();
                        switchPairPort.setId(portNumber);
                        switchPairPort.setLabel("SwitchPairPort" + portNumber);
                        switchPairPorts.add(switchPairPort);
                    }
                }
                return switchPairPorts;
            }

            @Override
            public List<SoundPort> getEnabledSoundPorts() {
                if (soundPorts == null) {
                    soundPorts = new ArrayList<SoundPort>();
                    for (int portNumber = 0; portNumber < 4; portNumber++) {
                        SoundPort soundPort = new SoundPort();
                        soundPort.setId(portNumber);
                        soundPort.setLabel("SoundPort" + portNumber);
                        soundPorts.add(soundPort);
                    }
                }
                return soundPorts;
            }

            @Override
            public List<ServoPort> getServoPorts() {
                if (servoPorts == null) {
                    servoPorts = new ArrayList<ServoPort>();
                    for (int portNumber = 0; portNumber < 4; portNumber++) {
                        ServoPort servoPort = new ServoPort();
                        servoPort.setId(portNumber);
                        servoPort.setLabel("ServoPort" + portNumber);
                        servoPorts.add(servoPort);
                    }
                }
                return servoPorts;
            }

            @Override
            public List<MotorPort> getMotorPorts() {
                if (motorPorts == null) {
                    motorPorts = new ArrayList<MotorPort>();
                    for (int portNumber = 0; portNumber < 4; portNumber++) {
                        MotorPort motorPort = new MotorPort();
                        motorPort.setId(portNumber);
                        motorPort.setLabel("MotorPort" + portNumber);
                        motorPorts.add(motorPort);
                    }
                }
                return motorPorts;
            }

            @Override
            public List<LightPort> getLightPorts() {
                if (lightPorts == null) {
                    lightPorts = new ArrayList<LightPort>();
                    for (int portNumber = 0; portNumber < 12; portNumber++) {
                        LightPort lightPort = new LightPort();
                        lightPort.setId(portNumber);
                        lightPort.setLabel("LightPort" + portNumber);
                        lightPorts.add(lightPort);
                    }
                }
                return lightPorts;
            }

            @Override
            public List<LightPort> getEnabledLightPorts() {
                if (lightPorts == null) {
                    lightPorts = new ArrayList<LightPort>();
                    for (int portNumber = 0; portNumber < 12; portNumber++) {
                        LightPort lightPort = new LightPort();
                        lightPort.setId(portNumber);
                        lightPort.setLabel("LightPort" + portNumber);
                        lightPorts.add(lightPort);
                    }
                }
                return lightPorts;
            }

            @Override
            public List<BacklightPort> getBacklightPorts() {
                if (backlightPorts == null) {
                    backlightPorts = new ArrayList<BacklightPort>();
                    for (int portNumber = 0; portNumber < 12; portNumber++) {
                        BacklightPort lightPort = new BacklightPort();
                        lightPort.setId(portNumber);
                        lightPort.setLabel("BacklightPort" + portNumber);
                        backlightPorts.add(lightPort);
                    }
                }
                return backlightPorts;
            }

            @Override
            public List<InputPort> getEnabledInputPorts() {
                if (inputPorts == null) {
                    inputPorts = new ArrayList<InputPort>();
                    for (int portNumber = 0; portNumber < 12; portNumber++) {
                        InputPort inputPort = new InputPort();
                        inputPort.setId(portNumber);
                        inputPort.setLabel("InputPort" + portNumber);
                        inputPorts.add(inputPort);
                    }
                }
                return inputPorts;
            }

            @Override
            public List<Flag> getFlags() {
                if (flags == null) {
                    flags = new ArrayList<Flag>();
                    for (int flagNumber = 0; flagNumber < 8; flagNumber++) {
                        Flag flag = new Flag();
                        flag.setId(flagNumber);
                        flags.add(flag);
                    }
                }
                return flags;
            }

            @Override
            public List<FeedbackPort> getFeedbackPorts() {
                if (feedbackPorts == null) {
                    feedbackPorts = new ArrayList<FeedbackPort>();
                    for (int portNumber = 0; portNumber < 48; portNumber++) {
                        FeedbackPort feedbackPort = new FeedbackPort();
                        feedbackPort.setId(portNumber);
                        feedbackPort.setLabel("FeedbackPort" + portNumber);
                        feedbackPorts.add(feedbackPort);
                    }
                }
                return feedbackPorts;
            }

            @Override
            public List<AnalogPort> getAnalogPorts() {
                if (analogPorts == null) {
                    analogPorts = new ArrayList<AnalogPort>();
                    for (int portNumber = 0; portNumber < 12; portNumber++) {
                        AnalogPort analogPort = new AnalogPort();
                        analogPort.setId(portNumber);
                        analogPort.setLabel("AnalogPort" + portNumber);
                        analogPorts.add(analogPort);
                    }
                }
                return analogPorts;
            }

            @Override
            public List<Macro> getMacros() {
                if (macros == null) {
                    macros = new ArrayList<Macro>();
                    for (int macroNumber = 0; macroNumber < 8; macroNumber++) {
                        Macro macro = new Macro(38 /* max macro points */);
                        macro.setId(macroNumber);
                        macros.add(macro);
                    }
                }
                return macros;
            }

            @Override
            public List<Accessory> getAccessories() {
                if (accessories == null) {
                    accessories = new ArrayList<Accessory>();
                    for (int accessoryNumber = 0; accessoryNumber < 8; accessoryNumber++) {
                        Accessory accessory = new Accessory();
                        accessory.setId(accessoryNumber);
                        accessories.add(accessory);
                    }
                }
                return accessories;
            }

            @Override
            public boolean isFlatPortModel() {
                return false;
            }

            @Override
            public List<InputPort> getInputPorts() {
                return getEnabledInputPorts();
            }

            @Override
            public List<SwitchPort> getSwitchPorts() {
                return getEnabledSwitchPorts();
            }

            @Override
            public List<GenericPort> getGenericPorts() {
                return null;
            }

            @Override
            public List<Port<?>> getPorts() {
                return null;
            }

            @Override
            public List<SoundPort> getSoundPorts() {
                return getEnabledSoundPorts();
            }
        };
    }

    @Test
    public void loadMacroWithJaxb() {
        String fileName = MacroFactoryTest.class.getResource("/macros/Macro_0.mxml").getPath();
        Macro macro = MacroFactory.loadMacro(fileName, ExportFormat.jaxb, portsProvider);

        Assertions.assertThat(macro).isNotNull();

        LOGGER.info("Loaded macro: {}", macro);

        Assertions.assertThat(macro.getId()).isEqualTo(0);
        Assertions.assertThat(macro.getLabel()).isEqualTo("Macro_0");
        Assertions.assertThat(macro.getFunctions().size()).isEqualTo(15);

        Assertions.assertThat(macro.getFunctions().get(0) instanceof LightPortAction).isTrue();
        Assertions.assertThat(macro.getFunctions().get(1) instanceof ServoPortAction).isTrue();
        Assertions.assertThat(macro.getFunctions().get(2) instanceof LightPortAction).isTrue();
        Assertions.assertThat(macro.getFunctions().get(3) instanceof ServoPortAction).isTrue();
        Assertions.assertThat(macro.getFunctions().get(4) instanceof LightPortAction).isTrue();
        Assertions.assertThat(macro.getFunctions().get(5) instanceof LightPortAction).isTrue();
        Assertions.assertThat(macro.getFunctions().get(6) instanceof SwitchPortAction).isTrue();
        Assertions.assertThat(macro.getFunctions().get(7) instanceof SoundPortAction).isTrue();
        Assertions.assertThat(macro.getFunctions().get(8) instanceof MotorPortAction).isTrue();
        Assertions.assertThat(macro.getFunctions().get(9) instanceof InputFunction).isTrue();
        Assertions.assertThat(macro.getFunctions().get(10) instanceof InputFunction).isTrue();
        Assertions.assertThat(macro.getFunctions().get(11) instanceof FlagFunction).isTrue();
        Assertions.assertThat(macro.getFunctions().get(12) instanceof FlagFunction).isTrue();
        Assertions.assertThat(macro.getFunctions().get(13) instanceof FlagFunction).isTrue();

        FlagFunction flagFunction1 = (FlagFunction) macro.getFunctions().get(11);
        Assertions.assertThat(flagFunction1.getAction()).isEqualTo(FlagStatus.SET);

        FlagFunction flagFunction2 = (FlagFunction) macro.getFunctions().get(12);
        Assertions.assertThat(flagFunction2.getAction()).isEqualTo(FlagStatus.CLEAR);

        FlagFunction flagFunction3 = (FlagFunction) macro.getFunctions().get(13);
        Assertions.assertThat(flagFunction3.getAction()).isEqualTo(FlagStatus.QUERY_1);

        FlagFunction flagFunction4 = (FlagFunction) macro.getFunctions().get(14);
        Assertions.assertThat(flagFunction4.getAction()).isEqualTo(FlagStatus.QUERY_0);
    }

    @Test
    public void loadMacroWithJaxbWithInvalidPortNumber() {
        String fileName = MacroFactoryTest.class.getResource("/macros/Macro_0.mxml").getPath();
        LightPort removedLightPort = null;
        Macro macro = null;
        try {
            // remove light port 0 to have the exception thrown
            removedLightPort = portsProvider.getLightPorts().remove(0);
            macro = MacroFactory.loadMacro(fileName, ExportFormat.jaxb, portsProvider);
        }
        finally {
            portsProvider.getLightPorts().add(removedLightPort);
        }

        Assertions.assertThat(macro).isNotNull();
        Assertions.assertThat(macro.getFunctions()).isNotNull();

        for (Function<?> function : macro.getFunctions()) {
            if (function instanceof LightPortAction) {
                LightPortAction lightPortAction = (LightPortAction) function;
                if (lightPortAction.getPort().getId() == removedLightPort.getId()) {
                    // should have been replaced by EmptyFunction
                    Assertions.fail("Found a lightport with the id of the removed lightport!");
                }
            }
        }
    }

    @Test
    public void loadMacroWithJaxbAnalogPorts() {
        String fileName = MacroFactoryTest.class.getResource("/macros/Makro_AnalogPorts.mxml").getPath();
        Macro macro = MacroFactory.loadMacro(fileName, ExportFormat.jaxb, portsProvider);

        Assertions.assertThat(macro).isNotNull();

        LOGGER.info("Loaded macro: {}", macro);

        Assertions.assertThat(macro.getId()).isEqualTo(3);
        Assertions.assertThat(macro.getLabel()).isEqualTo("Makro_3");
        Assertions.assertThat(macro.getFunctions().size()).isEqualTo(2);

        Assertions.assertThat(macro.getFunctions().get(0) instanceof AnalogPortAction).isTrue();
        AnalogPortAction analogPortAction = (AnalogPortAction) macro.getFunctions().get(0);
        Assertions.assertThat(analogPortAction.getPort()).isNotNull();
        Assertions.assertThat(analogPortAction.getPort().getId()).isEqualTo(0);
        Assertions.assertThat(analogPortAction.getAction()).isEqualTo(AnalogPortStatus.START);
        Assertions.assertThat(analogPortAction.getValue()).isEqualTo(23);
        Assertions.assertThat(analogPortAction.getDelay()).isEqualTo(20);

        Assertions.assertThat(macro.getFunctions().get(1) instanceof AnalogPortAction).isTrue();
        analogPortAction = (AnalogPortAction) macro.getFunctions().get(1);
        Assertions.assertThat(analogPortAction.getPort()).isNotNull();
        Assertions.assertThat(analogPortAction.getPort().getId()).isEqualTo(10);
        Assertions.assertThat(analogPortAction.getAction()).isEqualTo(AnalogPortStatus.START);
        Assertions.assertThat(analogPortAction.getValue()).isEqualTo(12);
        Assertions.assertThat(analogPortAction.getDelay()).isEqualTo(30);
    }

    @Test
    public void saveMacroWithJaxb() throws FileNotFoundException, IOException {

        List<Function<?>> newFunctions = new ArrayList<>();

        // add more ports
        SwitchPort switchPort = new SwitchPort();
        switchPort.setId(2);
        switchPort.setLabel("SwitchPort2");
        SwitchPortAction switchPortAction = new SwitchPortAction(SwitchPortStatus.ON, switchPort, 30);
        newFunctions.add(switchPortAction);

        SoundPort soundPort = new SoundPort();
        soundPort.setId(2);
        soundPort.setLabel("SoundPort2");
        SoundPortAction soundPortAction = new SoundPortAction(SoundPortStatus.PLAY, soundPort, 25, 10);
        newFunctions.add(soundPortAction);

        MotorPort motorPort = new MotorPort();
        motorPort.setId(2);
        motorPort.setLabel("MotorPort2");
        MotorPortAction motorPortAction = new MotorPortAction(MotorPortStatus.FORWARD, motorPort, 25, 10);
        newFunctions.add(motorPortAction);

        InputPort inputPort2 = new InputPort();
        inputPort2.setId(2);
        inputPort2.setLabel("InputPort2");
        InputFunction inputFunction0 = new InputFunction(InputStatus.QUERY0, inputPort2);
        newFunctions.add(inputFunction0);

        InputPort inputPort3 = new InputPort();
        inputPort3.setId(3);
        inputPort3.setLabel("InputPort3");
        InputFunction inputFunction1 = new InputFunction(InputStatus.QUERY1, inputPort3);
        newFunctions.add(inputFunction1);

        Flag flag1 = new Flag(1);
        FlagFunction flagFunction1 = new FlagFunction(FlagStatus.SET, flag1);
        newFunctions.add(flagFunction1);

        Flag flag2 = new Flag(2);
        FlagFunction flagFunction2 = new FlagFunction(FlagStatus.CLEAR, flag2);
        newFunctions.add(flagFunction2);

        Flag flag3 = new Flag(3);
        FlagFunction flagFunction3 = new FlagFunction(FlagStatus.QUERY_1, flag3);
        newFunctions.add(flagFunction3);

        Flag flag4 = new Flag(4);
        FlagFunction flagFunction4 = new FlagFunction(FlagStatus.QUERY_0, flag4);
        newFunctions.add(flagFunction4);

        final Macro macro = new Macro(newFunctions.size());
        macro.setLabel("Test Macro");
        macro.setFunctions(newFunctions);

        File exportFile = new File(EXPORTED_MACRO_TARGET_DIR, "macro_2.xml");
        MacroFactory.saveMacro(exportFile.getPath(), macro, ExportFormat.jaxb, false);
    }
}
