package org.bidib.wizard.mvc.main.view.panel;

import java.awt.BorderLayout;
import java.awt.Color;
import java.awt.Font;
import java.awt.event.ActionEvent;
import java.beans.PropertyChangeEvent;
import java.beans.PropertyChangeListener;

import javax.swing.AbstractAction;
import javax.swing.JButton;
import javax.swing.JLabel;
import javax.swing.JPanel;
import javax.swing.SwingConstants;
import javax.swing.SwingUtilities;
import javax.swing.UIManager;

import org.bidib.api.json.types.ConnectionPhase;
import org.bidib.wizard.api.locale.Resources;
import org.bidib.wizard.api.model.common.PreferencesPortType;
import org.bidib.wizard.common.model.settings.ConnectionConfiguration;
import org.bidib.wizard.core.model.connection.ConnectionRegistry;
import org.bidib.wizard.core.model.settings.GlobalSettings;
import org.bidib.wizard.core.service.SettingsService;
import org.bidib.wizard.core.utils.AopUtils;
import org.bidib.wizard.mvc.main.model.ConnectionPhaseModel;
import org.bidib.wizard.mvc.main.view.menu.listener.MainMenuListener;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.jgoodies.forms.builder.FormBuilder;
import com.jgoodies.forms.debug.FormDebugPanel;
import com.jgoodies.forms.factories.DefaultComponentFactory;
import com.jgoodies.forms.factories.Paddings;

/**
 * EmptyPanel
 * 
 */
public class EmptyPanel {
    private static final Logger LOGGER = LoggerFactory.getLogger(EmptyPanel.class);

    private static final String ENCODED_DIALOG_COLUMN_SPECS = "pref, 3dlu, pref, 3dlu, 20dlu:grow:fill";

    private static final String ENCODED_DIALOG_ROW_SPECS = "pref, 9dlu, pref";

    private JPanel content;

    private JLabel infoLabel;

    private JButton configureButton;

    private final SettingsService settingsService;

    private final MainMenuListener mainMenuListener;

    public EmptyPanel(final ConnectionPhaseModel connectionPhaseModel, final SettingsService settingsService,
        final MainMenuListener mainMenuListener) {
        LOGGER.debug("Create new EmptyPanel.");
        this.settingsService = settingsService;
        this.mainMenuListener = mainMenuListener;

        // create form builder
        FormBuilder dialogBuilder = null;
        boolean debugDialog = false;
        if (debugDialog) {
            JPanel panel = new FormDebugPanel();
            dialogBuilder =
                FormBuilder.create().columns(ENCODED_DIALOG_COLUMN_SPECS).rows(ENCODED_DIALOG_ROW_SPECS).panel(panel);
        }
        else {
            JPanel panel = new JPanel(new BorderLayout());
            dialogBuilder =
                FormBuilder.create().columns(ENCODED_DIALOG_COLUMN_SPECS).rows(ENCODED_DIALOG_ROW_SPECS).panel(panel);
        }
        dialogBuilder.border(Paddings.DIALOG);

        JLabel welcomeLabel =
            DefaultComponentFactory
                .getInstance()
                .createHeaderLabel("<html><b>" + Resources.getString(getClass(), "welcome") + "</b></html>");
        Font labelFont = welcomeLabel.getFont();
        Font welcomeFont = labelFont.deriveFont(Font.BOLD, labelFont.getSize() * 2.0f);
        welcomeLabel.setFont(welcomeFont);
        // welcomeLabel.setForeground(Color.BLUE);
        welcomeLabel.setHorizontalAlignment(SwingConstants.LEFT);

        dialogBuilder.add(welcomeLabel).xyw(1, 1, 5);

        // open preferences button
        configureButton =
            DefaultComponentFactory
                .getInstance()
                .createButton(new AbstractAction(Resources.getString("org.bidib.wizard.mvc.main.view.menu.MainMenuBar", "preferences")) {

                    private static final long serialVersionUID = 1L;

                    @Override
                    public void actionPerformed(ActionEvent e) {
                        firePreferences();
                    }
                });

        infoLabel = DefaultComponentFactory.getInstance().createLabel("");
        infoLabel.setHorizontalAlignment(SwingConstants.LEFT);
        dialogBuilder.add(infoLabel).xy(1, 3);

        dialogBuilder.add(configureButton).xy(3, 3);

        content = dialogBuilder.build();

        updatePanelText(settingsService);

        if (configureButton != null) {
            try {
                final GlobalSettings gs = AopUtils.getTargetObject(settingsService.getGlobalSettings());
                gs.addPropertyChangeListener(GlobalSettings.PROPERTY_SELECTED_PORTTYPE, new PropertyChangeListener() {
                    @Override
                    public void propertyChange(PropertyChangeEvent evt) {
                        // listen to changes
                        if (GlobalSettings.PROPERTY_SELECTED_PORTTYPE.equals(evt.getPropertyName())) {
                            LOGGER.info("The selected port type has changed.");

                            updatePanelText(settingsService);
                        }
                    }
                });
            }
            catch (Exception ex) {
                LOGGER.warn("Register for changes of selected port type failed.", ex);
            }
        }

        connectionPhaseModel.addPropertyChangeListener(ConnectionPhaseModel.PROPERTY_CONNECTIONPHASE, evt -> {
            ConnectionPhase phase = connectionPhaseModel.getConnectionPhase();
            String connectionId = connectionPhaseModel.getConnectionId();
            switch (phase) {
                case CONNECTED:
                    showMessageText(connectionId, true);
                    break;
                case DISCONNECTED:
                    showMessageText(connectionId, false);
                    break;
                default:
                    break;
            }
        });
    }

    public JPanel getContent() {
        return content;
    }

    private void updatePanelText(final SettingsService settingsService) {

        // must get the correct connection from the globalSettings
        PreferencesPortType preferencesPortType =
            ConnectionConfiguration
                .toPreferencesPortType(settingsService.getGlobalSettings().getConnectionConfigurations(),
                    ConnectionRegistry.CONNECTION_ID_MAIN);
        if (PreferencesPortType.isSimulation(preferencesPortType)) {
            // change the text on the info label
            infoLabel
                .setText(
                    "<html>" + Resources.getString(EmptyPanel.class, "simulation-connection-configured") + "</html>");

            infoLabel.setForeground(Color.RED);
            configureButton.setVisible(true);
        }
        else if (preferencesPortType != null
            && PreferencesPortType.isValidConnectionConfiguration(preferencesPortType)) {
            infoLabel.setText("<html><b>" + Resources.getString(EmptyPanel.class, "connect-to-system") + "</b></html>");

            infoLabel.setForeground(UIManager.getColor("Text.foreground"));
            configureButton.setVisible(false);
        }
        else {
            // change the text on the info label
            infoLabel
                .setText("<html><b>" + Resources.getString(EmptyPanel.class, "no-connection-to-system-configured")
                    + "</b></html>");

            infoLabel.setForeground(Color.RED);
            configureButton.setVisible(true);
        }
    }

    private void setLabelText(String labelText) {
        LOGGER.info("Set the label text: {}", labelText);

        infoLabel.setText(labelText);
        infoLabel.setForeground(UIManager.getColor("Text.foreground"));
    }

    public String getName() {
        return Resources.getString(getClass(), "name");
    }

    private void firePreferences() {
        this.mainMenuListener.preferences();
        // this.context.publishEvent(new MenuEvent(Action.preferences));
    }

    private void showMessageText(String connectionId, boolean connected) {

        LOGGER.info("Show message text for connection status, connected: {}", connected);

        if (connected) {
            if (!SwingUtilities.isEventDispatchThread()) {
                SwingUtilities.invokeLater(new Runnable() {
                    @Override
                    public void run() {
                        setLabelText(
                            "<html><b>" + Resources.getString(EmptyPanel.class, "selectNodeFromList") + "</b></html>");
                        configureButton.setVisible(false);
                    }
                });
            }
            else {
                setLabelText("<html><b>" + Resources.getString(EmptyPanel.class, "selectNodeFromList") + "</b></html>");
                configureButton.setVisible(false);
            }
        }
        else {

            if (!SwingUtilities.isEventDispatchThread()) {
                SwingUtilities.invokeLater(new Runnable() {
                    @Override
                    public void run() {
                        updatePanelText(settingsService);
                    }
                });
            }
            else {
                updatePanelText(settingsService);
            }
        }
    }

}
