package org.bidib.wizard.mvc.main.view.panel;

import java.awt.BorderLayout;
import java.awt.Color;
import java.awt.Dimension;
import java.beans.PropertyChangeEvent;
import java.beans.PropertyChangeListener;
import java.io.File;
import java.util.Collections;
import java.util.LinkedList;
import java.util.List;
import java.util.Optional;

import javax.swing.ImageIcon;
import javax.swing.JLabel;
import javax.swing.JPanel;
import javax.swing.JScrollPane;
import javax.swing.UIManager;

import org.apache.commons.collections4.CollectionUtils;
import org.apache.commons.collections4.IterableUtils;
import org.apache.commons.collections4.Predicate;
import org.apache.commons.lang3.StringUtils;
import org.bidib.jbidibc.core.schema.BidibFactory;
import org.bidib.jbidibc.core.schema.DecoderVendorFactory;
import org.bidib.jbidibc.core.schema.bidib.products.ProductType;
import org.bidib.jbidibc.core.schema.bidib2.FeatureCode;
import org.bidib.jbidibc.core.schema.bidib2.common.DocumentationType;
import org.bidib.jbidibc.core.schema.decodervendor.VendorType;
import org.bidib.jbidibc.exchange.bidib.ProductsUtils;
import org.bidib.jbidibc.messages.Feature;
import org.bidib.jbidibc.messages.StringData;
import org.bidib.jbidibc.messages.utils.ByteUtils;
import org.bidib.jbidibc.messages.utils.ConversionUtils;
import org.bidib.jbidibc.messages.utils.NodeUtils;
import org.bidib.wizard.api.LookupService;
import org.bidib.wizard.api.locale.Resources;
import org.bidib.wizard.api.model.NodeInterface;
import org.bidib.wizard.api.utils.XmlLocaleUtils;
import org.bidib.wizard.client.common.component.HtmlColorPane;
import org.bidib.wizard.client.common.view.BidibNodeNameUtils;
import org.bidib.wizard.client.common.view.TabPanelProvider;
import org.bidib.wizard.common.utils.ImageUtils;
import org.bidib.wizard.core.service.SettingsService;
import org.bidib.wizard.mvc.main.model.MainModel;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.jgoodies.forms.builder.FormBuilder;
import com.jgoodies.forms.factories.Paddings;

public class InfoPanel implements TabPanelProvider {

    private static final Logger LOGGER = LoggerFactory.getLogger(InfoPanel.class);

    private static final String ENCODED_DIALOG_COLUMN_SPECS = "fill:pref:grow, 5dlu, 200dlu";

    private static final String ENCODED_DIALOG_ROW_SPECS = "fill:200dlu:grow";

    private final MainModel mainModel;

    private final HtmlColorPane infoArea = new HtmlColorPane();

    private final JPanel contentPanel;

    private List<VendorType> vendors;

    // private Map<Integer, List<ProductType>> productsMap = new HashMap<>();

    private JScrollPane logsPane;

    private final PropertyChangeListener nodeListener;

    private NodeInterface selectedNode;

    private JLabel imageContainer;

    private List<FeatureCode> featureCodes;

    private final SettingsService settingsService;

    private final LookupService lookupService;

    public static class InfoFormPanel extends JPanel {

        private static final long serialVersionUID = 1L;

        public InfoFormPanel() {
            super(new BorderLayout());
        }

        @Override
        public String getName() {
            // this is used as tab title
            return Resources.getString(InfoPanel.class, "name");
        }
    }

    public InfoPanel(final MainModel model, final SettingsService settingsService, final LookupService lookupService) {
        this.mainModel = model;
        this.settingsService = settingsService;
        this.lookupService = lookupService;

        nodeListener = new PropertyChangeListener() {

            @Override
            public void propertyChange(PropertyChangeEvent evt) {
                switch (evt.getPropertyName()) {
                    case org.bidib.jbidibc.messages.Node.PROPERTY_FEATURES:
                        LOGGER.info("The features have changed, update the node info.");
                        // the features have changed
                        updateNodeInfo(selectedNode);
                        break;
                    case org.bidib.jbidibc.messages.Node.PROPERTY_USERNAME:
                        LOGGER.info("The username has changed, update the node info.");
                        // the features have changed
                        updateNodeInfo(selectedNode);
                        break;
                }
            }
        };

        // load the feature codes
        featureCodes = BidibFactory.getFeatureCodes();

        vendors = DecoderVendorFactory.getDecoderVendors();

        // preload opendcc products
        // SwingUtilities.invokeLater(() -> loadProducts(13));

        // create form builder
        final FormBuilder dialogBuilder =
            FormBuilder
                .create().columns(ENCODED_DIALOG_COLUMN_SPECS).rows(ENCODED_DIALOG_ROW_SPECS)
                .panel(new InfoFormPanel());
        dialogBuilder.border(Paddings.TABBED_DIALOG);

        // infoArea.setEditable(false);
        // infoArea.setFont(new Font("Monospaced", Font.PLAIN, 11));

        infoArea.initialize();

        logsPane = new JScrollPane(infoArea);

        // logsPane.setAutoscrolls(true);
        dialogBuilder.add(logsPane).xy(1, 1);

        imageContainer = new JLabel();
        imageContainer.setPreferredSize(new Dimension(200, 200));
        imageContainer.setMaximumSize(new Dimension(200, 200));

        JPanel imagePanel = new JPanel(new BorderLayout());
        imagePanel.setOpaque(false);
        imagePanel.add(imageContainer, BorderLayout.NORTH);

        dialogBuilder.add(imagePanel).xy(3, 1);

        JScrollPane scrollContent = new JScrollPane(dialogBuilder.build());
        this.contentPanel = new JPanel(new BorderLayout()) {
            private static final long serialVersionUID = 1L;

            @Override
            public String getName() {
                // this is used as tab title
                return Resources.getString(InfoPanel.class, "name");
            }
        };
        this.contentPanel.add(scrollContent, BorderLayout.CENTER);
    }

    @Override
    public JPanel getComponent() {
        return contentPanel;
    }

    public void nodeChanged() {
        LOGGER.debug("The selected node has changed.");

        updateComponentState();
    }

    private void updateComponentState() {

        final NodeInterface node = mainModel.getSelectedNode();

        if (selectedNode != null && !selectedNode.equals(node)) {
            // the selected node has changed.
            selectedNode.getNode().removePropertyChangeListener(nodeListener);
        }

        if (node != null) {
            selectedNode = node;

            // register the change listener for features
            selectedNode.getNode().addPropertyChangeListener(nodeListener);

            updateNodeInfo(node);
        }
        else {
            selectedNode = null;
        }
        // force scroll to top
        infoArea.setCaretPosition(0);
    }

    private void updateNodeInfo(final NodeInterface node) {
        if (node == null) {
            // no node available
            return;
        }

        final org.bidib.jbidibc.messages.Node coreNode = node.getNode();

        LOGGER
            .info("Update the node info panel for node: {}, coreNode: {}, coreNode identity: {}", node, coreNode,
                coreNode.getIdentity());

        // infoArea.setText(null);
        infoArea.clear();

        long uniqueId = node.getUniqueId();
        int vendorId = NodeUtils.getVendorId(node.getUniqueId());
        int productId = NodeUtils.getPid(node.getUniqueId(), coreNode.getRelevantPidBits());

        LOGGER
            .info("Search for vendorId: {} (0x{}), productId: {} (0x{})", vendorId, ByteUtils.intToHex(vendorId),
                productId, ByteUtils.intToHex(productId));

        // List<ProductType> products = productsMap.get(vendorId);
        // if (products == null) {
        // products = loadProducts(vendorId);
        // }
        Optional<ProductType> product = this.lookupService.getProduct(vendorId, productId);

        // prepare information of node
        StringBuilder sb = new StringBuilder();
        StringBuilder description = new StringBuilder();
        String url = null;
        if (product.isPresent()) {
            description.append(product.get().getName());
            // get the documentation in the correct language
            String language = XmlLocaleUtils.getXmlLocale();
            DocumentationType doc = ProductsUtils.getDocumentationOfLanguage(product.get(), language);
            if (doc != null) {
                if (StringUtils.isNotBlank(doc.getDescription())) {
                    description.append(" - ").append(doc.getDescription());
                }

                url = doc.getUrl();
            }
        }
        else {
            LOGGER.info("No products available to get the description for node: {}", node);
        }

        if (description.length() == 0) {
            description.append(Resources.getString(getClass(), "product-unknown"));
        }

        String productName = coreNode.getStoredString(StringData.INDEX_PRODUCTNAME);
        if (StringUtils.isBlank(productName)) {
            productName = "";
        }

        final String lineTerm = infoArea.getLineTerm();

        final Color foreGround = UIManager.getColor("Text.foreground");

        infoArea.append(foreGround, Resources.getString(getClass(), "description"));
        infoArea.append(foreGround, " : ");
        infoArea
            .append(foreGround,
                description.toString() /* WordUtils.wrap(description.toString(), 80, lineTerm, false) */ );
        infoArea.append(foreGround, lineTerm);
        sb.append(Resources.getString(getClass(), "productname")).append(" : ").append(productName).append(lineTerm);
        sb
            .append(Resources.getString(getClass(), "vendor")).append(" : ")
            .append(DecoderVendorFactory.getDecoderVendorName(vendors, vendorId)).append(lineTerm);
        sb.append("Homepage ..... : ");

        if (StringUtils.isBlank(url)) {
            sb.append("---");
        }
        else {
            infoArea.append(foreGround, sb.toString());
            sb.setLength(0);

            infoArea.appendLink(foreGround, url);
        }
        sb.append(lineTerm);
        sb.append("Unique ID .... : ");
        infoArea.append(foreGround, sb.toString());

        infoArea.append(Color.RED, ByteUtils.convertUniqueIdToString(ByteUtils.convertLongToUniqueId(uniqueId)));
        infoArea.append(foreGround, ", ");
        infoArea
            .append(UIManager.getColor("InfoPanel.uniqueId.formatted.foreground"),
                ByteUtils.getUniqueIdAsString(uniqueId));

        sb.setLength(0);
        sb.append(lineTerm);
        sb.append(Resources.getString(getClass(), "username")).append(" : ");
        infoArea.append(foreGround, sb.toString());
        infoArea.append(Color.GREEN.darker().darker(), BidibNodeNameUtils.getNodeName(node));

        sb.setLength(0);
        sb.append(lineTerm);
        sb
            .append(Resources.getString(getClass(), "address")).append(" : ")
            .append(NodeUtils.formatAddressLong(coreNode.getAddr())).append(lineTerm);
        sb.append("Firmware ..... : ").append(coreNode.getSoftwareVersion()).append(lineTerm);
        sb
            .append(Resources.getString(getClass(), "protocol")).append(" : ").append(coreNode.getProtocolVersion())
            .append(lineTerm);

        // add the features
        List<Feature> features = new LinkedList<>(coreNode.getFeatures());
        sb.append("Feature count &nbsp;: ").append(features.size()).append(lineTerm);
        sb.append("Features ..... : <br/>");

        if (CollectionUtils.isNotEmpty(features)) {
            sb
                .append("<table>\n").append("<tr><th>ID</th><th>").append(Resources.getString(getClass(), "value"))
                .append("</th><th>").append(Resources.getString(getClass(), "unit")).append("</th><th>Name</th></tr>");
            Collections.sort(features);
            for (Feature feature : features) {
                sb.append("<tr><td align=\"right\">");
                sb
                    .append(feature.getType()).append("</td><td align=\"right\">").append(feature.getValue())
                    .append("</td>");
                // sb.append(String.format(" ID %1$3d, Value : %2$3d", feature.getType(), feature.getValue()));
                sb
                    .append("<td align=\"right\">").append(StringUtils.leftPad(getFeatureUnit(feature), 10))
                    .append("</td>");
                // sb.append(" --- ");
                sb.append("<td>").append(feature.getFeatureName()).append(lineTerm);
                sb.append("</td></tr>");
            }
            sb.append("</table>\n");
        }

        infoArea.append(foreGround, sb.toString());

        // try to load image
        try {
            LOGGER.debug("Size of container: {}", imageContainer.getSize());

            ImageIcon icon = loadImage(vendorId, productId, 200, 200);
            imageContainer.setIcon(icon);
        }
        catch (Exception ex) {
            LOGGER.warn("Load image of node failed, vid: {}, pid: {}.", vendorId, productId, ex);
            imageContainer.setIcon(null);
        }

        infoArea.apply();

        // scroll to top
        infoArea.setCaretPosition(0);
    }

    private String getFeatureUnit(Feature feature) {

        if (feature != null && feature.getFeatureEnum() != null) {

            switch (feature.getFeatureEnum()) {
                case FEATURE_BST_AMPERE:
                    // calculate the current value
                    return String.format("%d mA", ConversionUtils.convertCurrent(feature.getValue()));
                default:
                    break;
            }

            FeatureCode featureCode = IterableUtils.find(featureCodes, new Predicate<FeatureCode>() {

                @Override
                public boolean evaluate(FeatureCode featureCode) {
                    return featureCode.getId() == feature.getType();
                }
            });
            if (featureCode != null) {
                String featureUnit = featureCode.getUnit();
                if (StringUtils.isNotEmpty(featureUnit)) {
                    return featureUnit;
                }
            }
        }
        return " ";
    }

    private ImageIcon loadImage(int vendorId, long productId, int width, int height) {

        String labelPath = settingsService.getMiscSettings().getBidibConfigDir();
        String searchPathUser = "/data/images";
        File fileUser = new File(labelPath, searchPathUser);
        searchPathUser = fileUser.getPath();

        ImageIcon icon = null;
        File imageFile = new File(fileUser, "bidib-" + vendorId + "-" + productId + ".png");
        if (imageFile.exists()) {
            icon = ImageUtils.loadImageIcon(imageFile, width, height);
        }

        if (icon == null) {
            icon =
                ImageUtils
                    .loadImageIcon(getClass(), "/images/bidib-" + vendorId + "-" + productId + ".png", width, height);
        }

        return icon;
    }
}
