package org.bidib.wizard.mvc.main.view.panel;

import java.awt.event.ActionEvent;
import java.awt.event.ComponentAdapter;
import java.awt.event.ComponentEvent;
import java.awt.event.KeyEvent;
import java.awt.event.MouseEvent;
import java.util.Collections;
import java.util.LinkedList;
import java.util.List;

import javax.swing.AbstractAction;
import javax.swing.JOptionPane;
import javax.swing.JPanel;
import javax.swing.JTable;
import javax.swing.KeyStroke;
import javax.swing.table.JTableHeader;
import javax.swing.table.TableCellEditor;
import javax.swing.table.TableColumn;
import javax.swing.table.TableModel;

import org.apache.commons.collections4.CollectionUtils;
import org.bidib.jbidibc.messages.enums.LcOutputType;
import org.bidib.jbidibc.messages.utils.ProductUtils;
import org.bidib.wizard.api.locale.Resources;
import org.bidib.wizard.api.model.NodeInterface;
import org.bidib.wizard.api.model.listener.LightPortListener;
import org.bidib.wizard.client.common.controller.NodeSelectionProvider;
import org.bidib.wizard.client.common.dialog.LabelDialog;
import org.bidib.wizard.client.common.model.listener.PortTableLayoutListener;
import org.bidib.wizard.client.common.table.AbstractPortHierarchicalTable;
import org.bidib.wizard.client.common.table.DefaultPortListMenuListener;
import org.bidib.wizard.client.common.view.TabPanelProvider;
import org.bidib.wizard.client.common.view.menu.listener.PortListMenuListener;
import org.bidib.wizard.client.common.view.renderer.DmxChannelRenderer;
import org.bidib.wizard.client.common.view.renderer.PortAwareNumberRenderer;
import org.bidib.wizard.client.common.view.slider.SliderEditor;
import org.bidib.wizard.model.ports.LightPort;
import org.bidib.wizard.model.ports.Port;
import org.bidib.wizard.model.ports.event.PortConfigChangeEvent;
import org.bidib.wizard.model.status.LightPortStatus;
import org.bidib.wizard.mvc.common.view.editor.DmxChannelCellEditor;
import org.bidib.wizard.mvc.common.view.editor.NumberRangeEditor;
import org.bidib.wizard.client.common.view.slider.ConfigXAwareSliderRenderer;
import org.bidib.wizard.mvc.main.controller.LightPortPanelController;
import org.bidib.wizard.mvc.main.model.LightPortTableModel;
import org.bidib.wizard.mvc.main.model.MainModel;
import org.bidib.wizard.mvc.main.model.listener.LightPortModelListener;
import org.bidib.wizard.mvc.main.view.panel.listener.TabComponentCreator;
import org.bidib.wizard.mvc.main.view.panel.listener.TabVisibilityListener;
import org.bidib.wizard.mvc.main.view.panel.listener.TabVisibilityProvider;
import org.bidib.wizard.mvc.main.view.table.LightPortTable;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.jidesoft.grid.SortableTableModel;
import com.jidesoft.grid.TableColumnChooser;

import io.reactivex.rxjava3.subjects.PublishSubject;

public class LightPortListPanel
    extends
    SimpleHierarchicalPortListPanel<LightPortTableModel, LightPortStatus, LightPort, LightPortListener, LightPortModelListener>
    implements PortTableLayoutListener, TabVisibilityProvider, TabPanelProvider, TabComponentCreator {

    private static final Logger LOGGER = LoggerFactory.getLogger(LightPortListPanel.class);

    private static final long serialVersionUID = 1L;

    private final MainModel mainModel;

    private final TabVisibilityListener tabVisibilityListener;

    private final LightPortPanelController controller;

    private boolean isDmxNode = true;

    public LightPortListPanel(final LightPortPanelController controller, final LightPortTableModel tableModel,
        final MainModel mainModel, final TabVisibilityListener tabVisibilityListener,
        final PublishSubject<PortConfigChangeEvent> portConfigChangeEventSubject) {
        super(tableModel, Resources.getString(LightPortListPanel.class, "emptyTable"), portConfigChangeEventSubject, mainModel);

        this.controller = controller;

        this.mainModel = mainModel;
        this.tabVisibilityListener = tabVisibilityListener;
    }

    @Override
    protected AbstractPortHierarchicalTable<LightPort> createPortTable(
        LightPortTableModel tableModel, String emptyTableText) {

        return new LightPortTable(tableModel, emptyTableText);
    }

    @Override
    protected void createTable(
        final LightPortTableModel tableModel, String emptyTableText,
        final PublishSubject<PortConfigChangeEvent> portConfigChangeEventSubject, final NodeSelectionProvider nodeSelectionProvider) {

        super.createTable(tableModel, emptyTableText, portConfigChangeEventSubject, nodeSelectionProvider);

        // force repaint of slider renderers after resize panel
        addComponentListener(new ComponentAdapter() {

            @Override
            public void componentResized(ComponentEvent e) {
                LOGGER.trace("Component is resized: {}", e.getComponent());
                table.repaint();
            }
        });

        // let the left and right key change the slider value
        table
            .getInputMap(JTable.WHEN_ANCESTOR_OF_FOCUSED_COMPONENT)
            .put(KeyStroke.getKeyStroke(KeyEvent.VK_LEFT, 0), "left");
        table.getActionMap().put("left", new AbstractAction() {
            private static final long serialVersionUID = 1L;

            @Override
            public void actionPerformed(ActionEvent e) {
                JTable table = (JTable) e.getSource();
                TableCellEditor editor = table.getCellEditor(table.getSelectedRow(), table.getSelectedColumn());

                if (editor instanceof SliderEditor) {
                    table.editCellAt(table.getSelectedRow(), table.getSelectedColumn());
                    ((SliderEditor) editor).stepDown();
                }
            }
        });
        table
            .getInputMap(JTable.WHEN_ANCESTOR_OF_FOCUSED_COMPONENT)
            .put(KeyStroke.getKeyStroke(KeyEvent.VK_RIGHT, 0), "right");
        table.getActionMap().put("right", new AbstractAction() {
            private static final long serialVersionUID = 1L;

            @Override
            public void actionPerformed(ActionEvent e) {
                JTable table = (JTable) e.getSource();
                TableCellEditor editor = table.getCellEditor(table.getSelectedRow(), table.getSelectedColumn());

                if (editor instanceof SliderEditor) {
                    table.editCellAt(table.getSelectedRow(), table.getSelectedColumn());
                    ((SliderEditor) editor).stepUp();
                }
            }
        });

        // TODO is this still required ?
        // add listener for table layout changes
        tableModel.addPortTableLayoutListener(this);

        final PortListMenuListener portListMenuListener = createPortListMenuListener(this.table);
        table.setPortListMenuListener(portListMenuListener);
    }

    private PortListMenuListener createPortListMenuListener(final AbstractPortHierarchicalTable<LightPort> portTable) {
        final DefaultPortListMenuListener listener = new DefaultPortListMenuListener() {
            @Override
            public void editLabel(final MouseEvent popupEvent) {
                final int row = portTable.getRow(popupEvent.getPoint());
                if (row > -1) {
                    Object val = portTable.getValueAt(row, 0);
                    if (val instanceof Port<?>) {
                        val = ((Port<?>) val).toString();
                    }
                    final Object value = val;
                    if (value instanceof String) {
                        // show the port name editor
                        new LabelDialog((String) value, popupEvent.getXOnScreen(), popupEvent.getYOnScreen()) {
                            @Override
                            public void labelChanged(String label) {
                                portTable.setValueAt(label, row, 0);
                            }
                        };
                    }
                }
                else {
                    LOGGER.warn("The row is not available!");
                }
            }

            @Override
            public void mapPort(final MouseEvent popupEvent) {

                final int row = portTable.getRow(popupEvent.getPoint());
                if (row > -1) {
                    Object val = portTable.getValueAt(row, 0);
                    if (val instanceof LightPort) {
                        LightPort lightPairPort = (LightPort) val;
                        LOGGER.info("Change mapping for port: {}", lightPairPort);

                        // confirm switch to switch port
                        int result =
                            JOptionPane
                                .showConfirmDialog(JOptionPane.getFrameForComponent(LightPortListPanel.this),
                                    Resources.getString(LightPortListPanel.class, "switch-port-confirm"),
                                    Resources.getString(LightPortListPanel.class, "switch-port-title"),
                                    JOptionPane.OK_CANCEL_OPTION, JOptionPane.QUESTION_MESSAGE);
                        if (result == JOptionPane.OK_OPTION) {
                            LOGGER.info("Change the port to an light port.");

                            TableModel tm = table.getModel();
                            if (tm instanceof SortableTableModel) {
                                SortableTableModel stm = (SortableTableModel) tm;
                                tm = stm.getActualModel();
                            }
                            LightPortTableModel lightPortTableModel = (LightPortTableModel) tm;

                            lightPortTableModel.changePortType(LcOutputType.LIGHTPORT, lightPairPort);
                        }
                    }
                }
            }

            @Override
            public void insertPorts() {

                if (portTable.getSelectedRow() > -1) {
                    TableModel tm = table.getModel();
                    if (tm instanceof SortableTableModel) {
                        SortableTableModel stm = (SortableTableModel) tm;
                        tm = stm.getActualModel();
                    }
                    LightPortTableModel lightPortTableModel = (LightPortTableModel) tm;
                    controller.insertPorts(portTable.getSelectedRow(), lightPortTableModel);
                }
            }

        };
        return listener;
    }

    @Override
    public JPanel getComponent() {
        return this;
    }

    @Override
    public Object getCreator() {
        return this;
    }

    @Override
    public boolean equals(Object other) {
        if (other instanceof TabComponentCreator) {
            TabComponentCreator creator = (TabComponentCreator) other;
            // TODO if more than a single instance is available this must be changed
            if (creator.getCreator() instanceof LightPortListPanel) {
                return true;
            }
        }
        return false;
    }

    @Override
    public int hashCode() {
        return super.hashCode();
    }

    @Override
    public void listChanged() {
        LOGGER.info("The port list has changed.");
        super.listChanged();

        tabVisibilityListener.setTabVisible(this, isTabVisible());

        portTableLayoutChanged();
    }

    @Override
    public void portTableLayoutChanged() {

        boolean hasPortIdentifiers = false;

        List<LightPort> ports = new LinkedList<>();
        ports.addAll(getPorts());
        synchronized (ports) {
            for (LightPort port : ports) {
                if (port.isRemappingEnabled()) {
                    hasPortIdentifiers = true;
                    break;
                }
            }
        }

        boolean hasColorValue = false;
        boolean hasTransitionTime = false;
        boolean hasDmxMapping = false;

        if (CollectionUtils.isNotEmpty(ports)) {
            for (LightPort lightPort : ports) {
                if (lightPort.getRgbValue() != null) {
                    hasColorValue = true;
                }
                if (lightPort.getTransitionTime() != null) {
                    hasTransitionTime = true;
                }
                if (lightPort.getDmxMapping() != null) {
                    hasDmxMapping = true;
                }
            }

            // get the max dimm value
            try {
                LightPort lightPort = ports.get(0);
                if (lightPort != null) {
                    int dimmStretchMin = lightPort.getDimStretchMin();
                    int dimmStretchMax = lightPort.getDimStretchMax();
                    LOGGER
                        .info("Set the values for dimmStretchMin: {}, dimmStretchMax: {}", dimmStretchMin,
                            dimmStretchMax);

                    final ConfigXAwareSliderRenderer sliderRendererDimmMin =
                        (ConfigXAwareSliderRenderer) table
                            .getColumnModel().getColumn(LightPortTableModel.COLUMN_DIM_MIN).getCellRenderer();
                    final ConfigXAwareSliderRenderer sliderRendererDimmMax =
                        (ConfigXAwareSliderRenderer) table
                            .getColumnModel().getColumn(LightPortTableModel.COLUMN_DIM_MAX).getCellRenderer();
                    sliderRendererDimmMin.setMaxValue(dimmStretchMin);
                    sliderRendererDimmMax.setMaxValue(dimmStretchMax);
                }
            }
            catch (Exception ex) {
                LOGGER.warn("Get the dimm stretch values failed.", ex);
            }
        }

        LOGGER
            .info("hasColorValue: {}, hasPortIdentifiers: {}, hasTransitionTime: {}, hasDmxMapping: {}", hasColorValue,
                hasPortIdentifiers, hasTransitionTime, hasDmxMapping);

        // keep the column index of the column to insert in the view
        int viewColumnIndex = LightPortTableModel.COLUMN_RGB;

        // show/hide the RGB column
        viewColumnIndex = table.setColumnVisible(LightPortTableModel.COLUMN_RGB, viewColumnIndex, hasColorValue);

        // show/hide the transition time column
        viewColumnIndex =
            table.setColumnVisible(LightPortTableModel.COLUMN_TRANSITION_TIME, viewColumnIndex, hasTransitionTime);

        // show/hide the DMX mapping column
        viewColumnIndex =
            table.setColumnVisible(LightPortTableModel.COLUMN_DMX_MAPPING, viewColumnIndex, hasDmxMapping);

        // show/hide the port identifiers
        table.setColumnVisible(LightPortTableModel.COLUMN_PORT_IDENTIFIER, viewColumnIndex, hasPortIdentifiers);

        // check if the DMX channel column is visible to change the column label
        if (TableColumnChooser.isVisibleColumn(table, LightPortTableModel.COLUMN_DMX_MAPPING)) {
            final NodeInterface node = mainModel.getSelectedNode();
            LOGGER.info("The DMX column is visible for node: {}", node);
            if (node != null) {
                boolean newNodeIsDmxNode = ProductUtils.isOneDMX(node.getUniqueId());
                if (newNodeIsDmxNode != isDmxNode) {
                    LOGGER.info("The DMX support of node has changed, newNodeIsDmxNode: {}", newNodeIsDmxNode);

                    this.isDmxNode = newNodeIsDmxNode;

                    final TableColumn tcDmxMapping =
                        table.getColumnModel().getColumn(LightPortTableModel.COLUMN_DMX_MAPPING);
                    if (newNodeIsDmxNode) {
                        // show the DMX channel renderer and editor
                        tcDmxMapping.setCellRenderer(new DmxChannelRenderer());
                        tcDmxMapping.setCellEditor(new DmxChannelCellEditor(1, 0x100));

                        tcDmxMapping.setHeaderValue(Resources.getString(LightPortTableModel.class, "dmxMapping"));
                    }
                    else {
                        // show the channel renderer and editor
                        tcDmxMapping.setCellRenderer(new PortAwareNumberRenderer());
                        tcDmxMapping.setCellEditor(new NumberRangeEditor(0, 0xFF));

                        tcDmxMapping.setHeaderValue(Resources.getString(LightPortTableModel.class, "channel"));
                    }

                    // repaint the table header to make the new label appear
                    JTableHeader th = table.getTableHeader();
                    th.repaint();
                }
                else {
                    LOGGER.info("The DMX column is visible but the isDmxNode property has not changed: {}", isDmxNode);
                }
            }
        }
        else {
            LOGGER.info("The DMX column is not visible.");
        }

    }

    @Override
    public Class<?> getPortClass() {
        return LightPort.class;
    }

    @Override
    public boolean isTabVisible() {
        final NodeInterface node = mainModel.getSelectedNode();
        if (node != null) {
            boolean isTabVisible = node.hasLightPorts();
            LOGGER.debug("Check if tab is visible: {}", isTabVisible);
            return isTabVisible;
        }
        return false;
    }

    @Override
    protected List<LightPort> getPorts() {
        final NodeInterface node = mainModel.getSelectedNode();
        if (node != null) {
            return node.getLightPorts();
        }
        return Collections.emptyList();
    }

    public void refreshView() {
        listChanged();
    }
}
