package org.bidib.wizard.mvc.main.view.panel;

import java.awt.event.MouseEvent;
import java.util.Collections;
import java.util.LinkedList;
import java.util.List;

import javax.swing.JOptionPane;
import javax.swing.JPanel;
import javax.swing.table.TableModel;

import org.apache.commons.collections4.CollectionUtils;
import org.bidib.jbidibc.messages.BidibLibrary;
import org.bidib.jbidibc.messages.Feature;
import org.bidib.jbidibc.messages.enums.LcOutputType;
import org.bidib.wizard.api.locale.Resources;
import org.bidib.wizard.api.model.NodeInterface;
import org.bidib.wizard.api.model.listener.SwitchPairPortListener;
import org.bidib.wizard.client.common.controller.NodeSelectionProvider;
import org.bidib.wizard.client.common.dialog.LabelDialog;
import org.bidib.wizard.client.common.table.AbstractPortHierarchicalTable;
import org.bidib.wizard.client.common.table.DefaultPortListMenuListener;
import org.bidib.wizard.client.common.view.TabPanelProvider;
import org.bidib.wizard.client.common.view.menu.listener.PortListMenuListener;
import org.bidib.wizard.model.ports.Port;
import org.bidib.wizard.model.ports.SwitchPairPort;
import org.bidib.wizard.model.ports.event.PortConfigChangeEvent;
import org.bidib.wizard.model.status.SwitchPortStatus;
import org.bidib.wizard.mvc.main.controller.SwitchPairPortPanelController;
import org.bidib.wizard.mvc.main.model.MainModel;
import org.bidib.wizard.mvc.main.model.SwitchPairPortTableModel;
import org.bidib.wizard.mvc.main.model.listener.SwitchPairPortModelListener;
import org.bidib.wizard.mvc.main.view.panel.listener.TabComponentCreator;
import org.bidib.wizard.mvc.main.view.panel.listener.TabVisibilityListener;
import org.bidib.wizard.mvc.main.view.panel.listener.TabVisibilityProvider;
import org.bidib.wizard.mvc.main.view.table.SwitchPairPortTable;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.jidesoft.grid.SortableTableModel;

import io.reactivex.rxjava3.subjects.PublishSubject;

public class SwitchPairPortListPanel
    extends
    SimpleHierarchicalPortListPanel<SwitchPairPortTableModel, SwitchPortStatus, SwitchPairPort, SwitchPairPortListener, SwitchPairPortModelListener>
    implements TabVisibilityProvider, TabPanelProvider, TabComponentCreator {
    private static final Logger LOGGER = LoggerFactory.getLogger(SwitchPairPortListPanel.class);

    private static final long serialVersionUID = 1L;

    private final MainModel mainModel;

    private final TabVisibilityListener tabVisibilityListener;

    public SwitchPairPortListPanel(final SwitchPairPortPanelController controller,
        final SwitchPairPortTableModel tableModel, final MainModel mainModel,
        final TabVisibilityListener tabVisibilityListener,
        final PublishSubject<PortConfigChangeEvent> portConfigChangeEventSubject) {
        super(tableModel, Resources.getString(SwitchPairPortListPanel.class, "emptyTable"),
            portConfigChangeEventSubject, mainModel);
        this.mainModel = mainModel;
        this.tabVisibilityListener = tabVisibilityListener;
    }

    @Override
    protected AbstractPortHierarchicalTable<SwitchPairPort> createPortTable(
        SwitchPairPortTableModel tableModel, String emptyTableText) {

        return new SwitchPairPortTable(tableModel, emptyTableText);
    }

    @Override
    protected void createTable(
        final SwitchPairPortTableModel tableModel, String emptyTableText,
        final PublishSubject<PortConfigChangeEvent> portConfigChangeEventSubject, final NodeSelectionProvider nodeSelectionProvider) {

        super.createTable(tableModel, emptyTableText, portConfigChangeEventSubject, nodeSelectionProvider);

        final PortListMenuListener portListMenuListener = createPortListMenuListener(this.table);
        table.setPortListMenuListener(portListMenuListener);
    }

    private PortListMenuListener createPortListMenuListener(
        final AbstractPortHierarchicalTable<SwitchPairPort> portTable) {
        final DefaultPortListMenuListener listener = new DefaultPortListMenuListener() {
            @Override
            public void editLabel(final MouseEvent popupEvent) {
                final int row = portTable.getRow(popupEvent.getPoint());
                if (row > -1) {
                    Object val = portTable.getValueAt(row, 0);
                    if (val instanceof Port<?>) {
                        val = ((Port<?>) val).toString();
                    }
                    final Object value = val;
                    if (value instanceof String) {
                        // show the port name editor
                        new LabelDialog((String) value, popupEvent.getXOnScreen(), popupEvent.getYOnScreen()) {
                            @Override
                            public void labelChanged(String label) {
                                portTable.setValueAt(label, row, 0);
                            }
                        };
                    }
                }
                else {
                    LOGGER.warn("The row is not available!");
                }
            }

            @Override
            public void mapPort(final MouseEvent popupEvent) {

                final int row = portTable.getRow(popupEvent.getPoint());
                if (row > -1) {
                    Object val = portTable.getValueAt(row, 0);
                    if (val instanceof SwitchPairPort) {
                        SwitchPairPort switchPairPort = (SwitchPairPort) val;
                        LOGGER.info("Change mapping for port: {}", switchPairPort);

                        // confirm switch to switch port
                        int result =
                            JOptionPane
                                .showConfirmDialog(JOptionPane.getFrameForComponent(SwitchPairPortListPanel.this),
                                    Resources.getString(SwitchPairPortListPanel.class, "switch-port-confirm"),
                                    Resources.getString(SwitchPairPortListPanel.class, "switch-port-title"),
                                    JOptionPane.OK_CANCEL_OPTION, JOptionPane.QUESTION_MESSAGE);
                        if (result == JOptionPane.OK_OPTION) {
                            LOGGER.info("Change the port to an switchPair port.");

                            TableModel tm = table.getModel();
                            if (tm instanceof SortableTableModel) {
                                SortableTableModel stm = (SortableTableModel) tm;
                                tm = stm.getActualModel();
                            }
                            SwitchPairPortTableModel switchPairPortTableModel = (SwitchPairPortTableModel) tm;

                            switchPairPortTableModel.changePortType(LcOutputType.SWITCHPAIRPORT, switchPairPort);
                        }
                    }
                }
            }
        };
        return listener;
    }

    @Override
    public JPanel getComponent() {
        return this;
    }

    @Override
    public Object getCreator() {
        return this;
    }

    @Override
    public boolean equals(Object other) {
        if (other instanceof TabComponentCreator) {
            TabComponentCreator creator = (TabComponentCreator) other;
            // TODO if more than a single instance is available this must be changed
            if (creator.getCreator() instanceof SwitchPairPortListPanel) {
                return true;
            }
        }
        return false;
    }

    @Override
    public int hashCode() {
        return super.hashCode();
    }

    @Override
    public void listChanged() {
        LOGGER.info("The port list has changed.");

        super.listChanged();

        boolean hasPortIdentifiers = false;

        List<SwitchPairPort> ports = new LinkedList<>();
        ports.addAll(getPorts());
        synchronized (ports) {
            for (SwitchPairPort port : ports) {
                if (port.isRemappingEnabled()) {
                    hasPortIdentifiers = true;
                    break;
                }
            }
        }

        final NodeInterface node = mainModel.getSelectedNode();
        if (node != null) {
            LOGGER.info("A node is selected.");

            boolean hasSwitchPortConfigTicks = false;
            boolean hasSwitchPortConfigLoadType = false;

            if (node.getNode().isPortFlatModelAvailable() && CollectionUtils.isNotEmpty(node.getSwitchPairPorts())) {
                LOGGER.info("Check if at least one switch port has the switchPair port config available.");
                for (SwitchPairPort port : node.getSwitchPairPorts()) {
                    if (!hasSwitchPortConfigTicks) {
                        hasSwitchPortConfigTicks =
                            port.isPortConfigKeySupported(Byte.valueOf(BidibLibrary.BIDIB_PCFG_TICKS));
                    }
                    if (!hasSwitchPortConfigLoadType) {
                        hasSwitchPortConfigLoadType =
                            port.isPortConfigKeySupported(Byte.valueOf(BidibLibrary.BIDIB_PCFG_LOAD_TYPE));
                    }

                    if (hasSwitchPortConfigTicks && hasSwitchPortConfigLoadType) {
                        // found one -> show the column
                        break;
                    }
                }
            }
            else {
                Feature switchPortConfigAvailable =
                    Feature.findFeature(node.getNode().getFeatures(), BidibLibrary.FEATURE_SWITCH_CONFIG_AVAILABLE);
                if (switchPortConfigAvailable != null) {
                    hasSwitchPortConfigTicks = (switchPortConfigAvailable.getValue() > 0);
                }

                for (SwitchPairPort port : node.getEnabledSwitchPairPorts()) {
                    if (!hasSwitchPortConfigTicks) {
                        hasSwitchPortConfigTicks =
                            port.isPortConfigKeySupported(Byte.valueOf(BidibLibrary.BIDIB_PCFG_TICKS));
                    }
                    if (!hasSwitchPortConfigLoadType) {
                        hasSwitchPortConfigLoadType =
                            port.isPortConfigKeySupported(Byte.valueOf(BidibLibrary.BIDIB_PCFG_LOAD_TYPE));
                    }

                    if (hasSwitchPortConfigTicks && hasSwitchPortConfigLoadType) {
                        // found one -> show the column
                        break;
                    }
                }
            }
            LOGGER
                .info(
                    "List has changed, hasPortIdentifiers: {}, hasSwitchPortConfigTicks: {}, hasSwitchPortConfigLoadType: {}",
                    hasPortIdentifiers, hasSwitchPortConfigTicks, hasSwitchPortConfigLoadType);

            // keep the column index of the column to insert in the view
            int viewColumnIndex = SwitchPairPortTableModel.COLUMN_SWITCH_OFF_TIME;

            // show/hide the ticks column
            viewColumnIndex =
                table
                    .setColumnVisible(SwitchPairPortTableModel.COLUMN_SWITCH_OFF_TIME, viewColumnIndex,
                        hasSwitchPortConfigTicks);

            // show/hide the load type column
            viewColumnIndex =
                table
                    .setColumnVisible(SwitchPairPortTableModel.COLUMN_LOAD_TYPE, viewColumnIndex,
                        hasSwitchPortConfigLoadType);

            // show/hide the port identifier column
            viewColumnIndex =
                table
                    .setColumnVisible(SwitchPairPortTableModel.COLUMN_PORT_IDENTIFIER, viewColumnIndex,
                        hasPortIdentifiers);
        }

        tabVisibilityListener.setTabVisible(this, isTabVisible());
    }

    @Override
    public Class<?> getPortClass() {
        return SwitchPairPort.class;
    }

    @Override
    public boolean isTabVisible() {
        final NodeInterface node = mainModel.getSelectedNode();
        if (node != null) {
            boolean isTabVisible = node.hasSwitchPairPorts();
            LOGGER.debug("Check if tab is visible: {}", isTabVisible);
            return isTabVisible;
        }
        return false;
    }

    @Override
    protected List<SwitchPairPort> getPorts() {
        final NodeInterface node = mainModel.getSelectedNode();
        if (node != null) {
            List<SwitchPairPort> ports = new LinkedList<>();
            ports.addAll(node.getSwitchPairPorts());

            return ports;
        }
        return Collections.emptyList();
    }

    public void refreshView() {
        listChanged();
    }
}
