package org.bidib.wizard.mvc.main.view.table;

import java.beans.PropertyChangeEvent;
import java.util.concurrent.TimeUnit;
import java.util.function.Consumer;

import javax.swing.JButton;
import javax.swing.JComboBox;
import javax.swing.JComponent;
import javax.swing.JLabel;
import javax.swing.JPanel;
import javax.swing.JTextField;
import javax.swing.ListModel;
import javax.swing.SwingUtilities;

import org.bidib.jbidibc.messages.BidibLibrary;
import org.bidib.jbidibc.messages.enums.IoBehaviourSwitchEnum;
import org.bidib.jbidibc.messages.enums.LoadTypeEnum;
import org.bidib.jbidibc.messages.enums.PortConfigKeys;
import org.bidib.wizard.api.locale.Resources;
import org.bidib.wizard.api.utils.PortUtils;
import org.bidib.wizard.client.common.controller.NodeSelectionProvider;
import org.bidib.wizard.client.common.converter.StringToIntegerConverter;
import org.bidib.wizard.client.common.rxjava2.SwingScheduler;
import org.bidib.wizard.client.common.table.AbstractPortEditorPanel;
import org.bidib.wizard.client.common.text.InputValidationDocument;
import org.bidib.wizard.client.common.text.WizardComponentFactory;
import org.bidib.wizard.client.common.view.validation.IconFeedbackPanel;
import org.bidib.wizard.client.common.view.validation.PropertyValidationI18NSupport;
import org.bidib.wizard.model.ports.GenericPort;
import org.bidib.wizard.model.ports.ServoPort;
import org.bidib.wizard.model.ports.SwitchPort;
import org.bidib.wizard.model.status.SwitchPortStatus;
import org.bidib.wizard.client.common.view.renderer.BidibStatusListRenderer;
import org.bidib.wizard.client.common.view.validation.DefaultRangeValidationCallback;
import org.bidib.wizard.client.common.view.validation.IntegerInputValidationDocument;
import org.bidib.wizard.mvc.main.model.SwitchPortTableModel;
import org.bidib.wizard.mvc.main.view.panel.renderer.IoBehaviourSwitchCellRenderer;
import org.bidib.wizard.mvc.main.view.panel.renderer.LoadTypeCellRenderer;
import org.bidib.wizard.mvc.main.view.table.converter.PortStatusToStringConverter;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.jgoodies.binding.adapter.Bindings;
import com.jgoodies.binding.adapter.ComboBoxAdapter;
import com.jgoodies.binding.list.SelectionInList;
import com.jgoodies.binding.value.BufferedValueModel;
import com.jgoodies.binding.value.ConverterValueModel;
import com.jgoodies.binding.value.ValueHolder;
import com.jgoodies.binding.value.ValueModel;
import com.jgoodies.common.collect.ArrayListModel;
import com.jgoodies.forms.builder.FormBuilder;
import com.jgoodies.forms.factories.Paddings;
import com.jgoodies.validation.ValidationResult;
import com.jgoodies.validation.util.PropertyValidationSupport;
import com.jgoodies.validation.view.ValidationComponentUtils;

import io.reactivex.rxjava3.disposables.Disposable;
import io.reactivex.rxjava3.subjects.PublishSubject;

public class SwitchPortEditorPanel extends AbstractPortEditorPanel<SwitchPort> {

    private static final Logger LOGGER = LoggerFactory.getLogger(SwitchPortEditorPanel.class);

    private static final String ENCODED_DIALOG_COLUMN_SPECS =
        "pref, 3dlu, pref, 3dlu, pref, 3dlu, pref, 3dlu, fill:50dlu:grow";

    private static final String ENCODED_DIALOG_ROW_SPECS =
        "pref, 3dlu, pref, 3dlu, pref, 3dlu, pref, 3dlu, pref, 3dlu, pref, 3dlu, pref";

    private static final int TIME_BETWEEN_STATUS_EVENTS_MILLIS = 80;

    private ValueModel switchOffTimeConverterModel;

    private ValueModel selectionHolderIoBehaviour;

    private ValueModel selectionHolderLoadType;

    private JTextField portName;

    private JTextField switchOffTimeText;

    private JComboBox<IoBehaviourSwitchEnum> comboIoBehaviour;

    private JComboBox<LoadTypeEnum> comboLoadType;

    private JComboBox<SwitchPortStatus> comboPortStatus;

    private JButton btnPortStatus;

    private PublishSubject<SwitchPortStatus> statusEventSubject = PublishSubject.create();

    public SwitchPortEditorPanel(SwitchPort port, Consumer<SwitchPort> saveCallback,
                                 final Consumer<SwitchPort> valueCallback, final Consumer<SwitchPort> refreshCallback, final NodeSelectionProvider nodeSelectionProvider) {
        super(port, saveCallback, valueCallback, refreshCallback, nodeSelectionProvider);
    }

    @Override
    protected SwitchPort clonePort(final SwitchPort port) {
        // create a clone of the input port
        final SwitchPort clone =
            SwitchPort
                .builder() //
                .withOutputBehaviour(port.getOutputBehaviour()) //
                .withLoadType(port.getLoadType()) //
                .withSwitchOffTime(port.getSwitchOffTime()) //
                .withStatus(port.getStatus()) //
                .withRemappingEnabled(port.isRemappingEnabled()) //
                .withKnownPortConfigKeys(port.getKnownPortConfigKeys()) //
                .withId(port.getId()) //
                .withLabel(port.getLabel()) //
                .withEnabled(port.isEnabled()) //
                .withIsInactive(port.isInactive()) //
                .withPortIdentifier(port.getPortIdentifier()).build();
        return clone;
    }

    @Override
    protected JPanel doCreateComponent(final SwitchPort port) {

        int row = 1;
        FormBuilder dialogBuilder = null;
        boolean debugDialog = false;
        if (debugDialog) {
            JPanel panel = new PortEditorPanelDebugContainer(this);
            dialogBuilder =
                FormBuilder.create().columns(ENCODED_DIALOG_COLUMN_SPECS).rows(ENCODED_DIALOG_ROW_SPECS).panel(panel);
        }
        else {
            JPanel panel = new PortEditorPanelContainer(this);
            dialogBuilder =
                FormBuilder.create().columns(ENCODED_DIALOG_COLUMN_SPECS).rows(ENCODED_DIALOG_ROW_SPECS).panel(panel);
        }
        dialogBuilder.border(Paddings.TABBED_DIALOG);

        // port name
        final BufferedValueModel bufferedPortNameModel =
            getPresentationModel().getBufferedModel(SwitchPort.PROPERTY_LABEL);

        dialogBuilder.add(Resources.getString(SwitchPortTableModel.class, "label") + ":").xy(1, row);
        portName = WizardComponentFactory.createTextField(bufferedPortNameModel, false);
        portName.setEnabled(port.isEnabled());
        dialogBuilder.add(portName).xyw(3, row, 7);

        row += 2;

        // switchoff time
        this.switchOffTimeConverterModel =
            new ConverterValueModel(getPresentationModel().getBufferedModel(SwitchPort.PROPERTYNAME_SWITCHOFFTIME),
                new StringToIntegerConverter());

        // if switchOff time is not available we must not show it here
        if (isPortConfigKeySupported(BidibLibrary.BIDIB_PCFG_TICKS)) {
            switchOffTimeText = new JTextField();
            final IntegerInputValidationDocument switchOffTimeDocument =
                new IntegerInputValidationDocument(5, InputValidationDocument.NUMERIC);
            switchOffTimeDocument.setRangeValidationCallback(new DefaultRangeValidationCallback(0, 255));
            switchOffTimeText.setDocument(switchOffTimeDocument);
            switchOffTimeText.setColumns(2);
            switchOffTimeText.setHorizontalAlignment(JTextField.RIGHT);

            // bind manually because we changed the document of the textfield
            Bindings.bind(switchOffTimeText, switchOffTimeConverterModel, false);

            dialogBuilder.add(Resources.getString(SwitchPortTableModel.class, "switchOffTime")).xy(1, row);
            dialogBuilder.add(switchOffTimeText).xy(3, row);

            switchOffTimeText.setEnabled(port.isEnabled());

            ValidationComponentUtils.setMandatory(switchOffTimeText, true);
            ValidationComponentUtils.setMessageKeys(switchOffTimeText, "validation.switchOffTime_key");

            row += 2;
        }

        // ioBehaviour
        this.selectionHolderIoBehaviour =
            getPresentationModel().getBufferedModel(SwitchPort.PROPERTYNAME_OUTPUTBEHAVIOUR);

        // if IoBehaviour is not available we must not show it here
        if (isPortConfigKeySupported(BidibLibrary.BIDIB_PCFG_SWITCH_CTRL)) {
            final ArrayListModel<IoBehaviourSwitchEnum> ioBehaviourList = new ArrayListModel<>();
            for (IoBehaviourSwitchEnum value : IoBehaviourSwitchEnum.values()) {
                ioBehaviourList.add(value);
            }

            final SelectionInList<IoBehaviourSwitchEnum> ioBehaviourSelection =
                new SelectionInList<>((ListModel<IoBehaviourSwitchEnum>) ioBehaviourList);

            final ComboBoxAdapter<IoBehaviourSwitchEnum> comboBoxAdapterIoBehaviour =
                new ComboBoxAdapter<IoBehaviourSwitchEnum>(ioBehaviourSelection, selectionHolderIoBehaviour);
            this.comboIoBehaviour = new JComboBox<>();
            this.comboIoBehaviour.setModel(comboBoxAdapterIoBehaviour);
            this.comboIoBehaviour.setRenderer(new IoBehaviourSwitchCellRenderer());

            this.comboIoBehaviour.setEnabled(port.isEnabled());

            dialogBuilder.add(Resources.getString(SwitchPortTableModel.class, "ioBehaviour") + ":").xy(1, row);
            dialogBuilder.add(this.comboIoBehaviour).xy(3, row);

            row += 2;
        }

        // loadType
        this.selectionHolderLoadType = getPresentationModel().getBufferedModel(SwitchPort.PROPERTYNAME_LOADTYPE);

        // if loadType time is not available we must not show it here
        if (isPortConfigKeySupported(BidibLibrary.BIDIB_PCFG_LOAD_TYPE)) {
            final ArrayListModel<LoadTypeEnum> loadTypeList = new ArrayListModel<>();
            for (LoadTypeEnum value : LoadTypeEnum.values()) {
                loadTypeList.add(value);
            }

            final SelectionInList<LoadTypeEnum> loadTypeSelection =
                new SelectionInList<>((ListModel<LoadTypeEnum>) loadTypeList);

            final ComboBoxAdapter<LoadTypeEnum> comboBoxAdapterLoadType =
                new ComboBoxAdapter<LoadTypeEnum>(loadTypeSelection, selectionHolderLoadType);
            this.comboLoadType = new JComboBox<>();
            this.comboLoadType.setModel(comboBoxAdapterLoadType);
            this.comboLoadType.setRenderer(new LoadTypeCellRenderer());

            this.comboLoadType.setEnabled(port.isEnabled());

            dialogBuilder.add(Resources.getString(SwitchPortTableModel.class, "loadType") + ":").xy(1, row);
            dialogBuilder.add(this.comboLoadType).xy(3, row);

            row += 2;
        }

        // change the status
        final ValueModel selectionHolderPortStatus = getPresentationModel().getModel(SwitchPort.PROPERTY_STATUS);

        final ConverterValueModel portStatusConverterModel =
            new ConverterValueModel(selectionHolderPortStatus, new PortStatusToStringConverter(SwitchPortStatus.class));

        final JLabel statusLabel = WizardComponentFactory.createLabel(portStatusConverterModel);

        final SelectionInList<SwitchPortStatus> portStatusSelection =
            new SelectionInList<>(SwitchPortStatus.ON.getValues());

        final ValueModel portStatusHolder = new ValueHolder(selectionHolderPortStatus.getValue());
        ComboBoxAdapter<SwitchPortStatus> comboBoxAdapterPortStatus =
            new ComboBoxAdapter<>(portStatusSelection, portStatusHolder);
        this.comboPortStatus = new JComboBox<>();
        this.comboPortStatus.setRenderer(new BidibStatusListRenderer<>(SwitchPortStatus.class));
        this.comboPortStatus.setModel(comboBoxAdapterPortStatus);

        SwitchPortStatus oppositeStatus =
            PortUtils.getOppositeStatus((SwitchPortStatus) selectionHolderPortStatus.getValue());
        LOGGER.debug("Set the opposite status in the port status selection: {}", oppositeStatus);
        this.comboPortStatus.setSelectedItem(oppositeStatus);

        this.btnPortStatus = new JButton(Resources.getString(SwitchPortTableModel.class, "test"));

        this.btnPortStatus.addActionListener(evt -> {
            SwitchPortStatus portStatus = (SwitchPortStatus) portStatusHolder.getValue();
            LOGGER.info("The port status has been changed: {}", portStatus);

            statusEventSubject.onNext(portStatus);
        });

        final Disposable disp =
            this.statusEventSubject
                .throttleLatest(TIME_BETWEEN_STATUS_EVENTS_MILLIS, TimeUnit.MILLISECONDS, SwingScheduler.getInstance(),
                    true)
                .subscribe(portStatus -> sendStatusToPort(port, portStatus));
        getCompDisp().add(disp);

        dialogBuilder.add(Resources.getString(SwitchPortTableModel.class, "status") + ":").xy(1, row);
        dialogBuilder.add(statusLabel).xy(3, row);
        dialogBuilder.add(this.comboPortStatus).xy(5, row);
        dialogBuilder.add(this.btnPortStatus).xy(7, row);

        this.comboPortStatus.setEnabled(port.isEnabled());
        this.btnPortStatus.setEnabled(port.isEnabled());

        row += 2;

        // add buttons
        final JPanel buttonPanel = createButtonPanel();
        dialogBuilder.add(buttonPanel).xyw(1, row, 5);

        // check if we have validation enabled
        if (getValidationResultModel() != null) {
            LOGGER.debug("Create iconfeedback panel.");
            JComponent cvIconPanel = new IconFeedbackPanel(getValidationResultModel(), dialogBuilder.build());
            JPanel panel = new PortEditorPanelContainer(this);
            FormBuilder feedbackBuilder = FormBuilder.create().columns("p:g").rows("fill:p:grow").panel(panel);

            feedbackBuilder.add(cvIconPanel).xy(1, 1);

            setPanel(feedbackBuilder.build());
        }
        else {
            setPanel(dialogBuilder.build());
        }

        bufferedPortNameModel.addValueChangeListener(evt -> triggerValidation());
        this.switchOffTimeConverterModel.addValueChangeListener(evt -> triggerValidation());
        this.selectionHolderIoBehaviour.addValueChangeListener(evt -> triggerValidation());
        this.selectionHolderLoadType.addValueChangeListener(evt -> triggerValidation());

        enableComponents();

        triggerValidation();

        return getPanel();
    }

    // @Override
    // protected PropertyChangeListener createPortPropertyChangeListener() {
    // final PropertyChangeListener portListener = new PropertyChangeListener() {
    //
    // @Override
    // public void propertyChange(final PropertyChangeEvent evt) {
    // LOGGER
    // .info("The port property has been changed, propertyName: {}, new value: {}", evt.getPropertyName(),
    // evt.getNewValue());
    //
    // SwingUtilities.invokeLater(() -> {
    // try {
    // // the status of the port was changed
    // if (evt.getPropertyName().equals(LightPort.PROPERTY_STATUS)) {
    // LOGGER.info("The port status has changed: {}", evt.getNewValue());
    // final ValueModel selectionHolderPortStatus =
    // getPresentationModel().getModel(LightPort.PROPERTY_STATUS);
    // selectionHolderPortStatus.setValue(evt.getNewValue());
    //
    // if (SwitchPortEditorPanel.this.comboPortStatus != null) {
    // SwitchPortStatus oppositeStatus =
    // PortUtils
    // .getOppositeStatus((SwitchPortStatus) selectionHolderPortStatus.getValue());
    // LOGGER.info("Set the opposite status in the port status selection: {}", oppositeStatus);
    // SwitchPortEditorPanel.this.comboPortStatus.setSelectedItem(oppositeStatus);
    // }
    // }
    // }
    // catch (Exception ex) {
    // LOGGER.warn("Update the status failed.", ex);
    // }
    // });
    //
    // }
    // };
    // return portListener;
    // }

    @Override
    protected void propertyChanged(final PropertyChangeEvent evt) {
        LOGGER
            .info("The port property has been changed, propertyName: {}, new value: {}", evt.getPropertyName(),
                evt.getNewValue());

        super.propertyChanged(evt);

        SwingUtilities.invokeLater(() -> {
            try {
                // the status of the port was changed
                switch (evt.getPropertyName()) {
                    case SwitchPort.PROPERTY_STATUS:
                    case GenericPort.PROPERTY_PORT_STATUS:
                        final ValueModel selectionHolderPortStatus =
                            getPresentationModel().getModel(SwitchPort.PROPERTY_STATUS);
                        SwitchPortStatus status = getOriginalPort().getStatus();
                        LOGGER.info("Current status of original port: {}", status);
                        selectionHolderPortStatus.setValue(status);

                        if (SwitchPortEditorPanel.this.comboPortStatus != null) {
                            SwitchPortStatus oppositeStatus =
                                PortUtils.getOppositeStatus((SwitchPortStatus) selectionHolderPortStatus.getValue());
                            LOGGER.info("Set the opposite status in the port status selection: {}", oppositeStatus);
                            SwitchPortEditorPanel.this.comboPortStatus.setSelectedItem(oppositeStatus);
                        }
                        break;
                    case GenericPort.PROPERTY_PORT_CONFIG_CHANGED:
                        break;
                    case GenericPort.PROPERTY_PORT_TYPE_CHANGED:
                        LOGGER.info("The port type has changed: {}", evt.getNewValue());
                        enableComponents();
                        break;

                    default:
                        break;
                }
            }
            catch (Exception ex) {
                LOGGER.warn("Update the status failed.", ex);
            }
        });
    }

    @Override
    protected ValidationResult validate(final SwitchPort port) {
        PropertyValidationSupport support = new PropertyValidationI18NSupport(getPresentationModel(), "validation");

        if (this.switchOffTimeConverterModel.getValue() == null && port.isPortConfigKeySupported(PortConfigKeys.BIDIB_PCFG_TICKS)) {
            support.addError("switchOffTime_key", "not_empty_for_write");
        }

        if (this.selectionHolderIoBehaviour.getValue() == null && port.isPortConfigKeySupported(PortConfigKeys.BIDIB_PCFG_SWITCH_CTRL)) {
            support.addError("ioBehaviour_key", "not_empty_for_write");
        }

        if (this.selectionHolderLoadType.getValue() == null && port.isPortConfigKeySupported(PortConfigKeys.BIDIB_PCFG_LOAD_TYPE)) {
            support.addError("loadType_key", "not_empty_for_write");
        }

        ValidationResult validationResult = support.getResult();
        LOGGER.info("Prepared validationResult: {}", validationResult);
        return validationResult;
    }

    private void sendStatusToPort(final SwitchPort port, SwitchPortStatus portStatus) {
        LOGGER.info("Send the new portStatus to the port: {}", portStatus);

        triggerValidation();

        final SwitchPort switchPort = SwitchPort.builder().withStatus(portStatus).withId(port.getId()).build();

        getValueCallback().accept(switchPort);
    }

    @Override
    protected void doEnableComponents(final SwitchPort port) {

        boolean enabled = port.isEnabled();
        this.portName.setEnabled(enabled);

        if (this.comboIoBehaviour != null) {
            this.comboIoBehaviour
                .setEnabled(enabled && port.isPortConfigKeySupported(PortConfigKeys.BIDIB_PCFG_SWITCH_CTRL));
        }

        if (this.comboLoadType != null) {
            this.comboLoadType
                .setEnabled(enabled && port.isPortConfigKeySupported(PortConfigKeys.BIDIB_PCFG_LOAD_TYPE));
        }

        if (switchOffTimeText != null) {
            switchOffTimeText.setEnabled(enabled && port.isPortConfigKeySupported(PortConfigKeys.BIDIB_PCFG_TICKS));
        }

        this.comboPortStatus.setEnabled(enabled);
        this.btnPortStatus.setEnabled(enabled);
    }
}
