package org.bidib.wizard.mvc.firmware.controller;

import java.beans.PropertyChangeEvent;
import java.beans.PropertyChangeListener;
import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.io.File;
import java.io.FileOutputStream;
import java.nio.charset.StandardCharsets;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.LinkedList;
import java.util.List;
import java.util.concurrent.Executors;
import java.util.concurrent.ScheduledExecutorService;

import javax.swing.JFrame;
import javax.swing.SwingUtilities;

import org.apache.commons.collections4.CollectionUtils;
import org.apache.commons.io.IOUtils;
import org.apache.commons.lang3.StringUtils;
import org.bidib.jbidibc.core.schema.bidiblabels.NodeLabels;
import org.bidib.jbidibc.exchange.bidib.FirmwareFactory;
import org.bidib.jbidibc.exchange.firmware.FilenameType;
import org.bidib.jbidibc.exchange.firmware.FirmwareNode;
import org.bidib.jbidibc.exchange.vendorcv.VendorCV;
import org.bidib.jbidibc.exchange.vendorcv.VendorCvFactory;
import org.bidib.jbidibc.messages.FirmwareUpdateStat;
import org.bidib.jbidibc.messages.Node;
import org.bidib.jbidibc.messages.SoftwareVersion;
import org.bidib.jbidibc.messages.StringData;
import org.bidib.jbidibc.messages.enums.FirmwareUpdateOperation;
import org.bidib.jbidibc.messages.enums.FirmwareUpdateState;
import org.bidib.jbidibc.messages.exception.NoAnswerException;
import org.bidib.jbidibc.messages.helpers.Context;
import org.bidib.jbidibc.messages.helpers.DefaultContext;
import org.bidib.jbidibc.messages.utils.ByteUtils;
import org.bidib.jbidibc.messages.utils.NodeUtils;
import org.bidib.jbidibc.messages.utils.ProductUtils;
import org.bidib.jbidibc.messages.utils.ThreadFactoryBuilder;
import org.bidib.wizard.api.locale.Resources;
import org.bidib.wizard.api.model.NodeInterface;
import org.bidib.wizard.api.model.NodeListProvider;
import org.bidib.wizard.api.model.connection.BidibConnection;
import org.bidib.wizard.api.model.listener.DefaultNodeListListener;
import org.bidib.wizard.api.model.listener.NodeListListener;
import org.bidib.wizard.api.service.console.ConsoleColor;
import org.bidib.wizard.api.service.console.ConsoleService;
import org.bidib.wizard.api.service.node.CommandStationService;
import org.bidib.wizard.core.dialog.FileDialog;
import org.bidib.wizard.core.labels.DefaultWizardLabelFactory;
import org.bidib.wizard.core.model.connection.ConnectionRegistry;
import org.bidib.wizard.core.model.firmware.FirmwareUpdatePart;
import org.bidib.wizard.core.model.firmware.UpdateStatus;
import org.bidib.wizard.core.service.SettingsService;
import org.bidib.wizard.mvc.console.controller.ConsoleController;
import org.bidib.wizard.mvc.firmware.controller.listener.FirmwareControllerListener;
import org.bidib.wizard.mvc.firmware.model.FirmwareModel;
import org.bidib.wizard.mvc.firmware.view.FirmwareView;
import org.bidib.wizard.mvc.firmware.view.listener.FirmwareViewListener;
import org.bidib.wizard.utils.FirmwareUtils;
import org.bidib.wizard.utils.WindowUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;

public class FirmwareController {
    private static final Logger LOGGER = LoggerFactory.getLogger(FirmwareController.class);

    private final Collection<FirmwareControllerListener> listeners = new LinkedList<FirmwareControllerListener>();

    private final NodeInterface node;

    private final JFrame parent;

    private final FirmwareModel firmwareModel = new FirmwareModel();

    private final NodeListProvider nodeListProvider;

    private NodeListListener nodeListListener;

    @Autowired
    private CommandStationService commandStationService;

    @Autowired
    private ConnectionRegistry connectionRegistry;

    @Autowired
    private SettingsService settingsService;

    @Autowired
    private DefaultWizardLabelFactory wizardLabelFactory;

    @Autowired
    private ConsoleService consoleService;

    private final ScheduledExecutorService firmwareUpdateWorker =
        Executors
            .newScheduledThreadPool(1,
                new ThreadFactoryBuilder().setNameFormat("firmwareUpdateWorkers-thread-%d").build());

    public FirmwareController(final NodeInterface node, final NodeListProvider nodeListProvider, JFrame parent) {
        this.parent = parent;
        this.node = node;
        this.nodeListProvider = nodeListProvider;
    }

    public void addFirmwareControllerListener(FirmwareControllerListener l) {
        listeners.add(l);
    }

    private void fireClose() {
        LOGGER.info("Close the firmware controller.");
        if (nodeListListener != null) {
            LOGGER.info("Remove the nodelist listener.");
            this.nodeListProvider.removeNodeListListener(nodeListListener);
            nodeListListener = null;
        }

        for (FirmwareControllerListener l : listeners) {
            l.close();
        }
    }

    private String prepareProductName(final NodeInterface node) {
        String productName = node.getNode().getStoredString(StringData.INDEX_PRODUCTNAME);

        if (ProductUtils.isLightControl(node.getUniqueId())) {
            // check the number of macros to distinguish between standard and signals
            if (firmwareModel.getNode().getStorableMacroCount() != 40) {
                productName += " (SIGNALS)";
            }
            else {
                productName += " (STANDARD)";
            }
        }

        return productName;
    }

    /**
     * Start the firmware update controller.
     */
    public void start() {
        // final Communication communication = CommunicationFactory.getInstance();

        // get current values
        firmwareModel.setNode(node);
        firmwareModel
            .setNodeName(node.getLabel() != null ? node.getLabel() : NodeUtils.prepareNodeLabel(node.getNode()));

        firmwareModel.setProductName(prepareProductName(node));
        long uniqueId = node.getNode().getUniqueId();
        firmwareModel.setUniqueId(ByteUtils.getUniqueIdAsString(uniqueId));

        SoftwareVersion swVersion = node.getNode().getSoftwareVersion();
        firmwareModel.setNodeCurrentVersion(swVersion != null ? swVersion.toString() : "unknown");

        // check for ClassID1 Bit 7 and addr = 0
        boolean isBootloaderRootNode = false;
        isBootloaderRootNode = node.isUpdatable() && Arrays.equals(node.getNode().getAddr(), Node.ROOTNODE_ADDR);
        LOGGER.info("The current node is a bootloader rootnode: {}", isBootloaderRootNode);
        firmwareModel.setBootloaderRootNode(isBootloaderRootNode);

        nodeListListener = new DefaultNodeListListener() {

            @Override
            public void listNodeRemoved(NodeInterface node) {
                LOGGER.info("Node was removed. Verify that it's not the node to be updated: {}", node);

                if (node.equals(FirmwareController.this.node)) {
                    LOGGER.error("The node that is updated was removed! Cancel all firmware update operations.");

                    SwingUtilities.invokeLater(() -> {
                        firmwareModel.setUpdateStatus(UpdateStatus.NODE_LOST);

                        firmwareModel
                            .addProcessingStatus(Resources.getString(FirmwareController.class, "status.node-lost"), 1);
                    });
                }
            }
        };
        this.nodeListProvider.addNodeListListener(nodeListListener);

        // create the firmware update view
        final FirmwareView view = new FirmwareView(parent, firmwareModel, settingsService);

        WindowUtils.centerOnCurrentScreen(parent, view);

        // TODO use the FirmwareUpdateService instead of connection
        final BidibConnection connection = connectionRegistry.getConnection(ConnectionRegistry.CONNECTION_ID_MAIN);

        firmwareModel
            .addPropertyChangeListener(FirmwareModel.PROPERTYNAME_CV_DEFINITION_FILES, new PropertyChangeListener() {

                @Override
                public void propertyChange(PropertyChangeEvent evt) {

                    if (CollectionUtils.isNotEmpty(firmwareModel.getCvDefinitionFiles())) {

                        // import the firmware definition files

                        List<String> cvDefinitionFiles = firmwareModel.getCvDefinitionFiles();
                        String version = firmwareModel.getUpdateVersion();
                        LOGGER.info("Import the CV definition files: {}, version: {}", cvDefinitionFiles, version);

                        String labelPath = settingsService.getMiscSettings().getBidibConfigDir();
                        File searchPathLabelPath = new File(labelPath, "data/BiDiBNodeVendorData");

                        // make sure the path exists
                        searchPathLabelPath.mkdirs();

                        String firmwareArchivePath = firmwareModel.getCvDefinitionArchivePath();
                        File firmwareFile = new File(firmwareArchivePath);

                        for (String cvDefinitionFile : cvDefinitionFiles) {

                            List<String> cvDefinitionContent = new ArrayList<String>();
                            try {
                                LOGGER
                                    .info("Load CV definition from archive file into buffer: {}", firmwareArchivePath);

                                cvDefinitionContent =
                                    FirmwareFactory.getCvDefinitionContent(firmwareFile, cvDefinitionFile);

                                LOGGER
                                    .info("Load CV definition from file into buffer passed. Total number of lines: {}",
                                        cvDefinitionContent.size());

                                String targetFileName = cvDefinitionFile;

                                // the cvDefinition filename must be versioned ...
                                if (!FirmwareUtils.hasVersionInFilename(cvDefinitionFile)) {
                                    // if (!cvDefinitionFile.contains(version)) {
                                    targetFileName =
                                        cvDefinitionFile.substring(0, cvDefinitionFile.indexOf(".xml")) + "-" + version
                                            + ".xml";
                                    LOGGER.info("Prepared cvDefinitionFile with the version: {}", targetFileName);
                                }
                                else {
                                    LOGGER.info("The cvDefinitionFile already contains a version: {}", targetFileName);
                                }
                                // LOGGER.info("Current targetFileName: {}", targetFileName);

                                FileOutputStream fos = null;
                                try {
                                    File file = new File(searchPathLabelPath, targetFileName);

                                    if (file.exists()) {
                                        // ask user to overwrite the file
                                        boolean override = FileDialog.askOverrideExisting(parent, file);
                                        if (!override) {
                                            LOGGER.info("User decided to not overwrite the existing file.");
                                            continue;
                                        }
                                    }

                                    fos = new FileOutputStream(file);
                                    IOUtils.writeLines(cvDefinitionContent, null, fos, StandardCharsets.UTF_8);
                                    fos.flush();
                                }
                                catch (Exception ex1) {
                                    LOGGER.warn("Write CV definition file failed: {}", cvDefinitionFile, ex1);
                                }
                                finally {
                                    if (fos != null) {
                                        try {
                                            fos.close();
                                        }
                                        catch (Exception e1) {
                                            LOGGER.warn("Close fos failed.", e1);
                                        }
                                    }
                                }

                                // check if default labels are in cvDefinitionFile
                                try (final ByteArrayOutputStream baos = new ByteArrayOutputStream()) {
                                    IOUtils.writeLines(cvDefinitionContent, null, baos, StandardCharsets.UTF_8);
                                    baos.flush();

                                    final Context context = new DefaultContext();
                                    ByteArrayInputStream bais = new ByteArrayInputStream(baos.toByteArray());
                                    final VendorCV vendorCV = VendorCvFactory.loadVendorCvFile(bais, context);

                                    if (vendorCV.getExtension() != null
                                        && CollectionUtils.isNotEmpty(vendorCV.getExtension().getNodeLabels())) {
                                        LOGGER.info("Found default node labels to import.");

                                        File defaultLabelPath = new File(labelPath, "data/defaultLabels");

                                        // make sure the directory exists
                                        if (!defaultLabelPath.exists()) {
                                            LOGGER.info("Create the defaultLabelPath directory: {}", defaultLabelPath);
                                            defaultLabelPath.mkdirs();
                                        }

                                        // We need the VID and PID of the firmware: BiDiBCV-251-118-0.04.00.xml
                                        String vidAndPid = FirmwareUtils.getVidAndPid(targetFileName);

                                        for (NodeLabels nodeLabels : vendorCV.getExtension().getNodeLabels()) {
                                            String lang = nodeLabels.getLang();
                                            LOGGER.info("Found node labels for lang: {}", lang);
                                            // TODO copy the node labels

                                            try {

                                                // String lang = XmlLocaleUtils.getXmlLocaleVendorCV();
                                                final String shortLang = StringUtils.substringBefore(lang, "-");

                                                // "bidib-default-names-13-138-de.xml"
                                                StringBuilder sb = new StringBuilder("bidib-default-names-");
                                                sb.append(vidAndPid).append("-").append(shortLang).append(".xml");

                                                String defaultFileName = sb.toString();
                                                LOGGER
                                                    .info(
                                                        "Prepared default filename: {}, stored at defaultLabelPath: {}",
                                                        defaultFileName, defaultLabelPath.getPath());

                                                File file = new File(defaultLabelPath, defaultFileName);

                                                // save the default labels
                                                wizardLabelFactory.saveDefaultLabels(nodeLabels, file);
                                            }
                                            catch (Exception ex2) {
                                                LOGGER.warn("Write default lables file failed, lang: {}", lang, ex2);
                                            }

                                        }
                                    }
                                }
                                catch (Exception ex1) {
                                    LOGGER
                                        .warn("Write CV definition to byte array output stream failed: {}",
                                            cvDefinitionFile, ex1);
                                }

                            }
                            catch (Exception e) {
                                LOGGER.warn("Load CV definition from file into buffer failed", e);

                                return;
                            }
                        }
                    }

                }
            });

        firmwareModel
            .addPropertyChangeListener(FirmwareModel.PROPERTYNAME_VENDOR_IMAGE_FILES, new PropertyChangeListener() {

                @Override
                public void propertyChange(PropertyChangeEvent evt) {

                    if (CollectionUtils.isNotEmpty(firmwareModel.getVendorImageFiles())) {

                        // import the firmware definition files

                        List<String> vendorImageFiles = firmwareModel.getVendorImageFiles();
                        String version = firmwareModel.getUpdateVersion();
                        LOGGER.info("Import the vendor image files: {}, version: {}", vendorImageFiles, version);

                        String labelPath = settingsService.getMiscSettings().getBidibConfigDir();
                        File vendorImageTargetPath = new File(labelPath, "data/images");

                        // make sure the path exists
                        vendorImageTargetPath.mkdirs();

                        String vendorImageArchivePath = firmwareModel.getVendorImageArchivePath();
                        File vendorImagePath = new File(vendorImageArchivePath);

                        for (String vendorImageFile : vendorImageFiles) {

                            try {
                                LOGGER
                                    .info("Load vendor image from archive file into buffer: {}",
                                        vendorImageArchivePath);

                                final ByteArrayOutputStream buffer = new ByteArrayOutputStream();

                                FirmwareFactory.getBinaryContent(vendorImagePath, vendorImageFile, buffer);

                                LOGGER
                                    .info("Load CV definition from file into buffer passed. Total size: {}",
                                        buffer.size());

                                String targetFileName = vendorImageFile;

                                File file = new File(vendorImageTargetPath, targetFileName);
                                if (file.exists()) {
                                    // ask user to overwrite the file
                                    boolean override = FileDialog.askOverrideExisting(parent, file);
                                    if (!override) {
                                        LOGGER.info("User decided to not overwrite the existing file.");
                                        continue;
                                    }
                                }

                                try (FileOutputStream fos = new FileOutputStream(file)) {

                                    buffer.writeTo(fos);
                                    fos.flush();
                                }
                                catch (Exception ex1) {
                                    LOGGER.warn("Write vendor image file failed: {}", vendorImageFile, ex1);
                                }

                            }
                            catch (Exception e) {
                                LOGGER.warn("Load vendor image from file into buffer failed", e);

                                return;
                            }
                        }
                    }

                }
            });

        firmwareModel
            .addPropertyChangeListener(FirmwareModel.PROPERTYNAME_DEFAULT_LABELS_FILES, new PropertyChangeListener() {

                @Override
                public void propertyChange(PropertyChangeEvent evt) {

                    if (CollectionUtils.isNotEmpty(firmwareModel.getDefaultLabelsFiles())) {

                        // import the default labels files

                        List<FilenameType> defaultLabelsFiles = firmwareModel.getDefaultLabelsFiles();
                        String version = firmwareModel.getUpdateVersion();
                        LOGGER.info("Import the default labels files: {}, version: {}", defaultLabelsFiles, version);

                        String labelPath = settingsService.getMiscSettings().getBidibConfigDir();
                        File searchPathLabelPath = new File(labelPath, "data/defaultLabels");

                        // make sure the path exists
                        searchPathLabelPath.mkdirs();

                        String firmwareArchivePath = firmwareModel.getCvDefinitionArchivePath();
                        File firmwareFile = new File(firmwareArchivePath);

                        // import the files
                        for (FilenameType defaultLabelFile : defaultLabelsFiles) {
                            String lang = StringUtils.substringBefore(defaultLabelFile.getLang(), "-");
                            String labelFilename = defaultLabelFile.getFilename();
                            LOGGER
                                .info("Found configured default labels file for lang: {}, filename: {}", lang,
                                    labelFilename);

                            List<String> defaultLabelsContent = new ArrayList<String>();
                            try {
                                LOGGER
                                    .info("Load default labels from archive file into buffer: {}", firmwareArchivePath);

                                defaultLabelsContent =
                                    FirmwareFactory.getCvDefinitionContent(firmwareFile, labelFilename);

                                LOGGER
                                    .info("Load default labels from file into buffer passed. Total number of lines: {}",
                                        defaultLabelsContent.size());

                                String targetFileName = labelFilename;

                                // the default labels filename must be versioned ...
                                if (!FirmwareUtils.hasVersionInDefaultLablesFilename(labelFilename)) {

                                    targetFileName =
                                        labelFilename.substring(0, labelFilename.indexOf(".xml") - 3) + "-" + version
                                            + "-" + lang + ".xml";
                                    LOGGER.info("Prepared default labels file with the version: {}", targetFileName);
                                }
                                else {
                                    LOGGER
                                        .info("The default labels file already contains a version: {}", targetFileName);
                                }
                                // LOGGER.info("Current targetFileName: {}", targetFileName);

                                FileOutputStream fos = null;
                                try {
                                    File file = new File(searchPathLabelPath, targetFileName);

                                    if (file.exists()) {
                                        // ask user to overwrite the file
                                        boolean override = FileDialog.askOverrideExisting(parent, file);
                                        if (!override) {
                                            LOGGER.info("User decided to not overwrite the existing file.");
                                            continue;
                                        }
                                    }

                                    fos = new FileOutputStream(file);
                                    IOUtils.writeLines(defaultLabelsContent, null, fos, StandardCharsets.UTF_8);
                                    fos.flush();
                                }
                                catch (Exception ex1) {
                                    LOGGER.warn("Write default labels file failed: {}", labelFilename, ex1);
                                }
                                finally {
                                    if (fos != null) {
                                        try {
                                            fos.close();
                                        }
                                        catch (Exception e1) {
                                            LOGGER.warn("Close fos failed.", e1);
                                        }
                                    }
                                }

                            }
                            catch (Exception e) {
                                LOGGER.warn("Load default labels from file into buffer failed", e);

                                return;
                            }
                        }
                    }
                }
            });

        view.addFirmwareViewListener(new FirmwareViewListener() {

            @Override
            public void close() {

                if (UpdateStatus.NONE.equals(firmwareModel.getUpdateStatus())
                    || UpdateStatus.NODE_LOST.equals(firmwareModel.getUpdateStatus())) {
                    LOGGER.info("No software update was started or node was lost: {}", firmwareModel.getUpdateStatus());
                }
                else {
                    LOGGER.info("Send the firmware update operation EXIT command.");
                    try {
                        sendCommand(connection, FirmwareUpdateOperation.EXIT);
                        LOGGER.info("Send the EXIT command to the node passed.");

                        if (firmwareModel.isBootloaderRootNode()) {
                            LOGGER
                                .info(
                                    "A bootloader root node was updated. We must release the root node and get the root and all children again.");

                            // trigger release the root node
                            connection.releaseAndReloadRootNode(node);
                        }
                    }
                    catch (InterruptedException e) {
                        LOGGER.warn("Send exit operation failed.", e);
                        throw new RuntimeException("Send exit operation failed.", e);
                    }
                }
                LOGGER.info("Close the dialog.");
                fireClose();
            }

            @Override
            public void updateFirmware() {
                // start the firmware update with the selected files
                final List<FirmwareNode> firmwareFiles = firmwareModel.getFirmwareFiles();
                final String firmwareArchivePath = firmwareModel.getFirmwareArchivePath();

                LOGGER.info("Start the firmware update process, firmwareArchivePath: {}", firmwareArchivePath);

                if (CollectionUtils.isNotEmpty(firmwareFiles)) {

                    if (firmwareModel.isBootloaderRootNode()) {
                        if (commandStationService != null) {
                            LOGGER
                                .info(
                                    "Stop all watchdog tasks before update because we update the bootloader root node.");
                            commandStationService.stopAllWatchDogTasks();
                        }
                    }

                    // start a new task to send the firmware to the node
                    final Runnable firmwareUpdateTask = () -> {
                        LOGGER.info("Start the firmware update process.");

                        List<FirmwareUpdatePart> firmwareUpdateParts = new LinkedList<>();
                        int totalFileSize = 0;
                        // Load the firmware into memory
                        for (FirmwareNode firmwareNode : firmwareFiles) {
                            int fileSize = 0;

                            File firmwareFile = new File(firmwareArchivePath);
                            int destination = firmwareNode.getDestinationNumber();

                            List<String> firmwareContent = null;
                            try {
                                LOGGER.info("Load firmware from file into buffer: {}", firmwareArchivePath);

                                firmwareContent =
                                    FirmwareFactory.getFirmwareContent(firmwareFile, firmwareNode.getFilename());

                                if (CollectionUtils.isNotEmpty(firmwareContent)) {
                                    for (String line : firmwareContent) {
                                        fileSize += line.length();
                                    }
                                }
                                LOGGER
                                    .info(
                                        "Load firmware from file into buffer passed. Total number of packets to transfer: {}",
                                        firmwareContent.size());

                                firmwareModel
                                    .addProcessingStatus(
                                        Resources.getString(FirmwareController.class, "status.load-firmware-passed"), 0,
                                        firmwareNode.getFilename());
                            }
                            catch (Exception e) {
                                LOGGER.warn("Load firmware from file into buffer failed", e);

                                firmwareModel
                                    .addProcessingStatus(
                                        Resources.getString(FirmwareController.class, "status.load-firmware-failed"), 1,
                                        firmwareNode.getFilename());
                                firmwareModel.setUpdateStatus(UpdateStatus.PREPARE_FAILED);
                                firmwareModel.setInProgress(false);
                                return;
                            }

                            FirmwareUpdatePart firmwareUpdatePart =
                                new FirmwareUpdatePart(firmwareNode.getFilename(), firmwareContent, destination,
                                    fileSize);
                            firmwareUpdateParts.add(firmwareUpdatePart);

                            totalFileSize += fileSize;
                        }

                        try {
                            LOGGER.info("Transfer file with length: {}", totalFileSize);
                            firmwareModel.setUpdateStatus(UpdateStatus.PREPARE);
                            boolean enterFwUpdateModePassed =
                                sendCommand(connection, FirmwareUpdateOperation.ENTER,
                                    ByteUtils.convertLongToUniqueId(node.getNode().getUniqueId()));
                            if (enterFwUpdateModePassed) {

                                firmwareModel.setUpdateStatus(UpdateStatus.ENTRY_PASSED);

                                int currentSize = 0;
                                boolean errorDetected = false;
                                // transfer all selected parts
                                for (FirmwareUpdatePart part : firmwareUpdateParts) {
                                    int destIdentifier = part.getDestination();

                                    LOGGER.info("Set the destination for the firmware: {}", destIdentifier);
                                    if (sendCommand(connection, FirmwareUpdateOperation.SETDEST,
                                        ByteUtils.getLowByte(destIdentifier))) {

                                        LOGGER
                                            .info("Set the destination for the firmware passed: {}",
                                                ByteUtils.getLowByte(destIdentifier));

                                        firmwareModel.setUpdateStatus(UpdateStatus.DATA_TRANSFER);
                                        firmwareModel
                                            .addProcessingStatus(
                                                Resources.getString(FirmwareController.class, "status.start-transfer"),
                                                0, part.getFilename());

                                        int block = 0;
                                        for (String line : part.getFirmwareContent()) {
                                            currentSize += line.length();
                                            firmwareModel.setProgressValue(currentSize * 100 / totalFileSize);
                                            LOGGER.trace("Send block: {}, line: {}", block, line);

                                            if (!sendCommand(connection, FirmwareUpdateOperation.DATA,
                                                line.getBytes())) {
                                                LOGGER
                                                    .warn(
                                                        "Unexpected answer from node while sending fw data (block#: {}, line: {}).",
                                                        block, line);

                                                errorDetected = true;
                                                firmwareModel
                                                    .addProcessingStatus(Resources
                                                        .getString(FirmwareController.class,
                                                            "status.unexpected-answer"),
                                                        1);
                                                firmwareModel
                                                    .addProcessingStatus(Resources
                                                        .getString(FirmwareController.class, "status.transfer-aborted"),
                                                        1, part.getFilename());

                                                break;
                                            }
                                            block++;
                                        }

                                        if (!errorDetected) {
                                            // wait 20ms before send DONE
                                            LOGGER.info("Wait 20ms before send DONE command.");
                                            Thread.sleep(20);

                                            LOGGER.info("Send the firmware update done command.");
                                            if (!sendCommand(connection, FirmwareUpdateOperation.DONE)) {

                                                errorDetected = true;
                                                firmwareModel
                                                    .addProcessingStatus(
                                                        Resources
                                                            .getString(FirmwareController.class,
                                                                "status.transfer-finished-answer-failed"),
                                                        0, part.getFilename());

                                            }
                                            else {
                                                firmwareModel
                                                    .addProcessingStatus(Resources
                                                        .getString(FirmwareController.class,
                                                            "status.transfer-finished"),
                                                        0, part.getFilename());
                                            }
                                        }
                                        else {
                                            LOGGER.warn("There was an error detected during firmware update.");
                                        }
                                    }

                                    if (errorDetected) {
                                        LOGGER.info("Error detected during firmware update. Cancel update process.");

                                        firmwareModel.setUpdateStatus(UpdateStatus.DATA_TRANSFER_FAILED);
                                        break;
                                    }
                                }

                                if (!errorDetected) {
                                    // all parts were transfered
                                    firmwareModel.setUpdateStatus(UpdateStatus.DATA_TRANSFER_PASSED);
                                    LOGGER.info("The firmware update has passed.");
                                }
                                else {
                                    LOGGER.warn("The firmware update has not passed.");
                                }

                                firmwareModel.setProgressValue(100);
                                firmwareModel.setInProgress(false);
                            }
                            else {
                                LOGGER.warn("Enter firmware update operation failed.");

                                firmwareModel.setProgressValue(0);
                                firmwareModel.setInProgress(false);
                                firmwareModel
                                    .addProcessingStatus(Resources
                                        .getString(FirmwareController.class,
                                            "status.enter-firmware-update-mode-failed"),
                                        1);

                                firmwareModel.setUpdateStatus(UpdateStatus.NONE);
                            }
                        }
                        catch (Exception e) {
                            LOGGER.warn("Transfer firmware udpate to node failed.", e);

                            firmwareModel
                                .addProcessingStatus(
                                    Resources.getString(FirmwareController.class, "status.transfer-firmware-failed"),
                                    1);
                            firmwareModel.setUpdateStatus(UpdateStatus.DATA_TRANSFER_FAILED);
                            firmwareModel.setInProgress(false);
                        }
                    };
                    firmwareUpdateWorker.submit(firmwareUpdateTask);
                }
                else {
                    firmwareModel.setInProgress(false);
                }
            }
        });

        view.setVisible(true);
    }

    private boolean sendCommand(BidibConnection connection, FirmwareUpdateOperation operation, byte... data)
        throws InterruptedException {
        boolean result = false;

        FirmwareUpdateStat updateStat = null;
        // handle firmware update operations
        // send firmware operation to node and wait for result
        try {
            // firmwareUpdateService.sendFirmwareUpdateOperation(node, operation, data);
            updateStat = connection.sendFirmwareUpdateOperation(node, operation, data);
        }
        catch (NoAnswerException ex) {
            switch (operation) {
                case ENTER:
                    LOGGER.warn("No answer received during enter firmware update mode.", ex);
                    break;
                case EXIT:
                    LOGGER.warn("No answer received during exit firmware update mode.", ex);
                    break;
                default:
                    LOGGER.warn("No answer received during firmware update, try send data again.", ex);
                    traceTimeout(operation, true);
                    break;
            }
        }

        if (updateStat == null && FirmwareUpdateOperation.DATA == operation) {
            LOGGER.warn("Try to retransfer the data block.");
            try {
                updateStat = connection.sendFirmwareUpdateOperation(node, operation, data);
            }
            catch (NoAnswerException ex) {
                LOGGER
                    .warn(
                        "No answer received during firmware update when retry send data. Firmware update process will be aborted.",
                        ex);
                traceTimeout(operation, false);
            }
        }

        if (updateStat != null) {
            LOGGER
                .info("Received update stat, timeout: {}, state: {}, last operation: {}", updateStat.getTimeout(),
                    updateStat.getState(), operation);

            if (updateStat.getTimeout() > 0) {
                int extendedTime = updateStat.getTimeout() * 10;
                LOGGER.warn("The node requested a wait to complete: {} ms", extendedTime);
                traceMoreTimeRequested(extendedTime);
                Thread.sleep(extendedTime);
                traceContinueAfterMoreTimeRequested(extendedTime);
            }

            FirmwareUpdateState state = updateStat.getState();

            if ((operation == FirmwareUpdateOperation.ENTER && state == FirmwareUpdateState.READY)
                || (operation == FirmwareUpdateOperation.DONE && state == FirmwareUpdateState.READY)
                || (operation == FirmwareUpdateOperation.EXIT && state == FirmwareUpdateState.EXIT)
                || (operation == FirmwareUpdateOperation.SETDEST && state == FirmwareUpdateState.DATA)
                || (operation == FirmwareUpdateOperation.DATA && state == FirmwareUpdateState.DATA)) {
                result = true;
            }
            else {
                LOGGER
                    .warn(
                        "The firmware update state ({}) returned from the node does not match the firmware operation ({}).",
                        state, operation);
                // display warning in console
                ConsoleController.ensureConsoleVisible();

                consoleService
                    .addConsoleLine(ConsoleColor.red, String
                        .format(
                            "The firmware update state (%s) returned from the node does not match the firmware operation (%s).",
                            state != null ? state.name() : "unknown",
                            operation != null ? operation.name() : "unknown"));

                traceInvalidAnswer(state, operation);
            }
        }
        else {
            LOGGER.warn("No updateStat received for operation: {}", operation);
            // display warning in console
            ConsoleController.ensureConsoleVisible();

            consoleService
                .addConsoleLine(ConsoleColor.red, String
                    .format("No firmware update state returned from the node after send the firmware operation (%s).",
                        operation != null ? operation.name() : "unknown"));

            traceInvalidAnswer(operation);
        }
        LOGGER.debug("sendCommand return result: {}", result);
        return result;
    }

    private void traceTimeout(final FirmwareUpdateOperation operation, boolean retry) {
        if (retry) {
            firmwareModel
                .addProcessingStatus(
                    Resources.getString(FirmwareController.class, "status.transfer-firmware-timeout-retry"), 1,
                    operation);
        }
        else {
            firmwareModel
                .addProcessingStatus(Resources.getString(FirmwareController.class, "status.transfer-firmware-timeout"),
                    1, operation);
        }
    }

    private void traceMoreTimeRequested(int extendedTime) {
        firmwareModel
            .addProcessingStatus(
                Resources.getString(FirmwareController.class, "status.transfer-firmware-more-time-requested"), 1,
                extendedTime);
    }

    private void traceContinueAfterMoreTimeRequested(int extendedTime) {
        firmwareModel
            .addProcessingStatus(
                Resources.getString(FirmwareController.class, "status.transfer-firmware-continue-after-more-time-wait"),
                1, extendedTime);
    }

    private void traceInvalidAnswer(FirmwareUpdateState state, FirmwareUpdateOperation operation) {
        firmwareModel
            .addProcessingStatus(
                Resources.getString(FirmwareController.class, "status.transfer-firmware-invalid-answer"), 1,
                state != null ? state.name() : "unknown", operation != null ? operation.name() : "unknown");
    }

    private void traceInvalidAnswer(FirmwareUpdateOperation operation) {
        firmwareModel
            .addProcessingStatus(Resources.getString(FirmwareController.class, "status.transfer-firmware-no-answer"), 1,
                operation != null ? operation.name() : "unknown");
    }
}
