package org.bidib.wizard.mvc.main.view.table;

import java.util.concurrent.TimeUnit;
import java.util.function.Consumer;

import javax.swing.JComponent;
import javax.swing.JPanel;
import javax.swing.JTextField;

import org.bidib.jbidibc.messages.BidibLibrary;
import org.bidib.wizard.api.locale.Resources;
import org.bidib.wizard.client.common.component.SliderAndValuePanel;
import org.bidib.wizard.client.common.controller.NodeSelectionProvider;
import org.bidib.wizard.client.common.converter.StringToIntegerConverter;
import org.bidib.wizard.client.common.rxjava2.SwingScheduler;
import org.bidib.wizard.client.common.table.AbstractPortEditorPanel;
import org.bidib.wizard.client.common.text.InputValidationDocument;
import org.bidib.wizard.client.common.text.WizardBindings;
import org.bidib.wizard.client.common.text.WizardComponentFactory;
import org.bidib.wizard.client.common.view.validation.IconFeedbackPanel;
import org.bidib.wizard.client.common.view.validation.PropertyValidationI18NSupport;
import org.bidib.wizard.model.ports.BacklightPort;
import org.bidib.wizard.client.common.view.validation.DefaultRangeValidationCallback;
import org.bidib.wizard.client.common.view.validation.IntegerInputValidationDocument;
import org.bidib.wizard.mvc.main.model.BacklightPortTableModel;
import org.bidib.wizard.client.common.model.ServoPortTableModel;
import org.bidib.wizard.mvc.main.view.component.BacklightSliderAndValuePanel;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.jgoodies.binding.adapter.Bindings;
import com.jgoodies.binding.value.BindingConverter;
import com.jgoodies.binding.value.BufferedValueModel;
import com.jgoodies.binding.value.ConverterValueModel;
import com.jgoodies.binding.value.ValueModel;
import com.jgoodies.forms.builder.FormBuilder;
import com.jgoodies.forms.factories.Paddings;
import com.jgoodies.validation.ValidationResult;
import com.jgoodies.validation.util.PropertyValidationSupport;
import com.jgoodies.validation.view.ValidationComponentUtils;

import io.reactivex.rxjava3.disposables.Disposable;
import io.reactivex.rxjava3.subjects.PublishSubject;

public class BacklightPortEditorPanel extends AbstractPortEditorPanel<BacklightPort> {

    private static final Logger LOGGER = LoggerFactory.getLogger(BacklightPortEditorPanel.class);

    private static final String ENCODED_DIALOG_COLUMN_SPECS =
        "pref, 3dlu, pref, 3dlu, pref, 3dlu, pref, 3dlu, fill:50dlu:grow";

    private static final String ENCODED_DIALOG_ROW_SPECS =
        "pref, 3dlu, pref, 3dlu, pref, 3dlu, pref, 3dlu, pref, 3dlu, pref, 3dlu, pref, 3dlu, pref, 3dlu, pref";

    private static final int TIME_BETWEEN_VALUE_EVENTS_MILLIS = 80;

    private ValueModel portLevelOffConverterModel;

    private ValueModel portLevelOnConverterModel;

    private ValueModel dimmDownConverterModel;

    private ValueModel dimmUpConverterModel;

    private BufferedValueModel dimmUpBufferedValueModel;

    private BufferedValueModel dimmDownBufferedValueModel;

    private ValueModel dmxMappingConverterModel;

    private PublishSubject<Integer> valueEventSubject = PublishSubject.create();

    public BacklightPortEditorPanel(BacklightPort port, Consumer<BacklightPort> saveCallback,
        final Consumer<BacklightPort> valueCallback, final Consumer<BacklightPort> refreshCallback, final NodeSelectionProvider nodeSelectionProvider) {
        super(port, saveCallback, valueCallback, refreshCallback, nodeSelectionProvider);
    }

    @Override
    protected BacklightPort clonePort(final BacklightPort port) {
        // create a clone of the input port
        final BacklightPort clone =
            BacklightPort
                .builder() //
                .withDimSlopeUp(port.getDimSlopeUp()) //
                .withDimSlopeDown(port.getDimSlopeDown()) //
                .withDimStretchMax(port.getDimStretchMax()) //
                .withDimStretchMin(port.getDimStretchMin()) //
                .withDmxMapping(port.getDmxMapping()) //
                .withStatus(port.getStatus()) //
                .withRemappingEnabled(port.isRemappingEnabled()) //
                .withKnownPortConfigKeys(port.getKnownPortConfigKeys()) //
                .withId(port.getId()) //
                .withLabel(port.getLabel()) //
                .withEnabled(port.isEnabled()) //
                .withIsInactive(port.isInactive()) //
                .withPortIdentifier(port.getPortIdentifier()).build();
        return clone;
    }

    @Override
    protected JPanel doCreateComponent(final BacklightPort port) {

        int row = 1;
        FormBuilder dialogBuilder = null;
        boolean debugDialog = false;
        if (debugDialog) {
            JPanel panel = new PortEditorPanelDebugContainer(this);
            dialogBuilder =
                FormBuilder.create().columns(ENCODED_DIALOG_COLUMN_SPECS).rows(ENCODED_DIALOG_ROW_SPECS).panel(panel);
        }
        else {
            JPanel panel = new PortEditorPanelContainer(this);
            dialogBuilder =
                FormBuilder.create().columns(ENCODED_DIALOG_COLUMN_SPECS).rows(ENCODED_DIALOG_ROW_SPECS).panel(panel);
        }
        dialogBuilder.border(Paddings.TABBED_DIALOG);

        // port name
        final BufferedValueModel bufferedPortNameModel =
            getPresentationModel().getBufferedModel(BacklightPort.PROPERTY_LABEL);

        dialogBuilder.add(Resources.getString(BacklightPortTableModel.class, "label") + ":").xy(1, row);
        final JTextField portName = WizardComponentFactory.createTextField(bufferedPortNameModel, false);
        dialogBuilder.add(portName).xyw(3, row, 7);

        row += 2;

        // if dimm min is not available we must not show it here
        if (isPortConfigKeySupported(BidibLibrary.BIDIB_PCFG_DIMM_DOWN)
            || isPortConfigKeySupported(BidibLibrary.BIDIB_PCFG_DIMM_DOWN_8_8)) {

            int maxRange = isPortConfigKeySupported(BidibLibrary.BIDIB_PCFG_DIMM_DOWN_8_8) ? 65535 : 255;

            // final BufferedValueModel valueValueModel =
            this.dimmDownBufferedValueModel =
                getPresentationModel().getBufferedModel(BacklightPort.PROPERTY_DIM_SLOPE_DOWN);

            // dimm min
            this.dimmDownConverterModel =
                new ConverterValueModel(this.dimmDownBufferedValueModel, new StringToIntegerConverter());

            JTextField dimmDownText = new JTextField();
            final IntegerInputValidationDocument dimmDownDocument =
                new IntegerInputValidationDocument(5, InputValidationDocument.NUMERIC);
            dimmDownDocument.setRangeValidationCallback(new DefaultRangeValidationCallback(0, maxRange));
            dimmDownText.setDocument(dimmDownDocument);
            dimmDownText.setColumns(5);
            dimmDownText.setHorizontalAlignment(JTextField.RIGHT);

            // bind manually because we changed the document of the textfield
            Bindings.bind(dimmDownText, dimmDownConverterModel, false);

            dialogBuilder.add(Resources.getString(BacklightPortTableModel.class, "dimSlopeDown")).xy(1, row);
            dialogBuilder.add(dimmDownText).xy(3, row);

            dimmDownText.setEnabled(port.isEnabled());

            ValidationComponentUtils.setMandatory(dimmDownText, true);
            ValidationComponentUtils.setMessageKeys(dimmDownText, "validation.dimmDown_key");

            // use the clone port to provide the values
            final SliderAndValuePanel valueSlider = new SliderAndValuePanel(0, maxRange, 10);
            valueSlider.createComponent();
            WizardBindings.bind(valueSlider, this.dimmDownBufferedValueModel);

            dialogBuilder.add(valueSlider).xyw(5, row, 5);

            this.dimmDownBufferedValueModel.addValueChangeListener(evt -> {
                if (this.dimmDownBufferedValueModel.getValue() instanceof Integer) {
                    int dimmDownValue = ((Integer) this.dimmDownBufferedValueModel.getValue()).intValue();

                    LOGGER.info("The dimmDown value has been changed: {}", dimmDownValue);

                    // positionEventSubject.onNext(portLevelOnValue);
                }
            });

            row += 2;
        }

        // if dimm max is not available we must not show it here
        if (isPortConfigKeySupported(BidibLibrary.BIDIB_PCFG_DIMM_UP)
            || isPortConfigKeySupported(BidibLibrary.BIDIB_PCFG_DIMM_UP_8_8)) {

            int maxRange = isPortConfigKeySupported(BidibLibrary.BIDIB_PCFG_DIMM_UP_8_8) ? 65535 : 255;

            // final BufferedValueModel valueValueModel =
            this.dimmUpBufferedValueModel =
                getPresentationModel().getBufferedModel(BacklightPort.PROPERTY_DIM_SLOPE_UP);

            // dimm max
            this.dimmUpConverterModel =
                new ConverterValueModel(this.dimmUpBufferedValueModel, new StringToIntegerConverter());

            JTextField dimmUpText = new JTextField();
            final IntegerInputValidationDocument dimmUpDocument =
                new IntegerInputValidationDocument(5, InputValidationDocument.NUMERIC);
            dimmUpDocument.setRangeValidationCallback(new DefaultRangeValidationCallback(0, maxRange));
            dimmUpText.setDocument(dimmUpDocument);
            dimmUpText.setColumns(5);
            dimmUpText.setHorizontalAlignment(JTextField.RIGHT);

            // bind manually because we changed the document of the textfield
            Bindings.bind(dimmUpText, dimmUpConverterModel, false);

            dialogBuilder.add(Resources.getString(BacklightPortTableModel.class, "dimSlopeUp")).xy(1, row);
            dialogBuilder.add(dimmUpText).xy(3, row);

            dimmUpText.setEnabled(port.isEnabled());

            ValidationComponentUtils.setMandatory(dimmUpText, true);
            ValidationComponentUtils.setMessageKeys(dimmUpText, "validation.dimmUp_key");

            // use the clone port to provide the values
            final SliderAndValuePanel valueSlider = new SliderAndValuePanel(0, maxRange, 10);
            valueSlider.createComponent();
            WizardBindings.bind(valueSlider, this.dimmUpBufferedValueModel);

            dialogBuilder.add(valueSlider).xyw(5, row, 5);

            this.dimmUpBufferedValueModel.addValueChangeListener(evt -> {
                if (this.dimmUpBufferedValueModel.getValue() instanceof Integer) {
                    int dimmUpValue = ((Integer) this.dimmUpBufferedValueModel.getValue()).intValue();

                    LOGGER.info("The dimmUp value has been changed: {}", dimmUpValue);

                    // positionEventSubject.onNext(dimmUpValue);
                }
            });

            row += 2;
        }

        // // if port level off is not available we must not show it here
        // if (isPortConfigKeySupported(BidibLibrary.BIDIB_PCFG_LEVEL_PORT_OFF)) {
        //
        // final BufferedValueModel valueValueModel =
        // getPresentationModel().getBufferedModel(BacklightPort.PROPERTY_PWM_MIN);
        //
        // // port level off
        // this.portLevelOffConverterModel = new ConverterValueModel(valueValueModel, new StringToIntegerConverter());
        //
        // JTextField portLevelOffText = new JTextField();
        // final IntegerInputValidationDocument portLevelOffDocument =
        // new IntegerInputValidationDocument(5, InputValidationDocument.NUMERIC);
        // portLevelOffDocument.setRangeValidationCallback(new DefaultRangeValidationCallback(0, 255));
        // portLevelOffText.setDocument(portLevelOffDocument);
        // portLevelOffText.setColumns(2);
        // portLevelOffText.setHorizontalAlignment(JTextField.RIGHT);
        //
        // // bind manually because we changed the document of the textfield
        // Bindings.bind(portLevelOffText, portLevelOffConverterModel, false);
        //
        // dialogBuilder.add(Resources.getString(BacklightPortTableModel.class, "portLevelOff")).xy(1, row);
        // dialogBuilder.add(portLevelOffText).xy(3, row);
        //
        // portLevelOffText.setEnabled(port.isEnabled());
        //
        // ValidationComponentUtils.setMandatory(portLevelOffText, true);
        // ValidationComponentUtils.setMessageKeys(portLevelOffText, "validation.portLevelOff_key");
        //
        // // use the clone port to provide the values
        // final SliderAndValuePanel valueSlider = new SliderAndValuePanel(0, 255, 10);
        // valueSlider.createComponent();
        // WizardBindings.bind(valueSlider, valueValueModel);
        //
        // dialogBuilder.add(valueSlider).xyw(5, row, 5);
        //
        // valueValueModel.addValueChangeListener(evt -> {
        // if (valueValueModel.getValue() instanceof Integer) {
        // int portLevelOffValue = ((Integer) valueValueModel.getValue()).intValue();
        //
        // LOGGER.info("The portLevelOff value has been changed: {}", portLevelOffValue);
        //
        // // positionEventSubject.onNext(portLevelOffValue);
        // }
        // });
        //
        // row += 2;
        // }
        //
        // // if port level on is not available we must not show it here
        // if (isPortConfigKeySupported(BidibLibrary.BIDIB_PCFG_LEVEL_PORT_ON)) {
        //
        // final BufferedValueModel valueValueModel =
        // getPresentationModel().getBufferedModel(BacklightPort.PROPERTY_PWM_MAX);
        //
        // // PWM max
        // this.portLevelOnConverterModel = new ConverterValueModel(valueValueModel, new StringToIntegerConverter());
        //
        // JTextField portLevelOnText = new JTextField();
        // final IntegerInputValidationDocument portLevelOnDocument =
        // new IntegerInputValidationDocument(5, InputValidationDocument.NUMERIC);
        // portLevelOnDocument.setRangeValidationCallback(new DefaultRangeValidationCallback(0, 255));
        // portLevelOnText.setDocument(portLevelOnDocument);
        // portLevelOnText.setColumns(2);
        // portLevelOnText.setHorizontalAlignment(JTextField.RIGHT);
        //
        // // bind manually because we changed the document of the textfield
        // Bindings.bind(portLevelOnText, portLevelOnConverterModel, false);
        //
        // dialogBuilder.add(Resources.getString(BacklightPortTableModel.class, "portLevelOn")).xy(1, row);
        // dialogBuilder.add(portLevelOnText).xy(3, row);
        //
        // portLevelOnText.setEnabled(port.isEnabled());
        //
        // ValidationComponentUtils.setMandatory(portLevelOnText, true);
        // ValidationComponentUtils.setMessageKeys(portLevelOnText, "validation.portLevelOn_key");
        //
        // // use the clone port to provide the values
        // final SliderAndValuePanel valueSlider = new SliderAndValuePanel(0, 255, 10);
        // valueSlider.createComponent();
        // WizardBindings.bind(valueSlider, valueValueModel);
        //
        // dialogBuilder.add(valueSlider).xyw(5, row, 5);
        //
        // valueValueModel.addValueChangeListener(evt -> {
        // if (valueValueModel.getValue() instanceof Integer) {
        // int portLevelOnValue = ((Integer) valueValueModel.getValue()).intValue();
        //
        // LOGGER.info("The portLevelOn value has been changed: {}", portLevelOnValue);
        //
        // // positionEventSubject.onNext(portLevelOnValue);
        // }
        // });
        //
        // row += 2;
        // }

        // if DMX mapping is not available we must not show it here
        if (isPortConfigKeySupported(BidibLibrary.BIDIB_PCFG_OUTPUT_MAP)) {
            this.dmxMappingConverterModel =
                new ConverterValueModel(getPresentationModel().getBufferedModel(BacklightPort.PROPERTY_DMX_MAPPING),
                    new StringToIntegerConverter());

            JTextField dmxMappingText = new JTextField();
            final IntegerInputValidationDocument dmxMappingDocument =
                new IntegerInputValidationDocument(3, InputValidationDocument.NUMERIC);
            dmxMappingDocument.setRangeValidationCallback(new DefaultRangeValidationCallback(0, 255));
            dmxMappingText.setDocument(dmxMappingDocument);
            dmxMappingText.setColumns(2);
            dmxMappingText.setHorizontalAlignment(JTextField.RIGHT);

            // bind manually because we changed the document of the textfield
            Bindings.bind(dmxMappingText, dmxMappingConverterModel, false);

            dialogBuilder.add(Resources.getString(BacklightPortTableModel.class, "dmxMapping")).xy(1, row);
            dialogBuilder.add(dmxMappingText).xy(3, row);

            ValidationComponentUtils.setMandatory(dmxMappingText, true);
            ValidationComponentUtils.setMessageKeys(dmxMappingText, "validation.dmxMapping_key");

            row += 2;
        }

        // value slider

        // add value slider
        final BufferedValueModel valueValueModel =
            getPresentationModel().getBufferedModel(BacklightPort.PROPERTYNAME_VALUE);

        final ValueModel valueConverterModel =
            new ConverterValueModel(valueValueModel, new BacklightValueToRelativeValueConverter());

        // use the clone port to provide the values
        final BacklightSliderAndValuePanel valueSlider = new BacklightSliderAndValuePanel(0, 100, 10, () -> {

            // if the dimm up and down values are adjusted the range will change
            // check if buffered models are available
            int dimmUp =
                this.dimmUpBufferedValueModel != null ? ((Integer) this.dimmUpBufferedValueModel.getValue()).intValue()
                    : 255;
            int dimmDown =
                this.dimmDownBufferedValueModel != null
                    ? ((Integer) this.dimmDownBufferedValueModel.getValue()).intValue() : 0;

            final BacklightPort configPort =
                BacklightPort
                    .builder() //
                    .withDimSlopeUp(dimmUp) //
                    .withDimSlopeDown(dimmDown)//
                    .build();
            return configPort;
        });
        valueSlider.createComponent();
        WizardBindings.bind(valueSlider, valueConverterModel);

        valueValueModel.addValueChangeListener(evt -> {
            if (valueValueModel.getValue() instanceof Integer) {
                int position = ((Integer) valueValueModel.getValue()).intValue();

                LOGGER.info("The position has been changed by the slider. New position: {}", position);

                valueEventSubject.onNext(position);
            }
        });

        final Disposable disp =
            this.valueEventSubject
                .throttleLatest(TIME_BETWEEN_VALUE_EVENTS_MILLIS, TimeUnit.MILLISECONDS, SwingScheduler.getInstance(),
                    true)
                .subscribe(value -> sendValueToBacklightPort(port, value), err -> {
                    LOGGER.warn("The value event subject signalled an error.", err);
                }, () -> {
                    LOGGER.info("The value event subject has completed.");
                });
        getCompDisp().add(disp);

        dialogBuilder.add(Resources.getString(ServoPortTableModel.class, "value")).xy(1, 9);
        dialogBuilder.add(valueSlider).xyw(3, 9, 7);

        row += 2;

        // add buttons
        final JPanel buttonPanel = createButtonPanel();
        dialogBuilder.add(buttonPanel).xyw(1, row, 9);

        // check if we have validation enabled
        if (getValidationResultModel() != null) {
            LOGGER.debug("Create iconfeedback panel.");
            JComponent cvIconPanel = new IconFeedbackPanel(getValidationResultModel(), dialogBuilder.build());
            JPanel panel = new PortEditorPanelContainer(this);
            FormBuilder feedbackBuilder = FormBuilder.create().columns("p:g").rows("fill:p:grow").panel(panel);

            feedbackBuilder.add(cvIconPanel).xy(1, 1);

            setPanel(feedbackBuilder.build());
        }
        else {
            setPanel(dialogBuilder.build());
        }

        bufferedPortNameModel.addValueChangeListener(evt -> triggerValidation());
        if (this.portLevelOffConverterModel != null) {
            this.portLevelOffConverterModel.addValueChangeListener(evt -> triggerValidation());
        }
        if (this.portLevelOnConverterModel != null) {
            this.portLevelOnConverterModel.addValueChangeListener(evt -> triggerValidation());
        }
        if (this.dimmDownConverterModel != null) {
            this.dimmDownConverterModel.addValueChangeListener(evt -> triggerValidation());
        }
        if (this.dimmUpConverterModel != null) {
            this.dimmUpConverterModel.addValueChangeListener(evt -> triggerValidation());
        }
        if (this.dmxMappingConverterModel != null) {
            this.dmxMappingConverterModel.addValueChangeListener(evt -> triggerValidation());
        }

        triggerValidation();

        return getPanel();
    }

    @Override
    protected ValidationResult validate(final BacklightPort port) {
        PropertyValidationSupport support = new PropertyValidationI18NSupport(getPresentationModel(), "validation");

        if (this.portLevelOffConverterModel != null && this.portLevelOffConverterModel.getValue() == null) {
            support.addError("portLevelOn_key", "not_empty_for_write");
        }
        if (this.portLevelOnConverterModel != null && this.portLevelOnConverterModel.getValue() == null) {
            support.addError("portLevelOff_key", "not_empty_for_write");
        }
        if (this.dimmDownConverterModel != null && this.dimmDownConverterModel.getValue() == null) {
            support.addError("dimmDownTime_key", "not_empty_for_write");
        }
        if (this.dimmUpConverterModel != null && this.dimmUpConverterModel.getValue() == null) {
            support.addError("dimmUp_key", "not_empty_for_write");
        }
        if (this.dmxMappingConverterModel != null && this.dmxMappingConverterModel.getValue() == null) {
            support.addError("dmxMapping_key", "not_empty_for_write");
        }

        ValidationResult validationResult = support.getResult();
        return validationResult;
    }

    private void sendValueToBacklightPort(final BacklightPort port, int position) {
        LOGGER.info("Send the position to the backlight port: {}", position);

        triggerValidation();

        final BacklightPort servoPort = BacklightPort.builder().withValue(position).withId(port.getId()).build();

        getValueCallback().accept(servoPort);
    }

    private static final class BacklightValueToRelativeValueConverter implements BindingConverter<Integer, Integer> {

        @Override
        public Integer targetValue(Integer sourceValue) {
            if (sourceValue instanceof Integer) {
                int val = sourceValue.intValue();

                val = BacklightPort.getRelativeValue(val);
                LOGGER.trace("Converted source to target value: {} -> {}", sourceValue, val);

                return Integer.valueOf(val);
            }
            return null;
        }

        @Override
        public Integer sourceValue(Integer targetValue) {
            if (targetValue instanceof Integer) {
                int val = targetValue.intValue();

                val = BacklightPort.getAbsoluteValue(val);
                LOGGER.trace("Converted target to source value: {} -> {}", targetValue, val);

                return Integer.valueOf(val);
            }
            return null;
        }
    }

}
