package org.bidib.wizard.mvc.main.model;

import java.util.function.IntConsumer;

import org.bidib.jbidibc.messages.BidibLibrary;
import org.bidib.jbidibc.messages.enums.LcOutputType;
import org.bidib.jbidibc.messages.enums.PortConfigKeys;
import org.bidib.wizard.api.locale.Resources;
import org.bidib.wizard.api.utils.PortUtils;
import org.bidib.wizard.client.common.model.SimplePortTableModel;
import org.bidib.wizard.model.ports.SoundPort;
import org.bidib.wizard.model.status.SoundPortStatus;
import org.bidib.wizard.mvc.main.model.listener.SoundPortModelListener;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class SoundPortTableModel extends SimplePortTableModel<SoundPortStatus, SoundPort, SoundPortModelListener> {
    private static final Logger LOGGER = LoggerFactory.getLogger(SoundPortTableModel.class);

    private final MainModel mainModel;

    private static final long serialVersionUID = 1L;

    public static final int COLUMN_LABEL = 0;

    public static final int COLUMN_PULSE_TIME = 1;

    public static final int COLUMN_PORT_IDENTIFIER = 2;

    public static final int COLUMN_STATUS = 3;

    public static final int COLUMN_TEST = 4;

    public static final int COLUMN_PORT_INSTANCE = 5;

    public SoundPortTableModel(final MainModel model) {
        this.mainModel = model;
        // setColumnIdentifiers(COLUMN_NAMES);
    }

    @Override
    public int getColumnPortInstance() {
        return COLUMN_PORT_INSTANCE;
    }

    @Override
    public void notifyPortStatusChanged(final SoundPort port) {
        // the port status is signaled from the node
        for (int row = 0; row < getRowCount(); row++) {
            if (port.equals(getValueAt(row, COLUMN_PORT_INSTANCE))) {

                LOGGER.debug("The port state has changed: {}", port.getStatus());
                super.setValueAt(port.getStatus(), row, COLUMN_STATUS);

                // get the opposite status and set it
                SoundPortStatus oppositeStatus = PortUtils.getOppositeStatus(port.getStatus());

                LOGGER.info("Set the port status, oppositeStatus: {}", oppositeStatus);

                setValueAt(oppositeStatus, row, COLUMN_TEST);
                break;
            }
        }
    }

    @Override
    public void notifyPortConfigChanged(SoundPort port) {
        // TODO Auto-generated method stub

    }

    @Override
    protected void initialize() {
        columnNames =
            new String[] { Resources.getString(getClass(), "label"), Resources.getString(getClass(), "pulseTime"),
                Resources.getString(getClass(), "portIdentifier"), Resources.getString(getClass(), "status"),
                Resources.getString(getClass(), "test"), null };
    }

    @Override
    public void addRow(SoundPort port) {
        if (port != null) {
            Object[] rowData = new Object[columnNames.length];

            rowData[COLUMN_LABEL] = port.getLabel();
            rowData[COLUMN_PULSE_TIME] = port.getPulseTime();
            rowData[COLUMN_PORT_IDENTIFIER] = port.getPortIdentifier();
            rowData[COLUMN_STATUS] = port.getStatus();

            SoundPortStatus oppositeStatus = PortUtils.getOppositeStatus(port.getStatus());
            rowData[COLUMN_TEST] = oppositeStatus;
            rowData[COLUMN_PORT_INSTANCE] = port;
            addRow(rowData);
        }
    }

    @Override
    public boolean isCellEditable(int row, int column) {
        boolean isEditable = false;
        SoundPort soundPort = (SoundPort) getValueAt(row, COLUMN_PORT_INSTANCE);
        switch (column) {
            case COLUMN_LABEL:
                isEditable = true;
                break;
            case COLUMN_PULSE_TIME:
                if (soundPort.isEnabled() && soundPort.isPortConfigKeySupported(BidibLibrary.BIDIB_PCFG_TICKS)) {
                    isEditable = true;
                }
                break;
            case COLUMN_STATUS:
                // the status can never be changed.
                isEditable = false;
            case COLUMN_TEST:
                if (soundPort.isEnabled()) {
                    isEditable = true;
                }
                break;
            default:
                break;
        }
        return isEditable;
    }

    @Override
    public Class<?> getColumnClass(int column) {
        switch (column) {
            case COLUMN_LABEL:
                return String.class;
            case COLUMN_PORT_INSTANCE:
            case COLUMN_PORT_IDENTIFIER:
                return Object.class;
            case COLUMN_STATUS:
                return Object.class;
            case COLUMN_TEST:
                return Object.class;

            default:
                return Object.class;
        }
    }

    @Override
    public void setValueAt(Object value, int row, int column) {
        final Object o = getValueAt(row, COLUMN_PORT_INSTANCE);

        if (o instanceof SoundPort) {
            final SoundPort port = (SoundPort) o;

            switch (column) {
                case COLUMN_LABEL:
                    // if (value instanceof String) {
                    port.setLabel((String) value);
                    super.setValueAt(port.toString(), row, column);
                    fireLabelChanged(port, port.getLabel());
                    // }
                    // else {
                    // super.setValueAt(value, row, column);
                    // }
                    break;
                case COLUMN_PULSE_TIME:
                    int pulseTime = (Integer) value;
                    if (port.getPulseTime() != pulseTime) {
                        port.setPulseTime(pulseTime);
                        super.setValueAt(value, row, column);
                        fireConfigValuesChanged(port, PortConfigKeys.BIDIB_PCFG_TICKS);
                    }
                    else {
                        LOGGER.debug("The pulse time has not been changed.");
                    }
                    break;
                case COLUMN_STATUS:
                    port.setStatus((SoundPortStatus) value);
                    super.setValueAt(value, row, column);
                    break;
                case COLUMN_TEST:
                    LOGGER.debug("Status of sound port is updated: {}, port: {}", value, port);
                    if (value instanceof SoundPortStatus) {
                        SoundPortStatus currentStatus = (SoundPortStatus) value;

                        // port.setStatus(currentStatus);
                        super.setValueAt(currentStatus, row, column);
                    }
                    else {
                        LOGGER.warn("Set an invalid value: {}", value);
                        super.setValueAt(value, row, column);
                    }
                    break;
                default:
                    super.setValueAt(value, row, column);
                    break;
            }
        }
        else {
            super.setValueAt(value, row, column);
        }
    }

    @Override
    public Object getValueAt(int row, int column) {
        switch (column) {
            case COLUMN_PORT_IDENTIFIER:
            case COLUMN_LABEL:
            case COLUMN_PULSE_TIME:
            case COLUMN_TEST:
                column = COLUMN_PORT_INSTANCE;
                break;
            default:
                break;
        }
        return super.getValueAt(row, column);
    }

    @Override
    public void notifyPortLabelChanged(final SoundPort port) {
        LOGGER.info("The port label was changed for port: {}", port);

        for (int row = 0; row < getRowCount(); row++) {
            if (port.equals(getValueAt(row, SoundPortTableModel.COLUMN_PORT_INSTANCE))) {
                super.setValueAt(port.toString(), row, SoundPortTableModel.COLUMN_LABEL);

                break;
            }
        }
    }

    @Override
    public void refreshRow(final SoundPort port, final IntConsumer rowConsumer) {
        int row = findRow(port);
        rowConsumer.accept(row);
    }

    private int findRow(final SoundPort port) {
        for (int row = 0; row < getRowCount(); row++) {
            SoundPort current = (SoundPort) getValueAt(row, SoundPortTableModel.COLUMN_PORT_INSTANCE);

            if (port.equals(current)) {
                return row;
            }
        }
        return -1;
    }

    private void fireConfigValuesChanged(SoundPort port, PortConfigKeys... portConfigKeys) {

        LOGGER.info("The values of the port have changed: {}", port);
        // for (SoundPortListener<SoundPortStatus> l : portListeners) {
        // l.valuesChanged(port, portConfigKeys);
        // }
        portListener.configChanged(port, portConfigKeys);

    }

    // @Override
    // public void buttonPressed(int row, int column) {
    // LOGGER.info("The button was pressed, row: {}, column: {}", row, column);
    //
    // // be careful: if we check the column we must evaluate if the SPORT configuration columns are displayed or not
    // // ...
    // final Object portInstance = getValueAt(row, COLUMN_PORT_INSTANCE);
    // if (portInstance instanceof SoundPort) {
    // final SoundPort port = (SoundPort) portInstance;
    // // TODO this should provide the status from the TEST column
    // SoundPortStatus newStatus = (SoundPortStatus) getValueAt(row, COLUMN_TEST);
    // fireTestButtonPressed(port, newStatus);
    // }
    // else {
    // LOGGER.warn("The current portInstance is not a SoundPort: {}", portInstance);
    // }
    // }

    /**
     * Change the port type.
     * 
     * @param portType
     *            the new port type
     * @param port
     *            the port
     */
    public void changePortType(LcOutputType portType, SoundPort port) {

        // for (SoundPortListener<SoundPortStatus> l : portListeners) {
        // l.changePortType(portType, port);
        // }
        portListener.changePortType(portType, port);
    }
}
