package org.bidib.wizard.mvc.main.view.panel;

import java.awt.BorderLayout;
import java.util.List;

import javax.swing.JPanel;
import javax.swing.JScrollPane;
import javax.swing.event.TableModelEvent;
import javax.swing.event.TableModelListener;
import javax.swing.table.TableColumn;

import org.apache.commons.collections4.IterableUtils;
import org.bidib.wizard.api.locale.Resources;
import org.bidib.wizard.api.model.Flag;
import org.bidib.wizard.api.model.NodeInterface;
import org.bidib.wizard.api.model.listener.FlagListListener;
import org.bidib.wizard.client.common.table.AbstractEmptyTable;
import org.bidib.wizard.client.common.view.TabPanelProvider;
import org.bidib.wizard.client.common.view.slider.SliderRenderer;
import org.bidib.wizard.common.script.node.types.TargetType;
import org.bidib.wizard.mvc.main.controller.FlagPanelController;
import org.bidib.wizard.mvc.main.model.FlagTableModel;
import org.bidib.wizard.mvc.main.model.MainModel;
import org.bidib.wizard.mvc.main.model.listener.FlagListener;
import org.bidib.wizard.mvc.main.view.table.FlagEditor;
import org.bidib.wizard.mvc.main.view.table.FlagRenderer;
import org.bidib.wizard.nodescript.script.node.ChangeLabelSupport;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.jgoodies.forms.builder.FormBuilder;
import com.jgoodies.forms.debug.FormDebugPanel;
import com.jgoodies.forms.factories.Paddings;
import com.jidesoft.grid.TableColumnChooser;
import com.jidesoft.swing.DefaultOverlayable;
import com.jidesoft.swing.StyledLabelBuilder;

public class FlagListPanel implements TabPanelProvider, ChangeLabelSupport {

    private static final Logger LOGGER = LoggerFactory.getLogger(FlagListPanel.class);

    private static final String ENCODED_DIALOG_COLUMN_SPECS = "fill:pref:grow, 5dlu, 200dlu";

    private static final String ENCODED_DIALOG_ROW_SPECS = "fill:200dlu:grow";

    private final MainModel mainModel;

    private final JPanel contentPanel;

    private final AbstractEmptyTable table;

    private final FlagTableModel tableModel;

    private final FlagPanelController flagsPanelController;

    private NodeInterface node;

    private final FlagListListener flagListListener;

    public FlagListPanel(final FlagPanelController flagsPanelController, final MainModel mainModel) {
        this.mainModel = mainModel;
        this.flagsPanelController = flagsPanelController;

        // create form builder
        FormBuilder dialogBuilder = null;
        boolean debugDialog = false;
        if (debugDialog) {
            JPanel panel = new FormDebugPanel() {
                private static final long serialVersionUID = 1L;

                @Override
                public String getName() {
                    // this is used as tab title
                    return Resources.getString(FlagListPanel.class, "name");
                }
            };
            dialogBuilder =
                FormBuilder.create().columns(ENCODED_DIALOG_COLUMN_SPECS).rows(ENCODED_DIALOG_ROW_SPECS).panel(panel);
        }
        else {
            JPanel panel = new JPanel(new BorderLayout()) {
                private static final long serialVersionUID = 1L;

                @Override
                public String getName() {
                    // this is used as tab title
                    return Resources.getString(FlagListPanel.class, "name");
                }
            };
            dialogBuilder =
                FormBuilder.create().columns(ENCODED_DIALOG_COLUMN_SPECS).rows(ENCODED_DIALOG_ROW_SPECS).panel(panel);
        }
        dialogBuilder.border(Paddings.TABBED_DIALOG);

        tableModel = new FlagTableModel(mainModel);

        table = new AbstractEmptyTable(tableModel, Resources.getString(getClass(), "emptyTable")) {
            private static final long serialVersionUID = 1L;

            @Override
            public void adjustRowHeight() {
                // set the correct row height
                SliderRenderer sliderEditor =
                    new SliderRenderer(0, 255, 10);

                int rowHeight =
                    sliderEditor.getTableCellRendererComponent(this, 1, false, false, 0, 0).getPreferredSize().height
                        + 6;
                LOGGER.info("Set row height: {}", rowHeight);

                setRowHeight(rowHeight);
            }
        };

        table.putClientProperty("terminateEditOnFocusLost", Boolean.TRUE);
        table.adjustRowHeight();

        // do not allow drag columns to other position
        table.getTableHeader().setReorderingAllowed(false);
        // disabled sorting
        table.setSortable(false);

        TableColumn tc = table.getColumnModel().getColumn(FlagTableModel.COLUMN_LABEL);
        tc.setCellRenderer(new FlagRenderer());

        tc.setCellEditor(new FlagEditor());

        TableColumnChooser.hideColumn(table, FlagTableModel.COLUMN_FLAG_INSTANCE);

        tableModel.addFlagListener(new FlagListener() {

            @Override
            public void labelChanged(int flagId) {
                fireFlagLabelChanged(flagId);
            }
        });

        final DefaultOverlayable overlayTable = new DefaultOverlayable(new JScrollPane(table));
        tableModel.addTableModelListener(new TableModelListener() {
            @Override
            public void tableChanged(TableModelEvent e) {
                overlayTable.setOverlayVisible(tableModel.getRowCount() == 0);
            }
        });

        overlayTable
            .addOverlayComponent(StyledLabelBuilder.createStyledLabel("{" + table.getEmptyTableText() + ":f:gray}"));

        dialogBuilder.add(overlayTable).xyw(1, 1, 3);

        contentPanel = dialogBuilder.build();

        flagListListener = new FlagListListener() {

            @Override
            public void listChanged() {
                LOGGER.info("The flag list has changed.");
                refreshFlags();
            }
        };

    }

    @Override
    public JPanel getComponent() {
        return contentPanel;
    }

    @Override
    public void changeLabel(TargetType portType) {

        LOGGER.info("Change the label of the flag: {}", portType);

        final int flagNum = portType.getPortNum();
        String label = portType.getLabel();

        final NodeInterface node = mainModel.getSelectedNode();
        if (node != null) {
            List<Flag> flags = node.getFlags();

            Flag flag = IterableUtils.find(flags, object -> object.getId() == flagNum);
            LOGGER.info("Change label of flag: {}", flag);

            if (flag != null) {
                flag.setLabel(label);

                fireFlagLabelChanged(flag.getId());
            }
        }

    }

    private void fireFlagLabelChanged(int flagId) {

        Flag flag = (Flag) tableModel.getValueAt(flagId, FlagTableModel.COLUMN_FLAG_INSTANCE);

        flagsPanelController.setFlagLabel(flagId, flag.getLabel());
    }

    public void nodeChanged() {
        LOGGER.debug("The selected node has changed.");

        if (node != null) {
            node.removeFlagListListener(flagListListener);
        }

        node = mainModel.getSelectedNode();
        if (node != null) {
            node.addFlagListListener(flagListListener);
        }

        refreshFlags();
    }

    protected void refreshFlags() {
        LOGGER.debug("refresh the flags.");
        // remove all rows
        tableModel.setRowCount(0);

        final NodeInterface node = mainModel.getSelectedNode();
        if (node != null) {
            List<Flag> flags = node.getFlags();

            for (Flag flag : flags) {

                LOGGER.debug("Adding row for flag: {}", flag);
                tableModel.addRow(flag);
            }
        }
    }

    public void refreshView() {
        refreshFlags();
    }
}
