package org.bidib.wizard.mvc.main.view.table;

import java.beans.PropertyChangeEvent;
import java.util.Objects;
import java.util.concurrent.TimeUnit;
import java.util.function.Consumer;

import javax.swing.JComponent;
import javax.swing.JPanel;
import javax.swing.JTextField;
import javax.swing.SwingUtilities;

import org.bidib.jbidibc.messages.BidibLibrary;
import org.bidib.jbidibc.messages.enums.LcOutputType;
import org.bidib.jbidibc.messages.port.BytePortConfigValue;
import org.bidib.jbidibc.messages.port.Int16PortConfigValue;
import org.bidib.jbidibc.messages.utils.ByteUtils;
import org.bidib.wizard.api.locale.Resources;
import org.bidib.wizard.client.common.component.SliderAndValuePanel;
import org.bidib.wizard.client.common.controller.NodeSelectionProvider;
import org.bidib.wizard.client.common.converter.StringToIntegerConverter;
import org.bidib.wizard.client.common.model.ServoPortTableModel;
import org.bidib.wizard.client.common.rxjava2.SwingScheduler;
import org.bidib.wizard.client.common.table.AbstractPortEditorPanel;
import org.bidib.wizard.client.common.text.InputValidationDocument;
import org.bidib.wizard.client.common.text.WizardBindings;
import org.bidib.wizard.client.common.text.WizardComponentFactory;
import org.bidib.wizard.client.common.view.validation.DefaultRangeValidationCallback;
import org.bidib.wizard.client.common.view.validation.IconFeedbackPanel;
import org.bidib.wizard.client.common.view.validation.IntegerInputValidationDocument;
import org.bidib.wizard.client.common.view.validation.PropertyValidationI18NSupport;
import org.bidib.wizard.model.ports.BacklightPort;
import org.bidib.wizard.model.ports.GenericPort;
import org.bidib.wizard.model.ports.LightPort;
import org.bidib.wizard.model.ports.event.PortConfigChangeEvent;
import org.bidib.wizard.mvc.main.model.BacklightPortTableModel;
import org.bidib.wizard.mvc.main.view.component.BacklightSliderAndValuePanel;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.jgoodies.binding.adapter.Bindings;
import com.jgoodies.binding.value.BindingConverter;
import com.jgoodies.binding.value.BufferedValueModel;
import com.jgoodies.binding.value.ConverterValueModel;
import com.jgoodies.binding.value.ValueHolder;
import com.jgoodies.binding.value.ValueModel;
import com.jgoodies.forms.builder.FormBuilder;
import com.jgoodies.forms.factories.Paddings;
import com.jgoodies.validation.ValidationResult;
import com.jgoodies.validation.util.PropertyValidationSupport;
import com.jgoodies.validation.view.ValidationComponentUtils;

import io.reactivex.rxjava3.disposables.Disposable;
import io.reactivex.rxjava3.subjects.PublishSubject;

public class BacklightPortEditorPanel extends AbstractPortEditorPanel<BacklightPort> {

    private static final Logger LOGGER = LoggerFactory.getLogger(BacklightPortEditorPanel.class);

    private static final String ENCODED_DIALOG_COLUMN_SPECS =
        "pref, 3dlu, pref, 3dlu, pref, 3dlu, pref, 3dlu, fill:50dlu:grow";

    private static final String ENCODED_DIALOG_ROW_SPECS =
        "pref, 3dlu, pref, 3dlu, pref, 3dlu, pref, 3dlu, pref, 3dlu, pref, 3dlu, pref, 3dlu, pref, 3dlu, pref";

    private static final int TIME_BETWEEN_VALUE_EVENTS_MILLIS = 80;

    private static final int TIME_BETWEEN_PORT_CONFIG_EVENTS_MILLIS = 100;

    private ValueModel dimmDownConverterModel;

    private ValueModel dimmUpConverterModel;

    private BufferedValueModel dimmUpBufferedValueModel;

    private BufferedValueModel dimmDownBufferedValueModel;

    private ValueModel dmxMappingConverterModel;

    private PublishSubject<Integer> valueEventSubject = PublishSubject.create();

    private final PublishSubject<PortConfigChangeEvent> localPortConfigChangeEventSubject = PublishSubject.create();

    private final PublishSubject<PortConfigChangeEvent> portConfigChangeEventSubject;

    public BacklightPortEditorPanel(BacklightPort port, Consumer<BacklightPort> saveCallback,
                                    final Consumer<BacklightPort> valueCallback, final Consumer<BacklightPort> refreshCallback,
                                    final PublishSubject<PortConfigChangeEvent> portConfigChangeEventSubject, final NodeSelectionProvider nodeSelectionProvider) {
        super(port, saveCallback, valueCallback, refreshCallback, nodeSelectionProvider);

        this.portConfigChangeEventSubject = portConfigChangeEventSubject;
    }

    @Override
    protected BacklightPort clonePort(final BacklightPort port) {
        // create a clone of the input port
        final BacklightPort clone =
            BacklightPort
                .builder() //
                .withDimSlopeUp(port.getDimSlopeUp()) //
                .withDimSlopeDown(port.getDimSlopeDown()) //
                .withDimStretchMax(port.getDimStretchMax()) //
                .withDimStretchMin(port.getDimStretchMin()) //
                .withDmxMapping(port.getDmxMapping()) //
                .withStatus(port.getStatus()) //
                .withRemappingEnabled(port.isRemappingEnabled()) //
                .withKnownPortConfigKeys(port.getKnownPortConfigKeys()) //
                .withId(port.getId()) //
                .withLabel(port.getLabel()) //
                .withEnabled(port.isEnabled()) //
                .withIsInactive(port.isInactive()) //
                .withPortIdentifier(port.getPortIdentifier()).build();
        return clone;
    }

    final ValueModel portValueHolder = new ValueHolder();

    @Override
    protected JPanel doCreateComponent(final BacklightPort port) {

        int row = 1;
        FormBuilder dialogBuilder = null;
        boolean debugDialog = false;
        if (debugDialog) {
            JPanel panel = new PortEditorPanelDebugContainer(this);
            dialogBuilder =
                FormBuilder.create().columns(ENCODED_DIALOG_COLUMN_SPECS).rows(ENCODED_DIALOG_ROW_SPECS).panel(panel);
        }
        else {
            JPanel panel = new PortEditorPanelContainer(this);
            dialogBuilder =
                FormBuilder.create().columns(ENCODED_DIALOG_COLUMN_SPECS).rows(ENCODED_DIALOG_ROW_SPECS).panel(panel);
        }
        dialogBuilder.border(Paddings.TABBED_DIALOG);

        // port name
        final BufferedValueModel bufferedPortNameModel =
            getPresentationModel().getBufferedModel(BacklightPort.PROPERTY_LABEL);

        dialogBuilder.add(Resources.getString(BacklightPortTableModel.class, "label") + ":").xy(1, row);
        final JTextField portName = WizardComponentFactory.createTextField(bufferedPortNameModel, false);
        dialogBuilder.add(portName).xyw(3, row, 7);

        row += 2;

        // if dimm min is not available we must not show it here
        if (isPortConfigKeySupported(BidibLibrary.BIDIB_PCFG_DIMM_DOWN)
            || isPortConfigKeySupported(BidibLibrary.BIDIB_PCFG_DIMM_DOWN_8_8)) {

            int maxRange = isPortConfigKeySupported(BidibLibrary.BIDIB_PCFG_DIMM_DOWN_8_8) ? 65535 : 255;

            // final BufferedValueModel valueValueModel =
            this.dimmDownBufferedValueModel =
                getPresentationModel().getBufferedModel(BacklightPort.PROPERTY_DIM_SLOPE_DOWN);

            // dimm min
            this.dimmDownConverterModel =
                new ConverterValueModel(this.dimmDownBufferedValueModel, new StringToIntegerConverter());

            JTextField dimmDownText = new JTextField();
            final IntegerInputValidationDocument dimmDownDocument =
                new IntegerInputValidationDocument(5, InputValidationDocument.NUMERIC);
            dimmDownDocument.setRangeValidationCallback(new DefaultRangeValidationCallback(0, maxRange));
            dimmDownText.setDocument(dimmDownDocument);
            dimmDownText.setColumns(5);
            dimmDownText.setHorizontalAlignment(JTextField.RIGHT);

            // bind manually because we changed the document of the textfield
            Bindings.bind(dimmDownText, dimmDownConverterModel, false);

            dialogBuilder.add(Resources.getString(BacklightPortTableModel.class, "dimSlopeDown")).xy(1, row);
            dialogBuilder.add(dimmDownText).xy(3, row);

            dimmDownText.setEnabled(port.isEnabled());

            ValidationComponentUtils.setMandatory(dimmDownText, true);
            ValidationComponentUtils.setMessageKeys(dimmDownText, "validation.dimmDown_key");

            // use the clone port to provide the values
            final SliderAndValuePanel valueSlider = new SliderAndValuePanel(0, maxRange, 10);
            valueSlider.createComponent();
            WizardBindings.bind(valueSlider, this.dimmDownBufferedValueModel);

            dialogBuilder.add(valueSlider).xyw(5, row, 5);

            final boolean dimm_8_8 = isPortConfigKeySupported(BidibLibrary.BIDIB_PCFG_DIMM_DOWN_8_8);

            this.dimmDownBufferedValueModel.addValueChangeListener(evt -> {
                if (this.dimmDownBufferedValueModel.getValue() instanceof Integer) {
                    int dimmDownValue = ((Integer) this.dimmDownBufferedValueModel.getValue()).intValue();
                    LOGGER.info("The dimmDown value has been changed: {}", dimmDownValue);

                    this.localPortConfigChangeEventSubject
                            .onNext(new PortConfigChangeEvent(null, 0L, port, dimm_8_8 ? BidibLibrary.BIDIB_PCFG_DIMM_DOWN_8_8 : BidibLibrary.BIDIB_PCFG_DIMM_DOWN,
                                    dimm_8_8 ? new Int16PortConfigValue(ByteUtils.getWORD(dimmDownValue)) : new BytePortConfigValue(ByteUtils.getLowByte(dimmDownValue))));
                }
            });

            row += 2;
        }

        // if dimm max is not available we must not show it here
        if (isPortConfigKeySupported(BidibLibrary.BIDIB_PCFG_DIMM_UP)
            || isPortConfigKeySupported(BidibLibrary.BIDIB_PCFG_DIMM_UP_8_8)) {

            int maxRange = isPortConfigKeySupported(BidibLibrary.BIDIB_PCFG_DIMM_UP_8_8) ? 65535 : 255;

            // final BufferedValueModel valueValueModel =
            this.dimmUpBufferedValueModel =
                getPresentationModel().getBufferedModel(BacklightPort.PROPERTY_DIM_SLOPE_UP);

            // dimm max
            this.dimmUpConverterModel =
                new ConverterValueModel(this.dimmUpBufferedValueModel, new StringToIntegerConverter());

            JTextField dimmUpText = new JTextField();
            final IntegerInputValidationDocument dimmUpDocument =
                new IntegerInputValidationDocument(5, InputValidationDocument.NUMERIC);
            dimmUpDocument.setRangeValidationCallback(new DefaultRangeValidationCallback(0, maxRange));
            dimmUpText.setDocument(dimmUpDocument);
            dimmUpText.setColumns(5);
            dimmUpText.setHorizontalAlignment(JTextField.RIGHT);

            // bind manually because we changed the document of the textfield
            Bindings.bind(dimmUpText, dimmUpConverterModel, false);

            dialogBuilder.add(Resources.getString(BacklightPortTableModel.class, "dimSlopeUp")).xy(1, row);
            dialogBuilder.add(dimmUpText).xy(3, row);

            dimmUpText.setEnabled(port.isEnabled());

            ValidationComponentUtils.setMandatory(dimmUpText, true);
            ValidationComponentUtils.setMessageKeys(dimmUpText, "validation.dimmUp_key");

            // use the clone port to provide the values
            final SliderAndValuePanel valueSlider = new SliderAndValuePanel(0, maxRange, 10);
            valueSlider.createComponent();
            WizardBindings.bind(valueSlider, this.dimmUpBufferedValueModel);

            dialogBuilder.add(valueSlider).xyw(5, row, 5);

            final boolean dimm_8_8 = isPortConfigKeySupported(BidibLibrary.BIDIB_PCFG_DIMM_UP_8_8);

            this.dimmUpBufferedValueModel.addValueChangeListener(evt -> {
                if (this.dimmUpBufferedValueModel.getValue() instanceof Integer) {
                    int dimmUpValue = ((Integer) this.dimmUpBufferedValueModel.getValue()).intValue();

                    LOGGER.info("The dimmUp value has been changed: {}", dimmUpValue);

                    this.localPortConfigChangeEventSubject
                            .onNext(new PortConfigChangeEvent(null, 0L, port, dimm_8_8 ? BidibLibrary.BIDIB_PCFG_DIMM_UP_8_8 : BidibLibrary.BIDIB_PCFG_DIMM_UP,
                                    dimm_8_8 ? new Int16PortConfigValue(ByteUtils.getWORD(dimmUpValue)) : new BytePortConfigValue(ByteUtils.getLowByte(dimmUpValue))));
                }
            });

            row += 2;
        }

        // if DMX mapping is not available we must not show it here
        if (isPortConfigKeySupported(BidibLibrary.BIDIB_PCFG_OUTPUT_MAP)) {
            final BufferedValueModel valueValueModel = getPresentationModel().getBufferedModel(BacklightPort.PROPERTY_DMX_MAPPING);
            this.dmxMappingConverterModel = new ConverterValueModel(valueValueModel, new StringToIntegerConverter());

            JTextField dmxMappingText = new JTextField();
            final IntegerInputValidationDocument dmxMappingDocument =
                new IntegerInputValidationDocument(3, InputValidationDocument.NUMERIC);
            dmxMappingDocument.setRangeValidationCallback(new DefaultRangeValidationCallback(MIN_DMX_MAPPING_VALUE, 255));
            dmxMappingText.setDocument(dmxMappingDocument);
            dmxMappingText.setColumns(2);
            dmxMappingText.setHorizontalAlignment(JTextField.RIGHT);

            // bind manually because we changed the document of the textfield
            Bindings.bind(dmxMappingText, dmxMappingConverterModel, false);

            dialogBuilder.add(Resources.getString(BacklightPortTableModel.class, "dmxMapping")).xy(1, row);
            dialogBuilder.add(dmxMappingText).xy(3, row);

            ValidationComponentUtils.setMandatory(dmxMappingText, true);
            ValidationComponentUtils.setMessageKeys(dmxMappingText, "validation.dmxMapping_key");

            valueValueModel.addValueChangeListener(evt -> {
                if (valueValueModel.getValue() instanceof Integer) {
                    int outputMap = ((Integer) valueValueModel.getValue()).intValue();

                    LOGGER.info("The outputMap value has been changed: {}", outputMap);

                    this.localPortConfigChangeEventSubject
                            .onNext(new PortConfigChangeEvent(null, 0L, port, BidibLibrary.BIDIB_PCFG_OUTPUT_MAP,
                                    new BytePortConfigValue(ByteUtils.getLowByte(outputMap))));
                }
            });

            row += 2;
        }

        // value slider

        // add value slider
        final BufferedValueModel valueValueModel =
            getPresentationModel().getBufferedModel(BacklightPort.PROPERTYNAME_VALUE);

        final ValueModel valueConverterModel =
            new ConverterValueModel(valueValueModel, new BacklightValueToRelativeValueConverter());

        // create a holder for the port value
        portValueHolder.setValue(valueConverterModel.getValue());

        // use the clone port to provide the values
        final BacklightSliderAndValuePanel valueSlider = new BacklightSliderAndValuePanel(0, 100, 10, () -> {

            int dimmUp = 255;
            int dimmDown = 0;

            final BacklightPort configPort =
                BacklightPort
                    .builder() //
                    .withDimSlopeUp(dimmUp) //
                    .withDimSlopeDown(dimmDown)//
                    .build();
            return configPort;
        });
        valueSlider.createComponent();
        WizardBindings.bind(valueSlider, portValueHolder);

        portValueHolder.addValueChangeListener(evt -> {

            if (Objects.equals(evt.getOldValue(), evt.getNewValue())) {
                LOGGER.debug("The value has not changed.");
                return;
            }

            if (portValueHolder.getValue() instanceof Integer) {
                int position = ((Integer) portValueHolder.getValue()).intValue();

                LOGGER.info("The position has been changed by the slider. New position: {}", position);

                valueEventSubject.onNext(position);
            }
        });

        final Disposable disp =
            this.valueEventSubject
                .throttleLatest(TIME_BETWEEN_VALUE_EVENTS_MILLIS, TimeUnit.MILLISECONDS, SwingScheduler.getInstance(),
                    true)
                .subscribe(value -> sendValueToBacklightPort(port, value), err -> {
                    LOGGER.warn("The value event subject signalled an error.", err);
                }, () -> {
                    LOGGER.info("The value event subject has completed.");
                });
        getCompDisp().add(disp);

        dialogBuilder.add(Resources.getString(ServoPortTableModel.class, "value")).xy(1, 9);
        dialogBuilder.add(valueSlider).xyw(3, 9, 7);

        row += 2;

        // add buttons
        final JPanel buttonPanel = createButtonPanel();
        dialogBuilder.add(buttonPanel).xyw(1, row, 9);

        // combine the port config events
        final Disposable dispLocalPortConfigChange =
                this.localPortConfigChangeEventSubject
                        .buffer(TIME_BETWEEN_PORT_CONFIG_EVENTS_MILLIS, TimeUnit.MILLISECONDS, SwingScheduler.getInstance())
                        .subscribe(evts -> {
                            // combine the values
                            PortConfigChangeEvent event = null;
                            for (PortConfigChangeEvent evt : evts) {
                                LOGGER.info("Process event: {}", evt);
                                if (event == null) {
                                    event = evt;
                                    continue;
                                }
                                event.getPortConfig().putAll(evt.getPortConfig());
                            }

                            if (event != null) {
                                LOGGER.info("Publish the config change event to the node: {}", event);
                                portConfigChangeEventSubject.onNext(event);
                            }
                        });
        getCompDisp().add(dispLocalPortConfigChange);

        // check if we have validation enabled
        if (getValidationResultModel() != null) {
            LOGGER.debug("Create iconfeedback panel.");
            JComponent cvIconPanel = new IconFeedbackPanel(getValidationResultModel(), dialogBuilder.build());
            JPanel panel = new PortEditorPanelContainer(this);
            FormBuilder feedbackBuilder = FormBuilder.create().columns("p:g").rows("fill:p:grow").panel(panel);

            feedbackBuilder.add(cvIconPanel).xy(1, 1);

            setPanel(feedbackBuilder.build());
        }
        else {
            setPanel(dialogBuilder.build());
        }

        bufferedPortNameModel.addValueChangeListener(evt -> triggerValidation());
        if (this.dimmDownConverterModel != null) {
            this.dimmDownConverterModel.addValueChangeListener(evt -> triggerValidation());
        }
        if (this.dimmUpConverterModel != null) {
            this.dimmUpConverterModel.addValueChangeListener(evt -> triggerValidation());
        }
        if (this.dmxMappingConverterModel != null) {
            this.dmxMappingConverterModel.addValueChangeListener(evt -> triggerValidation());
        }

        triggerValidation();

        return getPanel();
    }

    @Override
    protected void propertyChanged(final PropertyChangeEvent evt) {
        LOGGER
                .info("The port property has been changed, propertyName: {}, new value: {}", evt.getPropertyName(),
                        evt.getNewValue());

        super.propertyChanged(evt);

        SwingUtilities.invokeLater(() -> {
            try {
                // the status of the port was changed
                switch (evt.getPropertyName()) {
                    case LightPort.PROPERTY_STATUS:
                    case GenericPort.PROPERTY_PORT_STATUS:
                        updatePortValueFromOriginalPort();
                        break;
                    case GenericPort.PROPERTY_PORT_CONFIG_CHANGED:
                        break;
                    case GenericPort.PROPERTY_PORT_TYPE_CHANGED:
                        LOGGER.info("The port type has changed: {}", evt.getNewValue());
                        if (getOriginalPort().getPortType() == LcOutputType.SERVOPORT) {
                            LOGGER.info("Current port type is SERVOPORT.");
                            updatePortValueFromOriginalPort();
                            enableComponents();
                        }
                        break;

                    default:
                        break;
                }
            }
            catch (Exception ex) {
                LOGGER.warn("Update the status failed.", ex);
            }
        });
    }

    private void updatePortValueFromOriginalPort() {
        final ValueModel selectionHolderPortStatus =
                getPresentationModel().getModel(LightPort.PROPERTY_STATUS);
        Integer value = getOriginalPort().getValue();
        LOGGER.info("Current value of original port: {}", value);
        selectionHolderPortStatus.setValue(value);
        this.portValueHolder.setValue(value);
    }

    @Override
    protected ValidationResult validate(final BacklightPort port) {
        PropertyValidationSupport support = new PropertyValidationI18NSupport(getPresentationModel(), "validation");

        if (this.dimmDownConverterModel != null && this.dimmDownConverterModel.getValue() == null) {
            support.addError("dimmDownTime_key", "not_empty_for_write");
        }
        if (this.dimmUpConverterModel != null && this.dimmUpConverterModel.getValue() == null) {
            support.addError("dimmUp_key", "not_empty_for_write");
        }
        if (this.dmxMappingConverterModel != null && this.dmxMappingConverterModel.getValue() == null) {
            support.addError("dmxMapping_key", "not_empty_for_write");
        }

        ValidationResult validationResult = support.getResult();
        return validationResult;
    }

    private void sendValueToBacklightPort(final BacklightPort port, int position) {
        LOGGER.info("Send the position to the backlight port: {}", position);

        triggerValidation();

        final BacklightPort servoPort = BacklightPort.builder().withValue(position).withId(port.getId()).build();

        getValueCallback().accept(servoPort);
    }

    private static final class BacklightValueToRelativeValueConverter implements BindingConverter<Integer, Integer> {

        @Override
        public Integer targetValue(Integer sourceValue) {
            if (sourceValue != null) {
                int val = sourceValue.intValue();

                val = BacklightPort.getRelativeValue(val);
                LOGGER.trace("Converted source to target value: {} -> {}", sourceValue, val);

                return Integer.valueOf(val);
            }
            return null;
        }

        @Override
        public Integer sourceValue(Integer targetValue) {
            if (targetValue != null) {
                int val = targetValue.intValue();

                val = BacklightPort.getAbsoluteValue(val);
                LOGGER.trace("Converted target to source value: {} -> {}", targetValue, val);

                return Integer.valueOf(val);
            }
            return null;
        }
    }

}
