package org.bidib.wizard.mvc.loco.view;

import java.awt.BorderLayout;
import java.awt.Color;
import java.awt.Cursor;
import java.awt.Dimension;
import java.awt.Graphics;
import java.awt.GridBagLayout;
import java.awt.Point;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.awt.event.KeyEvent;
import java.awt.event.MouseAdapter;
import java.awt.event.MouseEvent;
import java.beans.PropertyChangeEvent;
import java.beans.PropertyChangeListener;
import java.text.DecimalFormat;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.HashMap;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;

import javax.swing.AbstractAction;
import javax.swing.BorderFactory;
import javax.swing.ImageIcon;
import javax.swing.InputMap;
import javax.swing.JButton;
import javax.swing.JComboBox;
import javax.swing.JComponent;
import javax.swing.JLabel;
import javax.swing.JLayeredPane;
import javax.swing.JPanel;
import javax.swing.JRadioButton;
import javax.swing.JSlider;
import javax.swing.JTextField;
import javax.swing.KeyStroke;
import javax.swing.SwingUtilities;
import javax.swing.Timer;
import javax.swing.UIManager;
import javax.swing.border.EmptyBorder;
import javax.swing.event.ChangeEvent;
import javax.swing.event.ChangeListener;
import javax.swing.event.DocumentEvent;
import javax.swing.event.DocumentListener;
import javax.swing.plaf.basic.BasicSliderUI;

import org.apache.commons.collections4.CollectionUtils;
import org.apache.commons.collections4.Predicate;
import org.bidib.wizard.api.locale.Resources;
import org.bidib.wizard.client.common.converter.StringConverter;
import org.bidib.wizard.client.common.text.InputValidationDocument;
import org.bidib.wizard.client.common.text.IntegerRangeFilter;
import org.bidib.wizard.client.common.text.WizardComponentFactory;
import org.bidib.wizard.client.common.view.TabPanelProvider;
import org.bidib.wizard.common.script.loco.LocoViewScripting;
import org.bidib.wizard.common.utils.ImageUtils;
import org.bidib.wizard.core.service.SettingsService;
import org.bidib.wizard.model.status.DirectionStatus;
import org.bidib.wizard.model.status.SpeedSteps;
import org.bidib.wizard.mvc.common.view.ViewCloseListener;
import org.bidib.wizard.mvc.common.view.graph.LedBarGraph;
import org.bidib.wizard.mvc.common.view.graph.LedBarGraph.Orientation;
import org.bidib.wizard.mvc.common.view.panel.DisabledPanel;
import org.bidib.wizard.client.common.view.slider.JZeroSlider;
import org.bidib.wizard.mvc.loco.model.BinStateValue;
import org.bidib.wizard.mvc.loco.model.LocoModel;
import org.bidib.wizard.mvc.loco.model.RfBasisMode;
import org.bidib.wizard.mvc.loco.model.listener.LocoModelListener;
import org.bidib.wizard.mvc.loco.view.listener.LocoViewListener;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.jgoodies.binding.adapter.Bindings;
import com.jgoodies.binding.beans.PropertyAdapter;
import com.jgoodies.binding.beans.PropertyConnector;
import com.jgoodies.binding.value.ConverterValueModel;
import com.jgoodies.binding.value.ValueModel;
import com.jgoodies.forms.FormsSetup;
import com.jgoodies.forms.builder.ButtonBarBuilder;
import com.jgoodies.forms.builder.FormBuilder;
import com.jgoodies.forms.debug.FormDebugPanel;
import com.jgoodies.forms.factories.Paddings;
import com.jidesoft.swing.JideButton;
import com.jidesoft.swing.JideToggleButton;

import eu.hansolo.steelseries.gauges.Radial;

public class LocoView implements LocoViewScripting, TabPanelProvider {

    private static final Logger LOGGER = LoggerFactory.getLogger(LocoView.class);

    private static final String ENCODED_DIALOG_COLUMN_SPECS = "pref, 3dlu, pref:grow";

    private static final String ENCODED_DIALOG_ROW_SPECS = "pref, 3dlu, pref, 3dlu, pref, 9dlu, pref, 3dlu, pref";

    private static final int PAGE_STEPS = 12;

    private LocoModel locoModel;

    private JComboBox<SpeedSteps> speedStepsCombo;

    private JTextField address;

    private JLabel speed;

    private JSlider speedSlider;

    private JLabel reportedSpeed;

    private JButton stopButton;

    private JButton stopEmergencyButton;

    private Map<String, JButton> functionButtonMap = new HashMap<>();

    private final List<LocoViewListener> locoViewListeners = new LinkedList<LocoViewListener>();

    private final List<ViewCloseListener> viewCloseListeners = new LinkedList<ViewCloseListener>();

    private LocoModelListener locoModelListener;

    private JPanel directionPanel;

    private JPanel lightAndStopButtonPanel;

    private JPanel functionButtonPanel;

    private JPanel multiRfBaseButtonPanel;

    private JPanel counterPanel;

    private JPanel directBinStatePanel;

    private ScriptPanel scriptPanel;

    private LedBarGraph ledBarGraph;

    private JPanel contentPanel;

    private boolean m4SupportEnabled;

    // private JPanel imageOrVideoContainerPanel;

    // private JPanel imagePanel;

    public LocoView(final LocoModel locoModel, final SettingsService settingsService) {
        this.locoModel = locoModel;

        m4SupportEnabled = settingsService.getWizardSettings().isM4SupportEnabled();

        // create form builder
        FormBuilder formBuilder = null;
        boolean debugDialog = false;
        if (debugDialog) {
            JPanel panel = new FormDebugPanel() {
                private static final long serialVersionUID = 1L;

                @Override
                public String getName() {
                    // this is used as tab title
                    return Resources
                        .getString(LocoView.class, "title." + (locoModel.isCarControlEnabled() ? "car" : "loco"));
                }

                @Override
                public Dimension getPreferredSize() {
                    return new Dimension(super.getPreferredSize().width + 20, super.getPreferredSize().height);
                }
            };
            formBuilder =
                FormBuilder.create().columns(ENCODED_DIALOG_COLUMN_SPECS).rows(ENCODED_DIALOG_ROW_SPECS).panel(panel);
        }
        else {
            JPanel panel = new JPanel(new BorderLayout()) {

                private static final long serialVersionUID = 1L;

                @Override
                public String getName() {
                    // this is used as tab title
                    return Resources
                        .getString(LocoView.class, "title." + (locoModel.isCarControlEnabled() ? "car" : "loco"));
                }
            };
            formBuilder =
                FormBuilder.create().columns(ENCODED_DIALOG_COLUMN_SPECS).rows(ENCODED_DIALOG_ROW_SPECS).panel(panel);
        }
        formBuilder.border(Paddings.TABBED_DIALOG);

        try {
            formBuilder.add(createLocoAddressPanel()).xy(1, 1);
        }
        catch (Exception ex) {
            LOGGER.warn("Create loco address panel failed.", ex);
        }

        try {
            formBuilder.add(directionPanel = createDirectionPanel()).xy(1, 3);
            DisabledPanel.disable(directionPanel);

            formBuilder.add(createSpeedGaugePanel()).xy(3, 3);
        }
        catch (Exception ex) {
            LOGGER.warn("Create direction panel failed.", ex);
        }

        final List<JideButton> functionButtons = new LinkedList<JideButton>();

        try {
            formBuilder.add(lightAndStopButtonPanel = createLightAndStopButtonPanel(functionButtons)).xyw(1, 5, 3);
            DisabledPanel.disable(lightAndStopButtonPanel);
        }
        catch (Exception ex) {
            LOGGER.warn("Create light and buttons panel failed.", ex);
        }

        int row = 11;
        try {
            formBuilder.add(functionButtonPanel = createFunctionButtonPanel(functionButtons)).xyw(1, 7, 1);
            DisabledPanel.disable(functionButtonPanel);

            // TODO
            // imageOrVideoContainerPanel = new JPanel();
            // imageOrVideoContainerPanel.setOpaque(false);
            // formBuilder.add(imageOrVideoContainerPanel).xyw(3, 7, 1);
            //
            // imagePanel = createImagePanel(functionButtons);
            // if (locoModel.isCarControlEnabled()) {
            // imageOrVideoContainerPanel.add(imagePanel);
            // }

            if (locoModel.isCarControlEnabled()) {
                // add the RF basis buttons
                this.multiRfBaseButtonPanel = addMultiRfBasisButtons();
                formBuilder.add(this.multiRfBaseButtonPanel).xyw(1, 9, 1);
                DisabledPanel.disable(this.multiRfBaseButtonPanel);

                formBuilder.add(counterPanel = addCounterPanel()).xyw(3, 9, 1);
                DisabledPanel.disable(counterPanel);

                if (settingsService.getWizardSettings().isPowerUser()) {
                    formBuilder.appendRows("3dlu, pref");
                    formBuilder.add(directBinStatePanel = addDirectBinStatePanel()).xy(1, row);
                    DisabledPanel.disable(directBinStatePanel);

                    row += 2;
                }
            }
            else {
                LOGGER.info("No multi RF basis buttons added.");
            }

        }
        catch (Exception ex) {
            LOGGER.warn("Create function buttons panel failed.", ex);
        }

        // prepare the script panel
        scriptPanel = new ScriptPanel(this, settingsService);
        addLocoViewListener(scriptPanel);
        JPanel panel = scriptPanel.createPanel();

        panel
            .setBorder(BorderFactory
                .createTitledBorder(BorderFactory.createEtchedBorder(),
                    Resources.getString(getClass(), "script") + ":"));

        formBuilder.appendRows("3dlu, pref");
        formBuilder.add(panel).xyw(1, row, 3);

        // add loco model listener
        locoModelListener = new LocoModelListener() {

            @Override
            public void directionChanged(DirectionStatus direction) {
            }

            @Override
            public void functionChanged(int index, boolean value) {
                functionButtons.get(index).setSelected(value);
            }

            @Override
            public void speedStepsChanged(SpeedSteps speedSteps) {
                LOGGER.info("The speed steps have changed: {}", speedSteps);
                int steps = speedSteps.getSteps() - 1;
                speedSlider.setMaximum(steps);
                if (locoModel.isCarControlEnabled()) {
                    speedSlider.setMinimum(0);
                }
                else {
                    speedSlider.setMinimum(-steps);
                }
                speedSlider.setMajorTickSpacing(steps);
            }

            @Override
            public void dynStateEnergyChanged(final int dynStateEnergy) {

                LOGGER.info("The dynStateEnergy has changed, dynStateEnergy: {}", dynStateEnergy);

                SwingUtilities.invokeLater(new Runnable() {

                    @Override
                    public void run() {

                        if (ledBarGraph != null) {
                            ledBarGraph.setValue(dynStateEnergy);
                        }
                    }
                });

            }

            @Override
            public void binaryStateChanged(int state, boolean value) {

            }
        };
        locoModel.addLocoModelListener(locoModelListener);

        locoModel.addPropertyChangeListener(new PropertyChangeListener() {

            @Override
            public void propertyChange(PropertyChangeEvent evt) {
                switch (evt.getPropertyName()) {
                    case LocoModel.PROPERTYNAME_ADDRESS:
                        locoModel.setReportedSpeed(0);
                        locoModel.setDynStateEnergy(0);
                        break;
                    case LocoModel.PROPERTYNAME_SPEED:
                        try {
                            Integer speedValue = (Integer) evt.getNewValue();
                            if (speedValue != null) {
                                int speed = speedValue.intValue();
                                if (speed == 0 || speed == 1) {
                                    LOGGER.info("The current speed is stop or emergengy stop: {}", speed);
                                    changeSliderSilently(0);
                                }
                            }
                        }
                        catch (Exception ex) {
                            LOGGER.warn("Set speed failed.", ex);
                        }
                        break;
                    default:
                        break;
                }
            }
        });

        contentPanel = formBuilder.build();

        final EmptyAddressHandler handler = new EmptyAddressHandler(value -> {
            LOGGER.info("The address value is not empty: {}", value);
            if (value.booleanValue()) {
                DisabledPanel.enable(directionPanel);
                DisabledPanel.enable(lightAndStopButtonPanel);
                DisabledPanel.enable(functionButtonPanel);
                if (multiRfBaseButtonPanel != null) {
                    DisabledPanel.enable(multiRfBaseButtonPanel);
                }
                if (counterPanel != null) {
                    DisabledPanel.enable(counterPanel);
                }
                if (directBinStatePanel != null) {
                    DisabledPanel.enable(directBinStatePanel);
                }
            }
            else {
                DisabledPanel.disable(directionPanel);
                DisabledPanel.disable(lightAndStopButtonPanel);
                DisabledPanel.disable(functionButtonPanel);
                if (multiRfBaseButtonPanel != null) {
                    DisabledPanel.disable(multiRfBaseButtonPanel);
                }
                if (counterPanel != null) {
                    DisabledPanel.disable(counterPanel);
                }
                if (directBinStatePanel != null) {
                    DisabledPanel.disable(directBinStatePanel);
                }
            }
        });
        PropertyConnector
            .connect(locoModel, LocoModel.PROPERTYNAME_ADDRESS, handler, EmptyAddressHandler.PROPERTYNAME_ADDRESS)
            .updateProperty2();
    }

    private void changeSliderSilently(int speed) {
        ChangeListener[] changeListeners = speedSlider.getChangeListeners();
        // remove the change listeners to prevent signal the speed change to the interface
        // twice
        for (ChangeListener listener : changeListeners) {
            speedSlider.removeChangeListener(listener);
        }

        try {
            speedSlider.setValue(speed);
        }
        finally {
            // add the change listeners again
            for (ChangeListener listener : changeListeners) {
                speedSlider.addChangeListener(listener);
            }
        }
    }

    @Override
    public JPanel getComponent() {
        return contentPanel;
    }

    private JPanel createLocoAddressPanel() {
        final FormBuilder formBuilder =
            FormBuilder.create().columns("pref, 3dlu, 60dlu, 10dlu, pref, 3dlu, 30dlu, 3dlu:grow").rows("pref");

        address = new JTextField();
        address.setDocument(new InputValidationDocument(InputValidationDocument.NUMERIC));

        // set the default address
        if (locoModel.getAddress() != null) {
            address.setText(Integer.toString(locoModel.getAddress()));
        }
        else {
            address.setText(null);
        }

        address.getDocument().addDocumentListener(new DocumentListener() {
            @Override
            public void changedUpdate(DocumentEvent e) {
                valueChanged();
            }

            @Override
            public void removeUpdate(DocumentEvent e) {
                valueChanged();
            }

            @Override
            public void insertUpdate(DocumentEvent e) {
                valueChanged();
            }

            private void valueChanged() {
                Integer currentAddress = null;

                if (!address.getText().isEmpty()) {
                    try {
                        currentAddress = Integer.valueOf(address.getText());
                    }
                    catch (NumberFormatException e) {
                        LOGGER.warn("Parse decoder address failed.");
                    }
                }
                if (locoModel != null) {
                    LOGGER.info("Set the new address: {}", currentAddress);
                    locoModel.setAddress(currentAddress);

                    // must reset the reported cell number
                    locoModel.setReportedCellNumber(null);
                }
            }
        });

        formBuilder.add(Resources.getString(getClass(), "address")).xy(1, 1);
        formBuilder.add(address).xy(3, 1);

        if (m4SupportEnabled) {
            speedStepsCombo = new JComboBox<SpeedSteps>(SpeedSteps.values());
        }
        else {
            // filter the M4 speed steps

            List<SpeedSteps> speedSteps = new ArrayList<>();
            speedSteps.addAll(Arrays.asList(SpeedSteps.values()));
            CollectionUtils.filter(speedSteps, new Predicate<SpeedSteps>() {

                @Override
                public boolean evaluate(SpeedSteps speedStep) {
                    return (speedStep.ordinal() <= SpeedSteps.DCC_SDF.ordinal());
                }
            });

            speedStepsCombo = new JComboBox<SpeedSteps>(speedSteps.toArray(new SpeedSteps[0]));
        }

        speedStepsCombo.setSelectedItem(locoModel.getSpeedSteps());

        speedStepsCombo.addActionListener(new ActionListener() {
            @Override
            public void actionPerformed(ActionEvent e) {
                LOGGER.info("Speed steps combo has changed, update loco model.");
                locoModel.setSpeedSteps((SpeedSteps) speedStepsCombo.getSelectedItem());
            }
        });

        formBuilder.add(Resources.getString(getClass(), "speedSteps")).xy(5, 1);
        formBuilder.add(speedStepsCombo).xyw(7, 1, 2);

        return formBuilder.build();
    }

    private JPanel createDirectionPanel() {
        final FormBuilder formBuilder =
            FormBuilder
                .create().columns("pref, 3dlu, 60dlu:grow, 3dlu, pref")
                .rows("pref, 3dlu, pref, 3dlu, pref, 3dlu, pref");

        formBuilder.border(new EmptyBorder(10, 0, 10, 0));

        ValueModel speedModel = new PropertyAdapter<LocoModel>(locoModel, LocoModel.PROPERTYNAME_SPEED, true);
        final ValueModel speedConverterModel =
            new ConverterValueModel(speedModel, new StringConverter(new DecimalFormat("#")));

        speed = WizardComponentFactory.createLabel(speedConverterModel);
        formBuilder.add(Resources.getString(getClass(), "speed")).xy(1, 1);
        formBuilder.add(speed).xy(3, 1);

        ValueModel reportedSpeedModel =
            new PropertyAdapter<LocoModel>(locoModel, LocoModel.PROPERTYNAME_REPORTED_SPEED, true);
        final ValueModel reportedSpeedConverterModel =
            new ConverterValueModel(reportedSpeedModel, new StringConverter(new DecimalFormat("#")));

        reportedSpeed = WizardComponentFactory.createLabel(reportedSpeedConverterModel);
        formBuilder.add(Resources.getString(getClass(), "reportedSpeed")).xy(1, 3);
        formBuilder.add(reportedSpeed).xy(3, 3);

        speedSlider = new JZeroSlider();
        speedSlider.setOpaque(FormsSetup.getOpaqueDefault());

        int steps = locoModel.getSpeedSteps().getSteps() - 1;
        speedSlider.setMaximum(steps);
        if (locoModel.isCarControlEnabled()) {
            speedSlider.setMinimum(0);
        }
        else {
            speedSlider.setMinimum(-steps);
        }
        speedSlider.setPaintTicks(true);

        InputMap keyMap = (InputMap) UIManager.get("Slider.focusInputMap", speedSlider.getLocale());
        if (LOGGER.isDebugEnabled()) {
            if (keyMap != null) {
                Object binding = keyMap.get(KeyStroke.getKeyStroke(KeyEvent.VK_HOME, 0));
                LOGGER.debug("HOME is binded: {}", binding);
            }
        }

        speedSlider.getInputMap().put(KeyStroke.getKeyStroke(KeyEvent.VK_HOME, 0), "speedToMaxValue");
        speedSlider.getInputMap().put(KeyStroke.getKeyStroke(KeyEvent.VK_END, 0), "speedToMinValue");
        speedSlider.getInputMap().put(KeyStroke.getKeyStroke(KeyEvent.VK_PAGE_UP, 0), "speedUpValue");
        speedSlider.getInputMap().put(KeyStroke.getKeyStroke(KeyEvent.VK_PAGE_DOWN, 0), "speedDownValue");
        speedSlider.getInputMap().put(KeyStroke.getKeyStroke(KeyEvent.VK_SPACE, 0), "speedStop");

        speedSlider.getActionMap().put("speedToMaxValue", new AbstractAction() {
            private static final long serialVersionUID = 1L;

            @Override
            public void actionPerformed(ActionEvent e) {
                LOGGER.info("speedToMaxValue was called.");

                DirectionStatus dir = locoModel.getDirection();
                int steps = locoModel.getSpeedSteps().getSteps() - 1;
                if (DirectionStatus.FORWARD == dir) {
                    speedSlider.setValue(steps);
                }
                else {
                    speedSlider.setValue(-steps);
                }
            }
        });
        speedSlider.getActionMap().put("speedToMinValue", new AbstractAction() {
            private static final long serialVersionUID = 1L;

            @Override
            public void actionPerformed(ActionEvent e) {
                LOGGER.info("speedToMinValue was called.");
                speedSlider.setValue(0);
            }
        });
        speedSlider.getActionMap().put("speedUpValue", new AbstractAction() {
            private static final long serialVersionUID = 1L;

            @Override
            public void actionPerformed(ActionEvent e) {
                // increase speed
                Integer speed = locoModel.getSpeed();
                DirectionStatus dir = locoModel.getDirection();
                LOGGER.info("speedUpValue was called, current speed: {}, dir: {}", speed, dir);

                if (speed == null) {
                    speed = 0;
                }

                if (dir == DirectionStatus.FORWARD) {
                    if (speed < steps) {
                        speed += PAGE_STEPS;
                    }
                    if (speed > steps) {
                        speed = steps;
                    }
                }
                else {
                    // BACKWARDS
                    if (speed > 0) {
                        speed -= PAGE_STEPS + 1;

                        if (speed < 0) {
                            speed = 0;
                        }

                        // provide negative speed
                        speed = -speed;
                    }
                    else if (speed == 0) {
                        // change the direction
                        LOGGER.info("The speed is 0 and we must change the direction and increase speed.");
                        speed += PAGE_STEPS;
                    }
                }

                LOGGER.info("speedUpValue was called, new speed: {}", speed);
                speedSlider.setValue(speed);
            }
        });
        speedSlider.getActionMap().put("speedDownValue", new AbstractAction() {
            private static final long serialVersionUID = 1L;

            @Override
            public void actionPerformed(ActionEvent e) {
                // decrease speed
                Integer speed = locoModel.getSpeed();
                DirectionStatus dir = locoModel.getDirection();
                LOGGER.info("speedDownValue was called, current speed: {}, dir: {}", speed, dir);

                if (speed == null) {
                    speed = 0;
                }

                if (dir == DirectionStatus.FORWARD) {
                    if (speed > 0) {
                        speed -= PAGE_STEPS + 1;
                        if (speed < 0) {
                            speed = 0;
                        }
                    }
                    else if (speed == 0) {
                        // change the direction
                        LOGGER.info("The speed is 0 and we must change the direction and 'increase' speed.");
                        speed += PAGE_STEPS;

                        // provide negative speed
                        speed = -speed;
                    }
                }
                else {
                    // BACKWARDS
                    if (speed < steps) {
                        speed += PAGE_STEPS;
                    }
                    if (speed > steps) {
                        speed = steps;
                    }
                    speed = -speed;
                }
                LOGGER.info("speedDownValue was called, new speed: {}", speed);
                speedSlider.setValue(speed);
            }
        });
        speedSlider.getActionMap().put("speedStop", new AbstractAction() {
            private static final long serialVersionUID = 1L;

            @Override
            public void actionPerformed(ActionEvent e) {
                LOGGER.info("speedStop was called.");
                fireStop();
            }
        });

        // init value before add the change listener because otherwise stops the loco ...
        speedSlider.setValue(0);

        speedSlider.addChangeListener(new ChangeListener() {
            @Override
            public void stateChanged(ChangeEvent e) {
                JSlider source = (JSlider) e.getSource();
                if (!source.getValueIsAdjusting()) {

                    int value = speedSlider.getValue();
                    LOGGER.info("Current speedSlider value: {}", value);

                    if (value > 0) {
                        locoModel.setDirection(DirectionStatus.FORWARD);
                        value++;
                    }
                    else if (value < 0) {
                        locoModel.setDirection(DirectionStatus.BACKWARD);
                        value--;
                    }
                    value = Math.abs(value);

                    LOGGER.info("Set the speed value: {}", value);
                    locoModel.setSpeed(value);
                }
            }
        });

        // let the slider jump to the position the user clicked
        speedSlider.addMouseListener(new MouseAdapter() {
            @Override
            public void mousePressed(MouseEvent e) {
                JSlider sourceSlider = (JSlider) e.getSource();
                BasicSliderUI ui = (BasicSliderUI) sourceSlider.getUI();
                int value = ui.valueForXPosition(e.getX());

                // consume the mouse event to no let the other listeners be triggered
                e.consume();

                LOGGER.info("Mouse pressed on slider. Set the new value: {}", value);
                speedSlider.setValue(value);
            }
        });

        // allow control speed with mouse wheel
        int speedDelta = 1;
        this.speedSlider.addMouseWheelListener(evt -> {
            if (evt.getWheelRotation() < 0)// mouse wheel was rotated up/away from the user
            {
                int iNewValue = this.speedSlider.getValue() - speedDelta;
                if (iNewValue >= this.speedSlider.getMinimum()) {
                    this.speedSlider.setValue(iNewValue);
                }
                else {
                    this.speedSlider.setValue(0);
                }
            }
            else {
                int iNewValue = this.speedSlider.getValue() + speedDelta;
                if (iNewValue <= this.speedSlider.getMaximum()) {
                    this.speedSlider.setValue(iNewValue);
                }
                else {
                    this.speedSlider.setValue(this.speedSlider.getMaximum());
                }
            }
        });

        formBuilder.add(speedSlider).xyw(1, 5, 5);

        final JLabel minimumValueLabel = new JLabel(Resources.getString(getClass(), "backwards"));
        formBuilder.add(minimumValueLabel).xy(1, 7);
        if (locoModel.isCarControlEnabled()) {
            minimumValueLabel.setText(Resources.getString(LocoView.class, "stop"));
        }

        formBuilder.add(Resources.getString(LocoView.class, "forwards")).xy(5, 7);

        JPanel helperPanel = new JPanel(new BorderLayout());
        helperPanel.setOpaque(false);

        ledBarGraph = new LedBarGraph(10, Orientation.vertical);
        ledBarGraph.setOpaque(FormsSetup.getOpaqueDefault());

        helperPanel.add(formBuilder.build(), BorderLayout.CENTER);
        helperPanel.add(ledBarGraph, BorderLayout.EAST);

        locoModel.addPropertyChangeListener(LocoModel.PROPERTYNAME_CARCONTROLENABLED, new PropertyChangeListener() {

            @Override
            public void propertyChange(PropertyChangeEvent evt) {

                LOGGER.info("The car control enabled flag has changed.");
                if (locoModel != null && locoModel.getSpeedSteps() != null) {
                    int steps = locoModel.getSpeedSteps().getSteps() - 1;
                    speedSlider.setMaximum(steps);
                    if (locoModel.isCarControlEnabled()) {

                        // remove the speed after change slider range
                        try {
                            speedSlider.setMinimum(0);
                        }
                        finally {
                            locoModel.setSpeed(null);
                        }

                        minimumValueLabel.setText(Resources.getString(LocoView.class, "stop"));

                        // imageOrVideoContainerPanel.add(imagePanel);
                    }
                    else {
                        // remove the speed after change slider range
                        try {
                            speedSlider.setMinimum(-steps);
                        }
                        finally {
                            locoModel.setSpeed(null);
                        }
                        // imageOrVideoContainerPanel.remove(imagePanel);
                    }
                }
            }
        });

        locoModel.addPropertyChangeListener(LocoModel.PROPERTYNAME_SPEED, evt -> {

            boolean stopped = isSpeedStopped(locoModel.getSpeed());
            speedStepsCombo.setEnabled(stopped);
        });

        boolean stopped = isSpeedStopped(locoModel.getSpeed());
        speedStepsCombo.setEnabled(stopped);

        return helperPanel;
    }

    private boolean isSpeedStopped(Integer speed) {
        boolean stopped = false;
        if (speed != null) {
            switch (speed.intValue()) {
                case 0:
                case 1:
                    // stopped
                    stopped = true;
                    break;
                default:
                    break;
            }
        }
        else {
            // assume stopped
            stopped = true;
        }
        return stopped;
    }

    private JComponent createSpeedGaugePanel() {
        final Radial speedGauge = SpeedGaugeBuilder.speedGauge("Speed", "km/h");

        JPanel panel = new JPanel() {
            private static final long serialVersionUID = 1L;

            @Override
            public Dimension getPreferredSize() {
                return new Dimension(200, 200);
            }
        };
        panel.setOpaque(false);

        panel.setLayout(new BorderLayout());
        panel.add(speedGauge, BorderLayout.CENTER);

        final ValueModel reportedSpeedModel =
            new PropertyAdapter<LocoModel>(locoModel, LocoModel.PROPERTYNAME_REPORTED_SPEED, true);

        Bindings.bind(speedGauge, "valueAnimated", reportedSpeedModel);

        return panel;
    }

    private JPanel createLightAndStopButtonPanel(final List<JideButton> functionButtons) {

        final FormBuilder formBuilder = FormBuilder.create().columns("pref, 3dlu, 60dlu:grow, 3dlu, pref").rows("pref");

        // add the light icons
        ImageIcon lightOnIcon = ImageUtils.createImageIcon(LocoView.class, "/icons/16x16/lightbulb.png");
        ImageIcon lightOffIcon = ImageUtils.createImageIcon(LocoView.class, "/icons/16x16/lightbulb_off.png");

        final JideToggleButton lightButton = new JideToggleButton(lightOffIcon);
        lightButton.setSelectedIcon(lightOnIcon);
        lightButton.setButtonStyle(JideButton.TOOLBOX_STYLE);

        lightButton.addActionListener(new ActionListener() {
            @Override
            public void actionPerformed(ActionEvent e) {
                locoModel.setFunction(0, lightButton.isSelected());
            }
        });
        functionButtons.add(lightButton);
        functionButtonMap.put("F0", lightButton);

        formBuilder.add(lightButton).xy(1, 1);

        JPanel stopPanel = new JPanel(new GridBagLayout());

        stopPanel
            .setBorder(BorderFactory
                .createTitledBorder(BorderFactory.createEtchedBorder(),
                    Resources.getString(getClass(), "stopLoco") + ":"));

        stopButton = new JButton(Resources.getString(getClass(), "stop"));

        stopButton.addActionListener(new ActionListener() {
            @Override
            public void actionPerformed(ActionEvent e) {
                fireStop();
            }
        });

        stopEmergencyButton = new JButton(Resources.getString(getClass(), "emergencyStop"));

        stopEmergencyButton.addActionListener(new ActionListener() {
            @Override
            public void actionPerformed(ActionEvent e) {
                fireEmergencyStop();
            }
        });

        JPanel buttonPanel = new ButtonBarBuilder().addGlue().addButton(stopButton, stopEmergencyButton).build();
        formBuilder.add(buttonPanel).xy(3, 1);

        return formBuilder.build();
    }

    private JPanel createFunctionButtonPanel(final List<JideButton> functionButtons) {

        final FormBuilder formBuilder =
            FormBuilder
                .create()
                .columns(
                    "pref, 8dlu, pref, 8dlu, pref, 8dlu, pref, 8dlu, pref, 8dlu, pref, 8dlu, pref, 8dlu, pref, 8dlu, pref, 8dlu, pref")
                .rows("pref, 3dlu, pref, 3dlu, pref, 3dlu, pref, 3dlu, pref");

        formBuilder.addSeparator(Resources.getString(getClass(), "additionalFunctions")).xyw(1, 1, 19);

        final int columns = 10;

        for (int row = 0; row < 4; row++) {

            for (int column = 0; column < columns; column++) {
                final int functionIndex = row * columns + (column + 1);
                String buttonText = "F" + functionIndex;
                if (functionIndex > 28) {
                    buttonText = "B" + functionIndex;
                }

                // use JideToggleButton for function and binary function buttons
                final JideButton functionButton = new JideToggleButton(buttonText);
                functionButton.setButtonStyle(JideButton.TOOLBOX_STYLE);

                functionButton.addActionListener(event -> {
                    LOGGER.info("Set function with index: {}", functionIndex);
                    if (functionIndex < 29) {
                        locoModel.setFunction(functionIndex, functionButton.isSelected());
                    }
                    else {
                        locoModel.setBinaryState(functionIndex, true);
                    }
                });
                functionButtons.add(functionButton);

                functionButtonMap.put(buttonText, functionButton);

                formBuilder.add(functionButton).xy((column * 2) + 1, (row * 2) + 3);
            }
        }

        return formBuilder.build();
    }

    private JComponent[] rfBaseButtons;

    private JPanel addMultiRfBasisButtons() {

        final FormBuilder formBuilder =
            FormBuilder
                .create()
                .columns(
                    "pref, 5dlu, pref, 5dlu, pref, 5dlu, pref, 5dlu, pref, 5dlu, pref, 5dlu, pref, 5dlu, pref:grow")
                .rows("pref, 3dlu, pref");

        formBuilder.addSeparator(Resources.getString(getClass(), "multiRFBasis")).xyw(1, 1, 15);

        ValueModel activeBaseModel =
            new PropertyAdapter<LocoModel>(locoModel, LocoModel.PROPERTYNAME_ACTIVE_BASE, true);

        // add the radio buttons for the basis change
        rfBaseButtons = new JComponent[RfBasisMode.values().length];
        int column = 0;
        int col = 0;
        int row = 3;
        for (RfBasisMode rfBaseMode : RfBasisMode.values()) {

            JRadioButton radio =
                WizardComponentFactory
                    .createRadioButton(activeBaseModel, rfBaseMode,
                        Resources.getString(RfBasisMode.class, rfBaseMode.getKey()));
            rfBaseButtons[column] = radio;

            // add radio button
            formBuilder.add(radio).xy(col * 2 + 1, row);
            column++;
            col++;

            if (rfBaseMode == RfBasisMode.SINGLE) {
                formBuilder.appendRows("3dlu, pref");
                col = 0;
                row += 2;
            }
        }

        formBuilder.appendRows("3dlu, pref");
        col = 0;
        row += 2;

        ValueModel reportedCellNumberModel =
            new PropertyAdapter<LocoModel>(locoModel, LocoModel.PROPERTYNAME_REPORTED_CELLNUMBER, true);
        final ValueModel reportedCellNumberConverterModel =
            new ConverterValueModel(reportedCellNumberModel, new StringConverter(new DecimalFormat("#")));

        JTextField reportedCellNumberText =
            WizardComponentFactory.createTextField(reportedCellNumberConverterModel, false);
        reportedCellNumberText.setEditable(false);
        formBuilder.add("Reported RF Cellnumber").xyw(1, row, 5);
        formBuilder.add(reportedCellNumberText).xy(7, row);

        activeBaseModel.addValueChangeListener(evt -> {
            LOGGER.info("The active RF base has changed: {}", evt.getNewValue());

            RfBasisMode activeRfBase = locoModel.getActiveBase();
            if (activeRfBase != null) {
                final Integer functionIndex = activeRfBase.getFunctionIndex();

                if (functionIndex != null) {
                    // send the binary state
                    locoModel.setBinaryState(functionIndex, true);
                }
            }
        });

        return formBuilder.build();
    }

    private JPanel addCounterPanel() {
        final FormBuilder formBuilder =
            FormBuilder
                .create().columns("pref, 5dlu, pref:grow").rows("pref, 3dlu, pref, 3dlu, pref, 3dlu, pref, 3dlu, pref");

        formBuilder.addSeparator(Resources.getString(getClass(), "Counter")).xyw(1, 1, 3);

        // CS_DRIVE
        ValueModel counterCsDriveModel =
            new PropertyAdapter<LocoModel>(locoModel, LocoModel.PROPERTYNAME_COUNTER_CS_DRIVE, true);
        final ValueModel counterCsDriveConverterModel =
            new ConverterValueModel(counterCsDriveModel, new StringConverter(new DecimalFormat("#")));

        JTextField counterCsDriveText = WizardComponentFactory.createTextField(counterCsDriveConverterModel, false);
        counterCsDriveText.setEditable(false);
        formBuilder.add("CS_DRIVE").xy(1, 3);
        formBuilder.add(counterCsDriveText).xy(3, 3);

        // CS_BIN_STATE
        ValueModel counterCsBinStateModel =
            new PropertyAdapter<LocoModel>(locoModel, LocoModel.PROPERTYNAME_COUNTER_CS_BIN_STATE, true);
        final ValueModel counterCsBinStateConverterModel =
            new ConverterValueModel(counterCsBinStateModel, new StringConverter(new DecimalFormat("#")));

        JTextField counterCsBinStateText =
            WizardComponentFactory.createTextField(counterCsBinStateConverterModel, false);
        counterCsBinStateText.setEditable(false);
        formBuilder.add("CS_BIN_STATE").xy(1, 5);
        formBuilder.add(counterCsBinStateText).xy(3, 5);

        // CS_DRIVE_ACK
        ValueModel counterCsDriveAckModel =
            new PropertyAdapter<LocoModel>(locoModel, LocoModel.PROPERTYNAME_COUNTER_CS_DRIVE_ACK, true);
        final ValueModel counterCsDriveAckConverterModel =
            new ConverterValueModel(counterCsDriveAckModel, new StringConverter(new DecimalFormat("#")));

        JTextField counterCsDriveAckText =
            WizardComponentFactory.createTextField(counterCsDriveAckConverterModel, false);
        counterCsDriveAckText.setEditable(false);
        formBuilder.add("CS_DRIVE_ACK").xy(1, 7);
        formBuilder.add(counterCsDriveAckText).xy(3, 7);

        // reset button
        JButton resetButton = new JButton(Resources.getString(getClass(), "reset"));
        resetButton.addActionListener(evt -> {
            LOGGER.info("Reset the counters.");
            locoModel.resetCounterCsDrive();
            locoModel.resetCounterCsAckDrive();
            locoModel.resetCounterCsBinState();
        });

        formBuilder.add(resetButton).xy(1, 9);

        return formBuilder.build();
    }

    private JComponent[] devBinStateValueButtons;

    private JPanel addDirectBinStatePanel() {
        final FormBuilder formBuilder =
            FormBuilder
                .create().columns("pref, 5dlu, max(pref;30dlu), 5dlu, pref, 5dlu, pref, 5dlu, pref, 5dlu:grow")
                .rows("pref, 3dlu, pref");

        formBuilder.addSeparator(Resources.getString(getClass(), "devBinState")).xyw(1, 1, 10);

        ValueModel devBinStateNumberModel =
            new PropertyAdapter<LocoModel>(locoModel, LocoModel.PROPERTYNAME_DEV_BINSTATE_NUMBER, true);
        final ValueModel binStateNumberConverterModel =
            new ConverterValueModel(devBinStateNumberModel, new StringConverter(new DecimalFormat("#")));

        JTextField devBinStateNumberText = WizardComponentFactory.createTextField(binStateNumberConverterModel, false);
        InputValidationDocument stateNumberDocument = new InputValidationDocument(5, InputValidationDocument.NUMERIC);
        stateNumberDocument.setDocumentFilter(new IntegerRangeFilter(0, 32767));
        devBinStateNumberText.setDocument(stateNumberDocument);
        formBuilder.add("Number").xy(1, 3);
        formBuilder.add(devBinStateNumberText).xy(3, 3);

        ValueModel devBinStateValueModel =
            new PropertyAdapter<LocoModel>(locoModel, LocoModel.PROPERTYNAME_DEV_BINSTATE_VALUE, true);

        devBinStateValueButtons = new JComponent[BinStateValue.values().length];
        int column = 0;
        for (BinStateValue binStateValue : BinStateValue.values()) {

            JRadioButton radio =
                WizardComponentFactory
                    .createRadioButton(devBinStateValueModel, binStateValue,
                        Resources.getString(BinStateValue.class, binStateValue.getKey()));
            devBinStateValueButtons[column] = radio;

            // add radio button
            formBuilder.add(radio).xy(5 + column * 2, 3);
            column++;
        }

        JButton sendButton = new JButton(Resources.getString(getClass(), "send"));
        sendButton.addActionListener(evt -> {
            BinStateValue devBinStateValue = locoModel.getDevBinStateValue();
            if (devBinStateValue != null) {
                final boolean functionValue = devBinStateValue.getFunctionValue();
                Integer binStateNumber = locoModel.getDevBinStateNumber();

                if (binStateNumber != null) {
                    // send the binary state
                    locoModel.setBinaryState(binStateNumber.intValue(), functionValue);
                }
                else {
                    LOGGER.info("No bin state number available. The BIN_STATE is not fired.");
                }
            }
        });
        sendButton.setEnabled(false);

        formBuilder.add(sendButton).xy(9, 3);

        devBinStateNumberModel.addValueChangeListener(evt -> {
            Integer binStateNumber = locoModel.getDevBinStateNumber();
            sendButton.setEnabled(binStateNumber != null);
        });

        return formBuilder.build();
    }

    private BlinkerLed blinkerRight;

    private BlinkerLed blinkerLeft;

    private JPanel createImagePanel(final List<JideButton> functionButtons) {
        JPanel imagePanel = new JPanel(new BorderLayout());

        JLayeredPane layeredPane = new JLayeredPane();
        // layeredPane.setLayout(null);
        layeredPane.setPreferredSize(new Dimension(140, 140));

        // use the image
        ImageIcon carFrontIcon = ImageUtils.loadImageIcon(LocoView.class, "/images/Truck-Front.png", 120, 120);
        JLabel backgroundLabel = new JLabel(carFrontIcon);
        backgroundLabel.setOpaque(true);
        // backgroundLabel.setBounds(0, 0, carFrontIcon.getIconWidth(), carFrontIcon.getIconHeight());
        backgroundLabel.setBounds(0, 0, 140, 140);

        layeredPane.add(backgroundLabel, Integer.valueOf(10));

        blinkerRight = new BlinkerLed(0, 0, 8, 6, Color.ORANGE.brighter(), Color.ORANGE);
        blinkerLeft = new BlinkerLed(0, 0, 8, 6, Color.ORANGE.brighter(), Color.ORANGE);

        final BlinkerLed lightRight = new BlinkerLed(0, 0, 8, 11, Color.YELLOW, Color.WHITE);
        final BlinkerLed lightLeft = new BlinkerLed(0, 0, 8, 11, Color.YELLOW, Color.WHITE);

        layeredPane.add(blinkerLeft, Integer.valueOf(20));
        layeredPane.add(blinkerRight, Integer.valueOf(20));
        layeredPane.add(lightLeft, Integer.valueOf(20));
        layeredPane.add(lightRight, Integer.valueOf(20));

        blinkerLeft.setBounds(37, 84, 10, 7);
        blinkerRight.setBounds(93, 84, 10, 7);
        lightLeft.setBounds(37, 92, 10, 11);
        lightRight.setBounds(93, 92, 10, 11);

        final Cursor cursor = Cursor.getPredefinedCursor(Cursor.HAND_CURSOR);
        blinkerLeft.setCursor(cursor);
        blinkerRight.setCursor(cursor);
        lightLeft.setCursor(cursor);
        lightRight.setCursor(cursor);

        final JideToggleButton functionButtonLeft = getButton(functionButtons.get(2), JideToggleButton.class);
        functionButtonLeft.addChangeListener(new ChangeListener() {

            @Override
            public void stateChanged(ChangeEvent e) {
                blinkerLeft.setActive(functionButtonLeft.isSelected(), true);
            }
        });

        final JideToggleButton functionButtonRight = getButton(functionButtons.get(1), JideToggleButton.class);
        functionButtonRight.addChangeListener(new ChangeListener() {

            @Override
            public void stateChanged(ChangeEvent e) {
                blinkerRight.setActive(functionButtonRight.isSelected(), true);
            }
        });

        final JideToggleButton functionButtonLights = getButton(functionButtons.get(0), JideToggleButton.class);
        functionButtonLights.addChangeListener(new ChangeListener() {

            @Override
            public void stateChanged(ChangeEvent e) {
                LOGGER.info("Lights state has changed, selected: {}", functionButtonLights.isSelected());
                lightRight.setActive(functionButtonLights.isSelected(), false);
                lightLeft.setActive(functionButtonLights.isSelected(), false);
            }
        });

        backgroundLabel.addMouseListener(new MouseAdapter() {

            @Override
            public void mouseClicked(MouseEvent e) {

                Point point = e.getPoint();
                LOGGER.info("Mouse clicked, point: {}", point);

                if (point.getX() > 37 && point.getX() < 44) {
                    // left side

                    if (point.getY() > 83 && point.getY() < 89) {
                        // blinker
                        LOGGER.info("Turn left blinker on/off.");

                        int functionIndex = 2;
                        JideToggleButton functionButton =
                            getButton(functionButtons.get(functionIndex), JideToggleButton.class);
                        // toggle the button
                        functionButton.setSelected(!functionButton.isSelected());

                        locoModel.setFunction(functionIndex, functionButton.isSelected());
                    }
                    else if (point.getY() > 91 && point.getY() < 102) {
                        // front lights
                        LOGGER.info("Turn front lights on/off.");

                        int functionIndex = 0;
                        JideToggleButton functionButton =
                            getButton(functionButtons.get(functionIndex), JideToggleButton.class);
                        // toggle the button
                        functionButton.setSelected(!functionButton.isSelected());

                        locoModel.setFunction(functionIndex, functionButton.isSelected());
                    }
                }
                else if (point.getX() > 93 && point.getX() < 101) {
                    // right side

                    if (point.getY() > 83 && point.getY() < 89) {
                        // blinker
                        LOGGER.info("Turn right blinker on/off.");

                        int functionIndex = 1;
                        JideToggleButton functionButton =
                            getButton(functionButtons.get(functionIndex), JideToggleButton.class);
                        // toggle the button
                        functionButton.setSelected(!functionButton.isSelected());

                        locoModel.setFunction(functionIndex, functionButton.isSelected());
                    }
                    else if (point.getY() > 91 && point.getY() < 102) {
                        // front lights
                        LOGGER.info("Turn front lights on/off.");

                        int functionIndex = 0;
                        JideToggleButton functionButton =
                            getButton(functionButtons.get(functionIndex), JideToggleButton.class);
                        // toggle the button
                        functionButton.setSelected(!functionButton.isSelected());

                        locoModel.setFunction(functionIndex, functionButton.isSelected());
                    }
                }
            }
        });

        imagePanel.add(layeredPane, BorderLayout.CENTER);
        return imagePanel;
    }

    private static <T> T getButton(JButton button, Class<T> type) {
        // if same type
        if (type.isInstance(button)) {
            return type.cast(button);
        }

        throw new IllegalArgumentException("The provided button has not the requested type.");
    }

    protected void fireWriteState(int state, boolean value) {
        locoModel.setBinaryState(state, value);
    }

    private void addLocoViewListener(LocoViewListener listener) {
        locoViewListeners.add(listener);
    }

    public void addViewCloseListener(ViewCloseListener listener) {
        viewCloseListeners.add(listener);
    }

    private void fireEmergencyStop() {
        locoModel.setSpeed(1);

        for (LocoViewListener listener : locoViewListeners) {
            listener.emergencyStop();
        }
    }

    private void fireStop() {
        locoModel.setSpeed(0);

        for (LocoViewListener listener : locoViewListeners) {
            listener.stop();
        }
    }

    @Override
    public void selectDecoderAddress(int dccAddress) {
        LOGGER.info("Select the decoder address: {}", dccAddress);
        address.setText(String.valueOf(dccAddress));

        // set the speed to null because we want the new speed value triggered
        locoModel.setSpeed(null);
    }

    @Override
    public void setSpeedSteps(SpeedSteps speedSteps) {
        LOGGER.info("Set the speed steps: {}", speedSteps);
        speedStepsCombo.setSelectedItem(speedSteps);
    }

    @Override
    public void setSpeed(int speed) {
        LOGGER.info("Set the speed: {}", speed);

        changeSliderSilently(speed);
        locoModel.setSpeed(speed);
        // speedSlider.setValue(speed);
    }

    @Override
    public void setFunction(int function) {
        LOGGER.info("Set the function: {}", function);

        JideToggleButton functionButton = getButton(functionButtonMap.get("F" + function), JideToggleButton.class);
        LOGGER.warn("Fetched functionButton: {}", functionButton);

        functionButton.doClick();
    }

    @Override
    public void setBinState(int binStateNumber, boolean flag) {
        LOGGER.info("Set the binState, binStateNumber: {}, flag: {}", binStateNumber, flag);

        // TODO
        // send the binary state
        locoModel.setBinaryState(binStateNumber, flag);
    }

    @Override
    public void setStop() {
        LOGGER.info("Set stop.");

        stopButton.doClick();
    }

    @Override
    public void setStopEmergency() {
        LOGGER.info("Set stop emergency.");

        stopEmergencyButton.doClick();
    }

    public void cleanup() {
        LOGGER.info("The LocoView is disposed.");

        for (ViewCloseListener closeListener : viewCloseListeners) {
            try {
                closeListener.close();
            }
            catch (Exception ex) {
                LOGGER.warn("Notify view close listener failed.", ex);
            }
        }

        if (blinkerLeft != null) {
            blinkerLeft.setActive(false, true);
        }
        if (blinkerRight != null) {
            blinkerRight.setActive(false, true);
        }

        viewCloseListeners.clear();

        locoViewListeners.clear();

        if (locoModelListener != null) {
            locoModel.removeLocoModelListener(locoModelListener);
            locoModelListener = null;
        }

        if (scriptPanel != null) {
            scriptPanel.close();
            scriptPanel = null;
        }

        if (locoModel != null) {
            locoModel = null;
        }

        functionButtonMap.clear();
    }

    private static class BlinkerLed extends JComponent {

        private static final long serialVersionUID = 1L;

        private Timer blinkerTimer;

        private Color colorOn;

        private Color colorOff;

        private boolean on;

        private int x;

        private int y;

        private int width;

        private int height;

        public BlinkerLed(int x, int y, int width, int height, Color colorOn, Color colorOff) {
            this.x = x;
            this.y = y;
            this.width = width;
            this.height = height;
            this.colorOn = colorOn;
            this.colorOff = colorOff;
        }

        public void setActive(boolean active, boolean useTimer) {

            repaint();

            if (active) {
                if (useTimer && blinkerTimer == null) {
                    blinkerTimer = new Timer(500, evt -> {
                        on = !on;
                        repaint();
                    });
                }
                on = true;
                if (blinkerTimer != null) {
                    blinkerTimer.start();
                }
            }
            else if (!active) {
                on = false;
                if (blinkerTimer != null) {
                    blinkerTimer.stop();
                }
            }
        }

        @Override
        public void paintComponent(Graphics graphics) {

            if (on) {
                graphics.setColor(colorOn);
            }
            else {
                graphics.setColor(colorOff);
            }
            graphics.fillRect(x, y, width, height);
        }
    }
}
