package org.bidib.wizard.mvc.locolist.model;

import java.beans.PropertyChangeEvent;
import java.beans.PropertyChangeListener;
import java.util.Collections;
import java.util.Date;
import java.util.LinkedList;
import java.util.List;
import java.util.UUID;

import javax.swing.JOptionPane;
import javax.swing.SwingUtilities;

import org.apache.commons.collections4.IterableUtils;
import org.apache.commons.collections4.Predicate;
import org.apache.commons.lang3.StringUtils;
import org.bidib.jbidibc.core.schema.UserDevicesListFactory;
import org.bidib.jbidibc.core.schema.decoder.commontypes.DecoderTypeType;
import org.bidib.jbidibc.core.schema.decoder.userdevices.DecoderType;
import org.bidib.jbidibc.core.schema.decoder.userdevices.UserDeviceType;
import org.bidib.jbidibc.core.schema.decoder.userdevices.UserDevicesList;
import org.bidib.jbidibc.messages.DriveState;
import org.bidib.jbidibc.messages.enums.DirectionEnum;
import org.bidib.wizard.model.status.DirectionStatus;
import org.bidib.wizard.model.status.SpeedSteps;
import org.bidib.wizard.mvc.locolist.controller.listener.LocoTableControllerListener;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.jgoodies.binding.beans.Model;
import com.jgoodies.common.collect.ArrayListModel;

public class LocoTableModel extends Model {

    private static final long serialVersionUID = 1L;

    private static final Logger LOGGER = LoggerFactory.getLogger(LocoTableModel.class);

    public static final String PROPERTY_LOCOS = "locos";

    public static final String PROPERTY_CS_NODE_SELECTED = "csNodeSelected";

    private ArrayListModel<LocoModel> locoList = new ArrayListModel<>();

    private boolean csNodeSelected;

    private final PropertyChangeListener locoChangeListener;

    private final LocoTableControllerListener locoTableController;

    private UserDevicesList userDevicesList;

    public LocoTableModel(final LocoTableControllerListener locoTableController) {

        this.locoTableController = locoTableController;

        // create the propertyChangeListener to refresh the loco list
        locoChangeListener = new PropertyChangeListener() {

            @Override
            public void propertyChange(PropertyChangeEvent evt) {
                if (LocoModel.PROPERTY_LOCO_NAME.equals(evt.getPropertyName())) {
                    LocoModel loco = (LocoModel) evt.getSource();
                    LOGGER.info("The loco name has been changed: {}", loco);

                    if (locoTableController != null) {
                        try {
                            // create the user devices list from the locolist
                            UserDevicesList userDevicesList = toUserDevicesList(locoList);

                            locoTableController.saveUserDevicesList(userDevicesList);
                        }
                        catch (Exception ex) {
                            LOGGER.warn("Save user devices list failed.", ex);

                            JOptionPane
                                .showMessageDialog(null, "Save user devices list failed.", "User Devices",
                                    JOptionPane.ERROR_MESSAGE);
                        }
                    }
                }
            }
        };
    }

    private UserDevicesList toUserDevicesList(ArrayListModel<LocoModel> locoList) {

        if (userDevicesList == null) {
            LOGGER.info("Create the initial userDevicesList to store the data.");
            userDevicesList = UserDevicesListFactory.prepareEmptyList(new Date());
        }

        List<UserDeviceType> userDeviceList = userDevicesList.getUserDevices().getUserDevice();

        for (LocoModel loco : locoList) {

            // check if the loco is already in the user devices list
            UserDeviceType userDevice =
                userDeviceList
                    .stream()
                    .filter(
                        dev -> dev.getAddress() == loco.getLocoAddress() && dev.getDeviceType() == DecoderTypeType.LOCO)
                    .findFirst().orElse(null);
            if (userDevice == null) {

                UUID uuid = UUID.randomUUID();
                String randomUUIDString = uuid.toString();

                String locoName = loco.getLocoName();
                if (StringUtils.isBlank(locoName)) {
                    locoName = "Lok " + loco.getLocoAddress();
                }
                userDevice =
                    new UserDeviceType()
                        .withAddress(loco.getLocoAddress()).withId(randomUUIDString).withName(locoName)
                        .withDeviceType(DecoderTypeType.LOCO);

                int speedSteps = SpeedSteps.valueOf(loco.getSpeedSteps());
                userDevice.withDecoder(new DecoderType().withName("Unknown").withSpeedsteps(speedSteps));

                LOGGER.info("Created new userDevice entry: {}", userDevice);

                userDevicesList.getUserDevices().withUserDevice(userDevice);
            }
            else {
                String locoName = loco.getLocoName();
                if (StringUtils.isBlank(locoName)) {
                    locoName = "Lok " + loco.getLocoAddress();
                }
                userDevice.setName(locoName);
            }
        }

        return userDevicesList;
    }

    public void setUserDevicesList(final UserDevicesList userDevicesList) {
        // keep the list
        this.userDevicesList = userDevicesList;

        if (userDevicesList != null && userDevicesList.getUserDevices() != null) {
            List<UserDeviceType> locoDevices =
                UserDevicesListFactory
                    .findUserDevices(DecoderTypeType.LOCO, userDevicesList.getUserDevices().getUserDevice());

            for (UserDeviceType locoDevice : locoDevices) {
                addLoco(locoDevice);
            }
        }
    }

    public void addLoco(final UserDeviceType locoDevice) {
        synchronized (locoList) {
            int locoAddress = locoDevice.getAddress();
            final LocoModel loco = new LocoModel(locoAddress);
            if (!locoList.contains(loco)) {
                LOGGER.info("Add loco to loco list: {}, locoDevice: {}", loco, locoDevice);
                toLocoModel(locoDevice, loco);
                // add the property change listener
                loco.addPropertyChangeListener(locoChangeListener);

                List<LocoModel> oldValue = new LinkedList<>(locoList);
                locoList.add(loco);

                firePropertyChange(PROPERTY_LOCOS, oldValue, locoList);
            }
            else {
                LOGGER.warn("Loco is already in loco list: {}", loco);
                LocoModel existing = IterableUtils.find(locoList, new Predicate<LocoModel>() {

                    @Override
                    public boolean evaluate(LocoModel model) {
                        return model.equals(loco);
                    }

                });
                toLocoModel(locoDevice, existing);
            }
        }
    }

    private void toLocoModel(final UserDeviceType locoDevice, final LocoModel loco) {
        // update members of loco
        if (locoDevice.getDecoder() != null && locoDevice.getDecoder().getSpeedsteps() > 0) {
            loco.setSpeedSteps(SpeedSteps.valueOf(locoDevice.getDecoder().getSpeedsteps()));
        }
        loco.setLocoName(locoDevice.getName());
    }

    public void addLoco(final DriveState driveState) {
        synchronized (locoList) {
            int locoAddress = driveState.getAddress();
            final LocoModel loco = new LocoModel(locoAddress);
            if (!locoList.contains(loco)) {
                LOGGER.info("Add loco to loco list: {}, driveState: {}", loco, driveState);
                toLocoModel(driveState, loco);
                // add the property change listener
                loco.addPropertyChangeListener(locoChangeListener);

                List<LocoModel> oldValue = new LinkedList<>(locoList);
                locoList.add(loco);

                firePropertyChange(PROPERTY_LOCOS, oldValue, locoList);
            }
            else {
                LOGGER.warn("Loco is already in loco list: {}", loco);
                LocoModel existing = IterableUtils.find(locoList, new Predicate<LocoModel>() {

                    @Override
                    public boolean evaluate(LocoModel model) {
                        return model.equals(loco);
                    }

                });
                toLocoModel(driveState, existing);

                fireLocoChanged(existing);
            }
        }
    }

    private void fireLocoChanged(final LocoModel locoModel) {
        LOGGER.debug("The loco address has been changed.");
        int index = locoList.indexOf(locoModel);
        locoList.fireContentsChanged(index);
    }

    private void toLocoModel(final DriveState driveState, final LocoModel loco) {
        // update members of loco
        loco.setSpeed(driveState.getSpeed() & 0x7F);

        loco
            .setDirection((driveState.getDirection() == DirectionEnum.FORWARD ? DirectionStatus.FORWARD
                : DirectionStatus.BACKWARD));

        loco.setSpeedSteps(SpeedSteps.fromBidibFormat(driveState.getAddressFormat()));

        loco.setFunctions(driveState.getFunctions());

        loco.setOutputActive(driveState.getOutputActive() > 0);
    }

    public void removeLoco(int locoAddress) {
        synchronized (locoList) {
            LOGGER.info("Remove loco from loco list: {}", locoAddress);

            List<LocoModel> oldValue = new LinkedList<>(locoList);
            int index = locoList.indexOf(new LocoModel(locoAddress/* , null */));
            if (index > -1) {
                LocoModel loco = locoList.remove(index);
                if (loco != null) {
                    loco.removePropertyChangeListener(locoChangeListener);
                }

                firePropertyChange(PROPERTY_LOCOS, oldValue, locoList);
            }
        }
    }

    public void removeAllLocos() {
        synchronized (locoList) {
            LOGGER.info("Remove all locos from loco list.");

            if (SwingUtilities.isEventDispatchThread()) {

                internalRemoveAllLocos();
            }
            else {
                SwingUtilities.invokeLater(() -> internalRemoveAllLocos());
            }
        }
    }

    private void internalRemoveAllLocos() {
        List<LocoModel> oldValue = new LinkedList<>(locoList);
        locoList.clear();

        for (LocoModel loco : oldValue) {
            loco.removePropertyChangeListener(locoChangeListener);
        }

        firePropertyChange(PROPERTY_LOCOS, oldValue, locoList);
    }

    public void setDriveState(byte[] address, DriveState driveState) {
        LOGGER.info("Drive state was delivered: {}", driveState);

        // if (driveState.getOutputActive() > 0) {
        if (driveState.getAddress() > 0) {
            addLoco(driveState);
        }
        // TODO remove locos from list if no longer in stack?
        // else {
        // removeLoco(driveState.getAddress());
        // }

    }

    public ArrayListModel<LocoModel> getLocoListModel() {
        return locoList;
    }

    public List<LocoModel> getLocos() {
        return Collections.unmodifiableList(locoList);
    }

    public boolean isCsNodeSelected() {
        return csNodeSelected;
    }

    public void setCsNodeSelected(boolean csNodeSelected) {
        boolean oldValue = this.csNodeSelected;
        this.csNodeSelected = csNodeSelected;

        if (SwingUtilities.isEventDispatchThread()) {
            firePropertyChange(PROPERTY_CS_NODE_SELECTED, oldValue, csNodeSelected);
        }
        else {
            SwingUtilities.invokeLater(() -> firePropertyChange(PROPERTY_CS_NODE_SELECTED, oldValue, csNodeSelected));
        }
    }

    /**
     * Get the current user devices list based on the current loco list.
     * 
     * @return the current user devices list
     */
    public UserDevicesList getUserDevicesList() {

        UserDevicesList userDevicesList = toUserDevicesList(locoList);
        return userDevicesList;
    }

}
