package org.bidib.wizard.mvc.main.view.panel;

import java.awt.BorderLayout;
import java.awt.Component;
import java.awt.Point;
import java.awt.event.ActionEvent;
import java.awt.event.MouseAdapter;
import java.awt.event.MouseEvent;
import java.beans.PropertyChangeEvent;
import java.beans.PropertyChangeListener;
import java.text.DecimalFormat;
import java.util.List;
import java.util.Objects;
import java.util.SortedSet;
import java.util.TreeSet;

import javax.swing.AbstractAction;
import javax.swing.BorderFactory;
import javax.swing.BoxLayout;
import javax.swing.ImageIcon;
import javax.swing.JComboBox;
import javax.swing.JComponent;
import javax.swing.JLabel;
import javax.swing.JOptionPane;
import javax.swing.JPanel;
import javax.swing.JRadioButton;
import javax.swing.JScrollPane;
import javax.swing.JTable;
import javax.swing.JTextField;
import javax.swing.ListModel;
import javax.swing.ListSelectionModel;
import javax.swing.event.ListDataEvent;
import javax.swing.event.ListDataListener;
import javax.swing.event.TableModelEvent;
import javax.swing.event.TableModelListener;
import javax.swing.table.DefaultTableCellRenderer;
import javax.swing.table.TableCellEditor;
import javax.swing.table.TableCellRenderer;
import javax.swing.table.TableColumn;

import org.apache.commons.collections4.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.bidib.jbidibc.core.schema.bidibbase.BaseLabel;
import org.bidib.jbidibc.core.schema.bidiblabels.AccessoryLabel;
import org.bidib.jbidibc.core.schema.bidiblabels.AccessoryLabels;
import org.bidib.jbidibc.core.schema.bidiblabels.NodeLabels;
import org.bidib.jbidibc.exchange.vendorcv.VendorCV;
import org.bidib.jbidibc.messages.AccessoryState;
import org.bidib.jbidibc.messages.enums.AccessoryExecutionState;
import org.bidib.jbidibc.messages.enums.TimeBaseUnitEnum;
import org.bidib.jbidibc.messages.utils.ProductUtils;
import org.bidib.wizard.api.locale.Resources;
import org.bidib.wizard.api.model.Accessory;
import org.bidib.wizard.api.model.AccessorySaveState;
import org.bidib.wizard.api.model.AccessorySwitchTimeModel;
import org.bidib.wizard.api.model.Macro;
import org.bidib.wizard.api.model.MacroRef;
import org.bidib.wizard.api.model.NodeInterface;
import org.bidib.wizard.api.model.listener.AccessoryExecutionListener;
import org.bidib.wizard.api.model.listener.AccessoryListener;
import org.bidib.wizard.client.common.converter.StringConverter;
import org.bidib.wizard.client.common.text.WizardComponentFactory;
import org.bidib.wizard.client.common.view.slider.SliderRenderer;
import org.bidib.wizard.client.common.view.validation.IconFeedbackPanel;
import org.bidib.wizard.client.common.view.validation.PropertyValidationI18NSupport;
import org.bidib.wizard.common.context.DefaultApplicationContext;
import org.bidib.wizard.common.script.node.types.AccessoryAspectTargetType;
import org.bidib.wizard.common.utils.ImageUtils;
import org.bidib.wizard.core.labels.AccessoryLabelUtils;
import org.bidib.wizard.model.status.AccessoryAspectStatus;
import org.bidib.wizard.model.status.BidibStatus;
import org.bidib.wizard.mvc.accessory.model.AccessoryModel;
import org.bidib.wizard.mvc.accessory.view.panel.AbstractAccessoryPanel;
import org.bidib.wizard.client.common.view.renderer.MacroRefRenderer;
import org.bidib.wizard.client.common.view.renderer.MacroRenderer;
import org.bidib.wizard.client.common.view.validation.DefaultRangeValidationCallback;
import org.bidib.wizard.client.common.view.validation.IntegerInputValidationDocument;
import org.bidib.wizard.mvc.main.controller.AccessoryPanelController;
import org.bidib.wizard.mvc.main.model.AccessoryAspect;
import org.bidib.wizard.mvc.main.model.AccessoryAspectMacro;
import org.bidib.wizard.mvc.main.model.AccessoryAspectParam;
import org.bidib.wizard.mvc.main.model.AccessoryStartupAspectModel;
import org.bidib.wizard.mvc.main.model.AccessoryTableModel;
import org.bidib.wizard.mvc.main.model.MainModel;
import org.bidib.wizard.mvc.main.model.SelectedAccessoryModel;
import org.bidib.wizard.mvc.main.model.listener.AccessoryPortListener;
import org.bidib.wizard.mvc.main.model.listener.AccessorySelectionListener;
import org.bidib.wizard.mvc.main.view.menu.AccessoryTableMenu;
import org.bidib.wizard.mvc.main.view.menu.listener.AccessoryTableMenuListener;
import org.bidib.wizard.client.common.table.AbstractStatusEmptyTable;
import org.bidib.wizard.mvc.main.view.table.AccessoryAspectRenderer;
import org.bidib.wizard.mvc.main.view.table.ComboBoxEditor;
import org.bidib.wizard.mvc.main.view.table.ComboBoxRenderer;
import org.bidib.wizard.mvc.main.view.table.ComboBoxWithButtonEditor;
import org.bidib.wizard.mvc.main.view.table.ComboBoxWithButtonRenderer;
import org.bidib.wizard.mvc.stepcontrol.controller.StepControlControllerInterface;
import org.bidib.wizard.mvc.stepcontrol.model.StepControlAspect;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.jgoodies.binding.adapter.Bindings;
import com.jgoodies.binding.adapter.ComboBoxAdapter;
import com.jgoodies.binding.beans.PropertyAdapter;
import com.jgoodies.binding.list.SelectionInList;
import com.jgoodies.binding.value.BindingConverter;
import com.jgoodies.binding.value.ConverterValueModel;
import com.jgoodies.binding.value.ValueHolder;
import com.jgoodies.binding.value.ValueModel;
import com.jgoodies.common.collect.ArrayListModel;
import com.jgoodies.forms.builder.FormBuilder;
import com.jgoodies.validation.ValidationResult;
import com.jgoodies.validation.ValidationResultModel;
import com.jgoodies.validation.util.DefaultValidationResultModel;
import com.jgoodies.validation.util.PropertyValidationSupport;
import com.jgoodies.validation.view.ValidationComponentUtils;
import com.jidesoft.swing.DefaultOverlayable;
import com.jidesoft.swing.StyledLabelBuilder;

public class AccessoryPanel extends JPanel implements AccessoryExecutionListener, AccessoryListener {
    private static final long serialVersionUID = 1L;

    private static final Logger LOGGER = LoggerFactory.getLogger(AccessoryPanel.class);

    private static final String ENCODED_COLUMN_SPECS =
        "pref, 3dlu, pref, 10dlu, pref, 3dlu, pref, 3dlu, pref, 3dlu, pref, 3dlu, pref:grow";

    private static final String ENCODED_ROW_SPECS = "pref, 3dlu, pref, 3dlu, pref";

    private final AbstractStatusEmptyTable table;

    private final AccessoryTableModel tableModel;

    private final AccessoryTableMenu accessoryTableMenu;

    private JLabel currentAspectLabel;

    private JLabel executionStateIconLabel;

    private ImageIcon accessoryErrorIcon;

    private ImageIcon accessorySuccessfulIcon;

    private ImageIcon accessoryWaitIcon;

    private ImageIcon accessoryUnknownIcon;

    private SelectionInList<AccessoryAspect> startupAspectSelection;

    private ValueModel selectionHolderStartupAspect;

    private final AccessoryStartupAspectModel accessoryStartupAspectModel;

    private final AccessorySwitchTimeModel accessorySwitchTimeModel;

    private final AccessoryPanelController accessoryPanelController;

    final MainModel mainModel;

    private MacroRef[] macrosClipBoard;

    private final SelectedAccessoryModel selectedAccessoryModel;

    private final PropertyChangeListener labelChangedListener;

    private final ValueModel selectedAccessoryValueModel;

    private ValueModel switchTimeValueModel;

    private JLabel labelSwitchTime;

    private JTextField switchTimeText;

    private JComponent[] baseUnitButtons;

    private final PropertyChangeListener startupAspectChangeListener;

    private final PropertyChangeListener switchTimeChangeListener;

    private final JLabel labelInitialState;

    private final JComboBox<Accessory> comboStartupAspect;

    protected ValidationResultModel validationModel;

    private ValueModel saveButtonEnabledModel = new ValueHolder(false);

    private final static class AccessoryConverter implements BindingConverter<Accessory, String> {

        @Override
        public String targetValue(Accessory sourceValue) {
            return Objects.toString(sourceValue, null);
        }

        @Override
        public Accessory sourceValue(String targetValue) {
            return null;
        }
    }

    private final static class AccessoryLabelRenderer extends DefaultTableCellRenderer {
        private static final long serialVersionUID = 1L;

        @Override
        public Component getTableCellRendererComponent(
            JTable table, Object value, boolean isSelected, boolean hasFocus, int row, int column) {

            Component comp = super.getTableCellRendererComponent(table, value, isSelected, hasFocus, row, column);
            if (value == null) {
                String textValue = Resources.getString(AccessoryTableModel.class, "aspect") + "_" + row;
                LOGGER.debug("Set the default value: {}", textValue);
                setText(textValue);
            }

            return comp;
        }
    }

    public AccessoryPanel(final AccessoryPanelController accessoryPanelController, final MainModel model,
        final AccessoryStartupAspectModel accessoryStartupAspectModel,
        final AccessorySwitchTimeModel accessorySwitchTimeModel) {

        this.mainModel = model;
        this.accessoryPanelController = accessoryPanelController;
        this.accessoryStartupAspectModel = accessoryStartupAspectModel;
        this.accessorySwitchTimeModel = accessorySwitchTimeModel;

        selectedAccessoryModel = new SelectedAccessoryModel();

        validationModel = new DefaultValidationResultModel();

        labelChangedListener = new PropertyChangeListener() {

            @Override
            public void propertyChange(PropertyChangeEvent evt) {
                LOGGER.info("The label has changed.");

                selectedAccessoryModel.triggerLabelChanged();
            }
        };

        initialize();

        setLayout(new BorderLayout());

        accessoryTableMenu = new AccessoryTableMenu(model);
        accessoryTableMenu.addMenuListener(new AccessoryTableMenuListener() {
            @Override
            public void delete() {
                int[] rows = table.getSelectedRows();
                if (rows != null && rows.length > 0) {
                    fireDelete(rows);
                }
            }

            @Override
            public void insertEmptyAfter() {
                fireInsertEmptyAfter(table.getSelectedRow());
            }

            @Override
            public void insertEmptyBefore() {
                fireInsertEmptyBefore(table.getSelectedRow());
            }

            @Override
            public void selectAll() {
                int rowCount = table.getRowCount();

                if (rowCount > 0) {
                    table.setRowSelectionInterval(0, rowCount - 1);
                }
            }

            @Override
            public void copy() {
                int[] rows = table.getSelectedRows();
                if (rows != null && rows.length > 0) {
                    fireCopy(rows);
                }
            }

            @Override
            public void cut() {
                int[] rows = table.getSelectedRows();
                if (rows != null && rows.length > 0) {
                    fireCut(rows);
                }
            }

            @Override
            public void pasteAfter() {
                firePasteAfter(table.getSelectedRow());
            }
        });

        tableModel = new AccessoryTableModel(model);

        FormBuilder formBuilder = FormBuilder.create().columns(ENCODED_COLUMN_SPECS).rows(ENCODED_ROW_SPECS);

        formBuilder.border(BorderFactory.createEmptyBorder(5, 5, 5, 5));

        selectedAccessoryValueModel =
            new PropertyAdapter<SelectedAccessoryModel>(selectedAccessoryModel,
                SelectedAccessoryModel.PROPERTY_SELECTED_ACCESSORY, true);

        ValueModel valueConverterModel = new ConverterValueModel(selectedAccessoryValueModel, new AccessoryConverter());

        JLabel selectedAccessoryLabel = WizardComponentFactory.createLabel(valueConverterModel);

        formBuilder.add(Resources.getString(getClass(), "accessoryName")).xy(1, 1);
        formBuilder.add(selectedAccessoryLabel).xyw(3, 1, 5);

        // this must be an aspect and not an accessory
        startupAspectSelection =
            new SelectionInList<AccessoryAspect>(
                (ListModel<AccessoryAspect>) accessoryStartupAspectModel.getAccessoryAspectList());

        selectionHolderStartupAspect =
            new PropertyAdapter<AccessoryStartupAspectModel>(accessoryStartupAspectModel,
                AccessoryStartupAspectModel.PROPERTY_SELECTED_STARTUP_ASPECT, true);

        ComboBoxAdapter<Accessory> comboBoxAdapterStartupAspect =
            new ComboBoxAdapter<Accessory>(startupAspectSelection, selectionHolderStartupAspect);
        comboStartupAspect = new JComboBox<>();
        comboStartupAspect.setModel(comboBoxAdapterStartupAspect);

        ValidationComponentUtils.setMessageKey(comboStartupAspect, "validation.startupAspect_key");

        labelInitialState = new JLabel(Resources.getString(getClass(), "initialState"));
        formBuilder.add(labelInitialState).xy(1, 3);
        formBuilder.add(comboStartupAspect).xy(3, 3);

        switchTimeValueModel =
            new PropertyAdapter<>(accessorySwitchTimeModel, AccessoryModel.PROPERTYNAME_SWITCH_TIME, true);

        final ValueModel switchTimeConverterModel =
            new ConverterValueModel(switchTimeValueModel, new StringConverter(new DecimalFormat("#")));

        switchTimeText = new JTextField();
        IntegerInputValidationDocument switchTimeDocument =
            new IntegerInputValidationDocument(3, IntegerInputValidationDocument.NUMERIC);
        switchTimeDocument.setRangeValidationCallback(new DefaultRangeValidationCallback(0, 127));
        switchTimeText.setDocument(switchTimeDocument);
        switchTimeText.setColumns(3);

        // bind manually because we changed the document of the textfield
        Bindings.bind(switchTimeText, switchTimeConverterModel, false);

        labelSwitchTime = new JLabel(Resources.getString(AbstractAccessoryPanel.class, "switchTime"));
        formBuilder.add(labelSwitchTime).xy(5, 3);
        formBuilder.add(switchTimeText).xy(7, 3);

        ValidationComponentUtils.setMandatory(switchTimeText, true);
        ValidationComponentUtils.setMessageKeys(switchTimeText, "validation.switchtime_key");

        ValueModel modeModel =
            new PropertyAdapter<>(accessorySwitchTimeModel, AccessorySwitchTimeModel.PROPERTYNAME_TIME_BASE_UNIT, true);
        baseUnitButtons = new JComponent[TimeBaseUnitEnum.values().length];

        int index = 0;
        int column = 9;
        for (TimeBaseUnitEnum baseUnit : TimeBaseUnitEnum.values()) {

            JRadioButton radio =
                WizardComponentFactory
                    .createRadioButton(modeModel, baseUnit,
                        Resources.getString(TimeBaseUnitEnum.class, baseUnit.getKey()));
            baseUnitButtons[index++] = radio;

            // add radio button
            formBuilder.add(radio).xy(column, 3);
            column += 2;
        }

        for (JComponent comp : baseUnitButtons) {
            comp.setEnabled(false);
        }
        switchTimeText.setEnabled(false);
        labelSwitchTime.setEnabled(false);

        // prepare the startup aspect components
        labelInitialState.setEnabled(false);
        comboStartupAspect.setEnabled(false);

        startupAspectChangeListener = new PropertyChangeListener() {

            @Override
            public void propertyChange(PropertyChangeEvent evt) {

                AccessoryAspect accessoryAspect = accessoryStartupAspectModel.getSelectedStartupAspect();
                boolean startupAspectEnabled = (accessoryAspect != null);
                LOGGER
                    .info("The startup aspect has changed, startupAspectEnabled: {}, selected startup aspect: {}",
                        startupAspectEnabled, accessoryAspect);

                labelInitialState.setEnabled(startupAspectEnabled);
                comboStartupAspect.setEnabled(startupAspectEnabled);

                // update the accessory to trigger pending changes
                if (accessoryAspect != null) {
                    Accessory selectedAccessory = tableModel.getSelectedAccessory();
                    if (selectedAccessory != null) {

                        if (accessoryAspect instanceof AccessoryAspectMacro) {
                            // aspect is selected
                            AccessoryAspectMacro accessoryAspectMacro =
                                (AccessoryAspectMacro) accessoryStartupAspectModel.getSelectedStartupAspect();
                            Integer startupState = accessoryAspectMacro.getIndex();
                            LOGGER.info("Set the startup state: {}", startupState);
                            selectedAccessory.setStartupState(startupState);
                        }
                        else if (accessoryAspect instanceof AccessoryAspectParam) {
                            // param is selected
                            AccessoryAspectParam accessoryAspectParam =
                                (AccessoryAspectParam) accessoryStartupAspectModel.getSelectedStartupAspect();

                            if (accessoryAspectParam.isInvalid()) {
                                LOGGER.warn("Do not set the invalid param!");
                                // TODO throw exception and signal the error
                                // this breaks the property change handler
                                // throw new IllegalArgumentException("The startup aspect is invalid.");

                            }
                            else {
                                Integer startupState = accessoryAspectParam.getParam();
                                LOGGER.info("Set the startup state: {}", startupState);
                                selectedAccessory.setStartupState(startupState);
                            }
                        }
                        else {
                            LOGGER
                                .info("The startup state is not an macro aspect or a param: {}",
                                    accessoryStartupAspectModel.getSelectedStartupAspect());
                        }
                    }
                }

                validate();
            }
        };

        this.accessoryStartupAspectModel
            .addPropertyChangeListener(AccessoryStartupAspectModel.PROPERTY_SELECTED_STARTUP_ASPECT,
                startupAspectChangeListener);

        switchTimeChangeListener = new PropertyChangeListener() {

            @Override
            public void propertyChange(PropertyChangeEvent evt) {
                LOGGER.info("The switch time has changed, evt: {}", evt);

                Accessory selectedAccessory = tableModel.getSelectedAccessory();

                if (selectedAccessory != null) {
                    switch (evt.getPropertyName()) {
                        case AccessorySwitchTimeModel.PROPERTYNAME_SWITCH_TIME:
                            Integer switchTime = accessorySwitchTimeModel.getSwitchTime();
                            LOGGER.info("Set the switchTime: {}", switchTime);
                            selectedAccessory.setSwitchTime(switchTime);
                            break;
                        case AccessorySwitchTimeModel.PROPERTYNAME_TIME_BASE_UNIT:
                            TimeBaseUnitEnum timeBaseUnit = accessorySwitchTimeModel.getTimeBaseUnit();
                            LOGGER.info("Set the timeBaseUnit: {}", timeBaseUnit);
                            selectedAccessory.setTimeBaseUnit(timeBaseUnit);
                            break;
                        default:
                            break;
                    }
                }
                validate();
            }
        };
        this.accessorySwitchTimeModel.addPropertyChangeListener(switchTimeChangeListener);

        JPanel executionStatePanel = new JPanel();
        executionStatePanel.setLayout(new BoxLayout(executionStatePanel, BoxLayout.LINE_AXIS));
        JLabel executionStateTextLabel = new JLabel(Resources.getString(AccessoryPanel.class, "execution_state"));
        executionStateTextLabel.setBorder(BorderFactory.createEmptyBorder(5, 0, 5, 0));
        executionStatePanel.add(executionStateTextLabel);
        executionStateIconLabel = new JLabel();
        executionStatePanel.add(executionStateIconLabel);

        formBuilder.add(executionStatePanel).xyw(1, 5, 9);

        currentAspectLabel = new JLabel();
        executionStatePanel.add(currentAspectLabel);
        formBuilder.add(currentAspectLabel).xyw(11, 5, 3);

        table = new AbstractStatusEmptyTable(tableModel, Resources.getString(AccessoryPanel.class, "emptyTable")) {
            private static final long serialVersionUID = 1L;

            @Override
            public void adjustRowHeight() {
                // set the correct row height
                SliderRenderer sliderEditor =
                    new SliderRenderer(0, 255, 10);

                int rowHeight =
                    sliderEditor.getTableCellRendererComponent(this, 1, false, false, 0, 0).getPreferredSize().height
                        + 6;
                LOGGER.info("Set row height: {}", rowHeight);

                setRowHeight(rowHeight);
            }

            @Override
            public TableCellEditor getCellEditor(int row, int column) {
                TableCellEditor result = super.getCellEditor(row, column);

                switch (column) {
                    case AccessoryTableModel.COLUMN_MACRO_REF:
                        NodeInterface node = model.getSelectedNode();
                        Macro[] macros = node.getMacros().toArray(new Macro[0]);
                        MacroRef[] macroRefs = new MacroRef[macros.length];
                        int index = 0;
                        for (Macro macro : macros) {
                            macroRefs[index] = new MacroRef(macro.getId());
                            index++;
                        }
                        JComboBox<MacroRef> comboboxMacroRef = new JComboBox<>(macroRefs);
                        comboboxMacroRef.setRenderer(new MacroRefRenderer(macros));
                        result = new ComboBoxEditor<MacroRef>(comboboxMacroRef) {
                            private static final long serialVersionUID = 1L;

                            @Override
                            public Component getTableCellEditorComponent(
                                JTable table, Object value, boolean isSelected, int row, int column) {

                                Component comp =
                                    super.getTableCellEditorComponent(table, value, isSelected, row, column);
                                if (value instanceof MacroRef) {
                                    MacroRef macroRef = (MacroRef) value;
                                    JComboBox<MacroRef> combo = ((JComboBox<MacroRef>) editorComponent);
                                    for (int index = 0; index < combo.getItemCount(); index++) {

                                        MacroRef macro = combo.getItemAt(index);
                                        if (macro.getId() == macroRef.getId()) {
                                            combo.setSelectedItem(macro);
                                            break;
                                        }
                                    }
                                }
                                else {
                                    LOGGER.debug("Current value is not a MacroRef: {}", value);
                                }

                                return comp;
                            };
                        };
                        LOGGER.info("Prepared the MacroRef editor.");
                        // result = new ComboBoxEditor<Macro>(model.getMacros().toArray(new Macro[0]));
                        break;
                    case AccessoryTableModel.COLUMN_ACCESSORY_ASPECT_INSTANCE:
                        ComboBoxWithButtonEditor editor =
                            new ComboBoxWithButtonEditor(table.getActions(AccessoryAspectStatus.START), ">");

                        // TODO initialize the correct value

                        editor.addButtonListener(tableModel);
                        result = editor;
                        break;
                    default:
                        break;
                }
                return result;
            }

            @Override
            public TableCellRenderer getCellRenderer(int row, int column) {
                TableCellRenderer result = null;

                switch (column) {
                    case AccessoryTableModel.COLUMN_LABEL:
                        // use special renderer for label
                        result = new AccessoryLabelRenderer();
                        break;
                    case AccessoryTableModel.COLUMN_MACRO_REF:
                        NodeInterface node = model.getSelectedNode();
                        ComboBoxRenderer<Macro> macroRenderer =
                            new ComboBoxRenderer<Macro>(node.getMacros().toArray(new Macro[0])) {
                                private static final long serialVersionUID = 1L;

                                @Override
                                public Component getTableCellRendererComponent(
                                    JTable table, Object value, boolean isSelected, boolean hasFocus, int row,
                                    int column) {
                                    Component comp =
                                        super.getTableCellRendererComponent(table, value, isSelected, hasFocus, row,
                                            column);

                                    if (value instanceof MacroRef) {
                                        MacroRef macroRef = (MacroRef) value;
                                        for (int index = 0; index < getItemCount(); index++) {

                                            Macro macro = getItemAt(index);
                                            if (macro.getId() == macroRef.getId()) {
                                                setSelectedItem(macro);
                                                break;
                                            }
                                        }
                                    }
                                    else {
                                        LOGGER.debug("Current value is not a MacroRef: {}", value);
                                    }

                                    comp.setEnabled(true);
                                    Object val = table.getModel().getValueAt(row, 2 /* accessory */);
                                    if (val instanceof AccessoryAspectMacro) {
                                        AccessoryAspectMacro accessory = (AccessoryAspectMacro) val;
                                        comp.setEnabled(!accessory.isImmutableAccessory());
                                    }

                                    return comp;
                                };
                            };
                        macroRenderer.setRenderer(new MacroRenderer());
                        result = macroRenderer;
                        break;
                    case AccessoryTableModel.COLUMN_ACCESSORY_ASPECT_INSTANCE:
                        result =
                            new ComboBoxWithButtonRenderer<BidibStatus>(table.getActions(AccessoryAspectStatus.START),
                                ">");
                        break;
                    default:
                        result = super.getCellRenderer(row, column);
                        break;
                }
                return result;
            }
        };
        TableColumn tc = table.getColumnModel().getColumn(AccessoryTableModel.COLUMN_LABEL);
        tc.setCellRenderer(new AccessoryAspectRenderer(Resources.getString(AccessoryTableModel.class, "aspect") + "_"));

        table.addMouseListener(new MouseAdapter() {

            @Override
            public void mousePressed(MouseEvent e) {
                handleMouseEvent(e, accessoryTableMenu);
            }

            @Override
            public void mouseReleased(MouseEvent e) {
                handleMouseEvent(e, accessoryTableMenu);
            }
        });

        table.putClientProperty("terminateEditOnFocusLost", Boolean.TRUE);
        table.adjustRowHeight();
        table.setSelectionMode(ListSelectionModel.SINGLE_SELECTION);

        table.getInputMap(JComponent.WHEN_ANCESTOR_OF_FOCUSED_COMPONENT).put(AccessoryTableMenu.KEYSTROKE_CUT, "cut");
        table.getActionMap().put("cut", new AbstractAction() {
            private static final long serialVersionUID = 1L;

            @Override
            public void actionPerformed(ActionEvent e) {
                accessoryTableMenu.fireCut();
            }
        });
        table.getInputMap(JComponent.WHEN_ANCESTOR_OF_FOCUSED_COMPONENT).put(AccessoryTableMenu.KEYSTROKE_COPY, "copy");
        table.getActionMap().put("copy", new AbstractAction() {
            private static final long serialVersionUID = 1L;

            @Override
            public void actionPerformed(ActionEvent e) {
                accessoryTableMenu.fireCopy();
            }
        });
        table
            .getInputMap(JComponent.WHEN_ANCESTOR_OF_FOCUSED_COMPONENT)
            .put(AccessoryTableMenu.KEYSTROKE_PASTE, "paste");
        table.getActionMap().put("paste", new AbstractAction() {
            private static final long serialVersionUID = 1L;

            @Override
            public void actionPerformed(ActionEvent e) {
                accessoryTableMenu.firePasteAfter();
            }
        });

        table
            .getInputMap(JComponent.WHEN_ANCESTOR_OF_FOCUSED_COMPONENT)
            .put(AccessoryTableMenu.KEYSTROKE_DELETE, "delete");
        table.getActionMap().put("delete", new AbstractAction() {
            private static final long serialVersionUID = 1L;

            @Override
            public void actionPerformed(ActionEvent e) {
                accessoryTableMenu.fireDelete();
            }
        });
        tableModel.addPortListener(new AccessoryPortListener() {
            @Override
            public void testButtonPressed(int accessoryId, int aspectId) {
                fireTestButtonPressed(aspectId);
            }

            @Override
            public void labelChanged(int accessoryId, int aspectIndex) {
                fireAspectLabelChanged(accessoryId, aspectIndex);
            }
        });

        final DefaultOverlayable overlayTable = new DefaultOverlayable(new JScrollPane(table));
        tableModel.addTableModelListener(new TableModelListener() {
            @Override
            public void tableChanged(TableModelEvent e) {
                overlayTable.setOverlayVisible(tableModel.getRowCount() == 0);
            }
        });

        overlayTable
            .addOverlayComponent(StyledLabelBuilder.createStyledLabel("{" + table.getEmptyTableText() + ":f:gray}"));

        formBuilder.appendRows("3dlu, fill:300px:grow");
        // formBuilder.nextLine(2);
        formBuilder.add(overlayTable).xyw(1, 7, 13);

        // Padding for overlay icon
        IconFeedbackPanel iconPanel = new IconFeedbackPanel(this.validationModel, formBuilder.build());

        add(iconPanel, BorderLayout.CENTER);

        this.mainModel.addAccessorySelectionListener(new AccessorySelectionListener() {

            @Override
            public void accessoryChanged() {
                LOGGER.info("The selected accessory has changed.");

                // the selected accessory has changed
                Accessory selectedAccessory = tableModel.getSelectedAccessory();
                if (selectedAccessory != null) {
                    selectedAccessory.removeAccessoryListener(AccessoryPanel.this);
                    selectedAccessory.removePropertyChangeListener(Accessory.PROPERTY_LABEL, labelChangedListener);
                }

                // get the new selected accessory
                final Accessory accessory = AccessoryPanel.this.mainModel.getSelectedAccessory();

                if (accessory != null) {
                    accessory.addAccessoryListener(AccessoryPanel.this);
                    accessory.addPropertyChangeListener(Accessory.PROPERTY_LABEL, labelChangedListener);
                }
                else {
                    LOGGER.info("No accessory selected.");
                    selectedAccessoryModel.triggerLabelChanged();
                }

                // set the new selected accessory
                tableModel.setSelectedAccessory(accessory);
                selectedAccessoryModel.setSelectedAccessory(accessory);

                // signal that the macros of the selected accessories have been changed
                try {
                    macrosChanged();

                    // clear the accessory execution state
                    if (accessory != null) {

                        Integer currentAspect = null;
                        if (accessory.getAccessoryState() != null) {
                            currentAspect = accessory.getAccessoryState().getAspect();
                        }

                        accessoryStateChanged(accessory.getId(), currentAspect);
                    }
                    else {
                        accessoryStateChanged(null, null);
                    }
                }
                catch (Exception ex) {
                    LOGGER.warn("Set the startup aspect failed.", ex);
                }
            }
        });

    }

    public ValueModel getSaveButtonEnabledModel() {
        return saveButtonEnabledModel;
    }

    /**
     * Initialize the component
     */
    protected void initialize() {
        // Set the icon for accessory execution
        accessoryErrorIcon = ImageUtils.createImageIcon(AccessoryPanel.class, "/icons/accessory-error.png");
        accessorySuccessfulIcon = ImageUtils.createImageIcon(AccessoryPanel.class, "/icons/accessory-successful.png");
        accessoryWaitIcon = ImageUtils.createImageIcon(AccessoryPanel.class, "/icons/accessory-wait.png");
        accessoryUnknownIcon = ImageUtils.createImageIcon(AccessoryPanel.class, "/icons/accessory-unknown.png");
    }

    @Override
    public void validate() {
        LOGGER.debug("Validate the values.");
        PropertyValidationSupport support =
            new PropertyValidationI18NSupport(selectionHolderStartupAspect, "validation");

        AccessoryAspect accessoryAspect = accessoryStartupAspectModel.getSelectedStartupAspect();
        if (accessoryAspect instanceof AccessoryAspectParam) {
            AccessoryAspectParam param = (AccessoryAspectParam) accessoryAspect;
            if (param.isInvalid()) {
                support.addError("startupAspect_key", "invalid_value");
            }
        }

        ValidationResult validationResult = support.getResult();

        validationModel.setResult(validationResult);

        // update the save button
        boolean hasErrors = validationModel.hasErrors();
        LOGGER.info("The current model has errors: {}", hasErrors);
        updateSaveButtonEnabled(hasErrors);

        if (selectedAccessoryValueModel.getValue() != null) {
            try {
                Accessory selectedAccessory = (Accessory) selectedAccessoryValueModel.getValue();
                selectedAccessory.setError(hasErrors);
            }
            catch (Exception ex) {
                LOGGER.warn("The selected accessory has an error.", ex);
            }
        }
        else {
            LOGGER.info("No selected accessory available.");
        }
    }

    private void updateSaveButtonEnabled(boolean validationHasErrors) {
        LOGGER.debug("Update the save button, validationHasErrors: {}", validationHasErrors);

        saveButtonEnabledModel.setValue(!validationHasErrors);
    }

    private void fireTestButtonPressed(int aspectNumber) {

        // check if the accessory has pending changes
        Accessory accessory = mainModel.getSelectedAccessory();
        if (accessory != null && (AccessorySaveState.PERMANENTLY_STORED_ON_NODE != accessory.getAccessorySaveState())) {
            // show a hint to transfer the macro to the node
            int result =
                JOptionPane
                    .showConfirmDialog(this,
                        Resources.getString(AccessoryPanel.class, "accessory_transfer_pending_changes_before_test"),
                        Resources.getString(AccessoryPanel.class, "test"), JOptionPane.YES_NO_OPTION,
                        JOptionPane.WARNING_MESSAGE);

            if (result == JOptionPane.YES_OPTION) {
                accessoryPanelController.storeAccessory(accessory);
            }
        }

        accessoryPanelController.activateAspect(accessory, aspectNumber);
    }

    private int getRow(Point point) {
        return table.rowAtPoint(point);
    }

    private void handleMouseEvent(MouseEvent e, AccessoryTableMenu popupMenu) {
        if (e.isPopupTrigger()) {
            int row = getRow(e.getPoint());
            // stop table editing before the menu is displayed
            if (table.getCellEditor() != null) {
                table.getCellEditor().cancelCellEditing();
            }

            if (table.getSelectedRowCount() == 0 && table.getRowCount() > 0 && row >= 0 && row < table.getRowCount()) {
                table.setRowSelectionInterval(row, row);
            }

            int selectedRow = table.getSelectedRow();
            if (selectedRow > -1) {
                try {
                    Object value = tableModel.getValueAt(selectedRow, 2);
                    if (value instanceof AccessoryAspectMacro) {
                        boolean immutable = ((AccessoryAspectMacro) value).isImmutableAccessory();
                        if (immutable) {
                            popupMenu.updateImmutableMenuItems();
                        }
                    }
                }
                catch (Exception ex) {
                    LOGGER.warn("Update menu items for immutable aspect failed.", ex);
                }
            }

            table.requestFocusInWindow();
            // table.grabFocus();

            popupMenu.show(e.getComponent(), e.getX(), e.getY());
        }
    }

    @Override
    public void executionStateChanged(
        AccessoryExecutionState executionState, Integer accessoryId, Integer aspect, AccessoryState accessoryState) {
        LOGGER
            .info("The execution state has changed: {}, accessoryId: {}, aspect: {}", executionState, accessoryId,
                aspect);
        if (aspect == null || executionState == null) {
            executionState = AccessoryExecutionState.IDLE;
        }

        executionStateIconLabel.setToolTipText(null);
        switch (executionState) {
            case ERROR:
                executionStateIconLabel.setIcon(accessoryErrorIcon);
                if (accessoryState != null) {
                    executionStateIconLabel.setToolTipText(accessoryState.getErrorInformation());
                }
                break;
            case RUNNING:
                executionStateIconLabel.setIcon(accessoryWaitIcon);
                break;
            case SUCCESSFUL:
                executionStateIconLabel.setIcon(accessorySuccessfulIcon);
                break;
            case UNKNOWN:
                executionStateIconLabel.setIcon(accessoryUnknownIcon);
                break;
            default:
                executionStateIconLabel.setIcon(null);
                break;
        }

        if (selectedAccessoryModel.getSelectedAccessory() != null) {
            final Accessory accessory = selectedAccessoryModel.getSelectedAccessory();
            refreshCurrentAspectLabel(accessory, aspect, accessoryState);
        }
    }

    private void refreshCurrentAspectLabel(final Accessory accessory, Integer aspect, AccessoryState accessoryState) {
        LOGGER.info("The selected accessory: {}, active aspect: {}", accessory, aspect);

        currentAspectLabel.setText(null);

        if (accessoryState != null && !accessoryState.hasError()) {
            if (aspect != null && aspect.intValue() < tableModel.getRowCount()) {
                Object value = tableModel.getValueAt(aspect.intValue(), AccessoryTableModel.COLUMN_LABEL);

                if (value == null) {
                    value = Resources.getString(AccessoryTableModel.class, "aspect") + "_" + aspect;
                }

                currentAspectLabel
                    .setText(Resources
                        .getString(AccessoryPanel.class, "activeAspect",
                            String.format("%1$02d : %2$s", aspect, value)));
            }
        }
    }

    @Override
    public void labelChanged(String label) {
    }

    private ArrayListModel<StepControlAspect> stepControlAspects;

    private ListDataListener stepControlAspectsListener;

    private void loadStepControlMovingAspects(final AccessoryLabel accessoryLabel) {
        LOGGER.info("Load the StepControl moving aspects.");

        // remove all rows
        tableModel.setRowCount(0);
        accessoryStartupAspectModel.clearAccessoryAspects();

        int aspectNumber = 0;
        for (StepControlAspect stepControlAspect : stepControlAspects) {
            LOGGER.info("Create accessory aspect for stepControlAspect: {}", stepControlAspect);

            AccessoryAspectMacro aspectTurntable = new AccessoryAspectMacro(aspectNumber, null);
            aspectTurntable.setImmutableAccessory(true);

            // try to get the user-defined label
            aspectTurntable.setLabel(getAccessoryAspectLabel(accessoryLabel, aspectNumber));

            tableModel.addRow(aspectTurntable);
            accessoryStartupAspectModel.addAccessoryAspect(aspectTurntable);

            aspectNumber++;
        }

    }

    @Override
    public void macrosChanged() {
        LOGGER.info("The macros have changed.");

        // remove all rows
        tableModel.setRowCount(0);

        accessoryStartupAspectModel.clearAccessoryAspects();

        // release the stepControlAspects
        if (stepControlAspects != null) {
            if (stepControlAspectsListener != null) {
                stepControlAspects.removeListDataListener(stepControlAspectsListener);
            }
            stepControlAspects = null;
        }

        // get the selected accessory and prepare the aspects
        final Accessory selectedAccessory = tableModel.getSelectedAccessory();
        if (selectedAccessory != null) {

            final NodeInterface selectedNode = accessoryPanelController.getSelectedNode();
            long uniqueId = accessoryPanelController.getSelectedNode().getUniqueId();
            boolean isStepControl = ProductUtils.isStepControl(uniqueId);

            // in case of the Step Control we have the first 3 accessories fix
            if (isStepControl && selectedAccessory.getId() < 3) {
                LOGGER.info("Special processing for StepControl.");

                AccessoryLabels accessoryLabels = null;
                final VendorCV vendorCv = selectedNode.getVendorCV().getVendorCV();

                // stepControl has special labels in extension block
                if (vendorCv.getExtension() != null
                    && CollectionUtils.isNotEmpty(vendorCv.getExtension().getNodeLabels())) {
                    // get the default labels
                    final NodeLabels nodeLabels = vendorCv.getExtension().getNodeLabels().get(0);

                    if (nodeLabels != null) {
                        LOGGER.info("Use node labels from extension in vendorCV.");
                        accessoryLabels = nodeLabels.getAccessoryLabels();
                    }
                }

                LOGGER.info("Current accessory id: {}", selectedAccessory.getId());
                switch (selectedAccessory.getId()) {
                    case 0:
                        // get the configured moving aspects
                        try {
                            // get the saved accessory labels
                            final AccessoryLabel accessoryLabel =
                                accessoryPanelController.getAccessoryAspectsLabels(selectedAccessory.getId());

                            StepControlControllerInterface controller =
                                DefaultApplicationContext
                                    .getInstance().get(DefaultApplicationContext.KEY_STEPCONTROL_CONTROLLER,
                                        StepControlControllerInterface.class);

                            stepControlAspects = controller.getConfigureAspectsListModel();

                            if (stepControlAspectsListener == null) {
                                LOGGER.info("Create new stepControlAspectsListener.");
                                stepControlAspectsListener = new ListDataListener() {

                                    @Override
                                    public void intervalRemoved(ListDataEvent e) {
                                        LOGGER.info("intervalRemoved.");

                                        loadStepControlMovingAspects(accessoryLabel);
                                    }

                                    @Override
                                    public void intervalAdded(ListDataEvent e) {
                                        LOGGER.info("intervalAdded.");

                                        loadStepControlMovingAspects(accessoryLabel);
                                    }

                                    @Override
                                    public void contentsChanged(ListDataEvent e) {
                                        LOGGER.info("The contents of the stepControl aspects have changed.");
                                    }
                                };
                                LOGGER
                                    .info("Created new instance of stepControlAspectsListener: {}",
                                        stepControlAspectsListener);
                            }
                            stepControlAspects.addListDataListener(stepControlAspectsListener);

                            loadStepControlMovingAspects(accessoryLabel);
                        }
                        catch (Exception ex) {
                            LOGGER.warn("Load configured moving aspects failed", ex);
                        }
                        break;
                    case 1:
                        // the homing aspect

                        if (CollectionUtils.isNotEmpty(selectedAccessory.getAspects())
                            || selectedAccessory.getTotalAspects() > 0) {
                            AccessoryLabel accessoryLabel = null;
                            if (accessoryLabels != null) {
                                accessoryLabel =
                                    AccessoryLabelUtils.getAccessoryLabel(accessoryLabels, selectedAccessory.getId());
                            }
                            if (accessoryLabel == null) {
                                accessoryLabel =
                                    accessoryPanelController.getAccessoryAspectsLabels(selectedAccessory.getId());
                            }

                            // prepare the aspects of the selected accessory
                            int totalAspects = 0;
                            if (CollectionUtils.isNotEmpty(selectedAccessory.getAspects())) {
                                totalAspects = selectedAccessory.getAspects().size();
                            }
                            else {
                                totalAspects = selectedAccessory.getTotalAspects();
                            }
                            LOGGER.info("The homing accessory has totalAspects: {}", totalAspects);

                            for (int row = 0; row < totalAspects; row++) {

                                LOGGER.info("Adding new aspect, row: {}", row);

                                AccessoryAspectMacro accessoryAspect = new AccessoryAspectMacro(row, null);

                                // try to get the user-defined label
                                accessoryAspect.setLabel(getAccessoryAspectLabel(accessoryLabel, row));
                                accessoryAspect.setImmutableAccessory(true);

                                tableModel.addRow(accessoryAspect);

                                accessoryStartupAspectModel.addAccessoryAspect(accessoryAspect);
                            }
                        }
                        else {
                            LOGGER.warn("No aspects for HOMING accessory available.");
                        }

                        break;
                    default:
                        // the sound aspect
                        if (CollectionUtils.isNotEmpty(selectedAccessory.getAspects())
                            || selectedAccessory.getTotalAspects() > 0) {
                            AccessoryLabel accessoryLabel = null;
                            if (accessoryLabels != null) {
                                accessoryLabel =
                                    AccessoryLabelUtils.getAccessoryLabel(accessoryLabels, selectedAccessory.getId());
                            }
                            if (accessoryLabel == null) {
                                accessoryLabel =
                                    accessoryPanelController.getAccessoryAspectsLabels(selectedAccessory.getId());
                            }

                            // prepare the aspects of the selected accessory
                            int totalAspects = 0;
                            if (CollectionUtils.isNotEmpty(selectedAccessory.getAspects())) {
                                totalAspects = selectedAccessory.getAspects().size();
                            }
                            else {
                                totalAspects = selectedAccessory.getTotalAspects();
                            }
                            LOGGER.info("The sound accessory has totalAspects: {}", totalAspects);

                            for (int row = 0; row < totalAspects; row++) {

                                LOGGER.info("Adding new aspect, row: {}", row);

                                AccessoryAspectMacro accessoryAspect = new AccessoryAspectMacro(row, null);

                                // try to get the user-defined label
                                accessoryAspect.setLabel(getAccessoryAspectLabel(accessoryLabel, row));
                                accessoryAspect.setImmutableAccessory(true);

                                tableModel.addRow(accessoryAspect);

                                accessoryStartupAspectModel.addAccessoryAspect(accessoryAspect);
                            }
                        }
                        break;
                }
            }
            else {

                // processing of non-StepControl nodes
                AccessoryLabel accessoryLabel =
                    accessoryPanelController.getAccessoryAspectsLabels(selectedAccessory.getId());

                // check if the current accessory has aspects defined
                if (CollectionUtils.isNotEmpty(selectedAccessory.getAspects())) {

                    // prepare the aspects of the selected accessory
                    // get all macros
                    List<Macro> macros = accessoryPanelController.getMacros();

                    MacroRef[] aspects = selectedAccessory.getAspects().toArray(new MacroRef[0]);
                    for (int row = 0; row < aspects.length; row++) {
                        // get the number of the macro of the selected aspect

                        MacroRef currentMacro = aspects[row];
                        if (currentMacro != null && currentMacro.getId() != null) {
                            final int macroNumber = currentMacro.getId();

                            LOGGER.info("Adding new aspect, row: {}, macroNumber: {}", row, macroNumber);

                            if (macroNumber > -1 && macroNumber < macros.size()) {
                                AccessoryAspectMacro accessoryAspect = new AccessoryAspectMacro(row, currentMacro);

                                // try to get the user-defined label
                                accessoryAspect.setLabel(getAccessoryAspectLabel(accessoryLabel, row));

                                // tableModel.addRow(row, macros.get(macroNumber));
                                tableModel.addRow(accessoryAspect);

                                accessoryStartupAspectModel.addAccessoryAspect(accessoryAspect);
                            }
                        }
                    }
                }
                else if (selectedAccessory.getTotalAspects() > 0) {

                    int totalAspects = selectedAccessory.getTotalAspects();

                    if (totalAspects > 0) {
                        for (int row = 0; row < totalAspects; row++) {
                            AccessoryAspectMacro accessoryAspect = new AccessoryAspectMacro(row, null);

                            // try to get the user-defined label
                            accessoryAspect.setLabel(getAccessoryAspectLabel(accessoryLabel, row));

                            // TODO prepare AccessoryAspects for non macro mapped nodes
                            accessoryAspect.setImmutableAccessory(true);

                            tableModel.addRow(accessoryAspect);

                            accessoryStartupAspectModel.addAccessoryAspect(accessoryAspect);
                        }
                    }
                }
                else {
                    LOGGER.warn("No aspects available for accessory: {}", selectedAccessory);
                }

            }
        }

        // evaluate the startup aspect
        try {
            // remove the property change listener
            this.accessoryStartupAspectModel
                .removePropertyChangeListener(AccessoryStartupAspectModel.PROPERTY_SELECTED_STARTUP_ASPECT,
                    startupAspectChangeListener);

            if (selectedAccessory != null && selectedAccessory.getStartupState() != null) {

                int aspectCount =
                    (selectedAccessory.isMacroMapped() && selectedAccessory.getAspectCount() > 0)
                        ? selectedAccessory.getAspectCount() : selectedAccessory.getTotalAspects();
                try {
                    AccessoryAspect startupAspect =
                        accessoryStartupAspectModel.getAssignedAspect(selectedAccessory.getStartupState(), aspectCount);
                    accessoryStartupAspectModel.setSelectedStartupAspect(startupAspect);
                }
                catch (Exception ex) {
                    LOGGER.warn("Set the selected startup aspect failed. Set the invalid aspect.", ex);
                    accessoryStartupAspectModel.setSelectedStartupAspect(startupAspectSelection.getElementAt(0));
                }
            }
            else {
                LOGGER.info("No startup aspect available.");
                accessoryStartupAspectModel.setSelectedStartupAspect(null);
            }

            AccessoryAspect accessoryAspect = accessoryStartupAspectModel.getSelectedStartupAspect();
            boolean startupAspectEnabled = (accessoryAspect != null);
            LOGGER
                .info("The startup aspect has changed, startupAspectEnabled: {}, selected startup aspect: {}",
                    startupAspectEnabled, accessoryAspect);

            labelInitialState.setEnabled(startupAspectEnabled);
            comboStartupAspect.setEnabled(startupAspectEnabled);

        }
        finally {
            // add the property change listener
            this.accessoryStartupAspectModel
                .addPropertyChangeListener(AccessoryStartupAspectModel.PROPERTY_SELECTED_STARTUP_ASPECT,
                    startupAspectChangeListener);
        }

        // evaluate the switch time
        try {
            // remove the property change listener
            this.accessorySwitchTimeModel.removePropertyChangeListener(switchTimeChangeListener);

            boolean switchTimeEnabled = (selectedAccessory != null && selectedAccessory.getSwitchTime() != null);

            LOGGER.info("The accessory switch time has changed, startupAspectEnabled: {}", switchTimeEnabled);

            for (JComponent comp : baseUnitButtons) {
                comp.setEnabled(switchTimeEnabled);
            }
            switchTimeText.setEnabled(switchTimeEnabled);
            labelSwitchTime.setEnabled(switchTimeEnabled);

            if (switchTimeEnabled) {
                Integer switchTime = selectedAccessory.getSwitchTime();
                TimeBaseUnitEnum timeBaseUnit = selectedAccessory.getTimeBaseUnit();

                accessorySwitchTimeModel.setSwitchTime(switchTime);
                accessorySwitchTimeModel.setTimeBaseUnit(timeBaseUnit);
            }
            else {
                accessorySwitchTimeModel.setSwitchTime(null);
                accessorySwitchTimeModel.setTimeBaseUnit(null);
            }
        }
        finally {
            // add the property change listener
            this.accessorySwitchTimeModel.addPropertyChangeListener(switchTimeChangeListener);
        }

        validate();
    }

    private String getAccessoryAspectLabel(AccessoryLabel labels, int aspectIndex) {

        if (labels != null) {
            BaseLabel baseLabel = AccessoryLabelUtils.getAccessoryAspectLabel(labels, aspectIndex);
            if (baseLabel != null) {
                String labelString = baseLabel.getLabel();
                LOGGER.info("Found the aspect label: {}", labelString);
                return labelString;
            }
        }
        return null;
    }

    @Override
    public void accessoryStateChanged(Integer accessoryId, Integer aspect) {
        LOGGER.info("The accessory state has changed for accessory id: {}, aspect: {}", accessoryId, aspect);

        final Accessory selectedAccessory = tableModel.getSelectedAccessory();
        if (selectedAccessory != null && selectedAccessory.getId() == accessoryId) {
            AccessoryExecutionState executionState = selectedAccessory.getAccessoryExecutionState();
            AccessoryState accessoryState = selectedAccessory.getAccessoryState();

            executionStateChanged(executionState, accessoryId, aspect, accessoryState);
        }
        else {
            executionStateChanged(null, accessoryId, aspect, null);
        }
    }

    private void fireAspectLabelChanged(final int accessoryId, final int aspectIndex) {

        String label = null;

        if (aspectIndex < tableModel.getRowCount()) {
            label = (String) tableModel.getValueAt(aspectIndex, AccessoryTableModel.COLUMN_LABEL);

            MacroRef aspect = null;

            List<MacroRef> aspects = mainModel.getSelectedAccessory().getAspects();
            if (aspects.size() > aspectIndex) {
                aspect = mainModel.getSelectedAccessory().getAspects().toArray(new MacroRef[0])[aspectIndex];
            }
            LOGGER
                .info("The aspect label has changed, accessoryId: {}, aspectIndex: {}, label: {}", accessoryId,
                    aspectIndex, label);

            // if the label is empty we trim to null
            label = StringUtils.trimToNull(label);

            if (aspect != null) {
                LOGGER.info("Set the changed label on aspect: {}, label: {}", aspect, label);
                aspect.setLabel(label);
            }
        }

        accessoryPanelController.setAccessoryAspectLabel(accessoryId, aspectIndex, label);
    }

    private MacroRef[] getMacros(int[] rows) {
        MacroRef[] result = new MacroRef[rows.length];

        try {
            final Accessory selectedAccessory = tableModel.getSelectedAccessory();
            MacroRef[] aspects = selectedAccessory.getAspects().toArray(new MacroRef[0]);
            for (int index = 0; index < rows.length; index++) {
                result[index] = aspects[rows[index]];
            }

            LOGGER.info("Prepared macros: {}", new Object[] { result });
        }
        catch (Exception ex) {
            LOGGER.warn("Get selected macros failed.", ex);
        }
        return result;
    }

    public void saveAccessory(final Accessory selectedAccessory) {
        LOGGER.info("Save the values to the selected accessory: {}", selectedAccessory);

        if (accessoryStartupAspectModel.getSelectedStartupAspect() instanceof AccessoryAspectMacro) {
            // aspect is selected
            AccessoryAspectMacro accessoryAspectMacro =
                (AccessoryAspectMacro) accessoryStartupAspectModel.getSelectedStartupAspect();
            Integer startupState = accessoryAspectMacro.getIndex();
            LOGGER.info("Set the startup state: {}", startupState);
            selectedAccessory.setStartupState(startupState);
        }
        else if (accessoryStartupAspectModel.getSelectedStartupAspect() instanceof AccessoryAspectParam) {
            // param is selected
            AccessoryAspectParam accessoryAspectParam =
                (AccessoryAspectParam) accessoryStartupAspectModel.getSelectedStartupAspect();

            if (accessoryAspectParam.isInvalid()) {
                // TODO throw an exception
                LOGGER.warn("The startup aspect is invalid.");

                throw new IllegalArgumentException("The startup aspect is invalid.");
            }
            else {
                Integer startupState = accessoryAspectParam.getParam();
                LOGGER.info("Set the startup state of the param: {}", startupState);
                selectedAccessory.setStartupState(startupState);
            }
        }
        else {
            LOGGER
                .info("The startup state is not an macro aspect or a param: {}",
                    accessoryStartupAspectModel.getSelectedStartupAspect());
        }
    }

    private void setMacrosToClipboard(MacroRef[] macros) {
        if (macros != null) {
            try {
                this.macrosClipBoard = new MacroRef[macros.length];
                for (int index = 0; index < macros.length; index++) {
                    if (macros[index] != null) {
                        this.macrosClipBoard[index] = new MacroRef(macros[index].getId());
                    }
                }
            }
            catch (Exception e) {
                throw new RuntimeException(e);
            }
        }
        else {
            this.macrosClipBoard = null;
        }
    }

    private MacroRef[] getMacrosFromClipboard() {
        MacroRef[] result = null;

        if (macrosClipBoard != null) {
            try {
                result = new MacroRef[macrosClipBoard.length];
                for (int index = 0; index < macrosClipBoard.length; index++) {
                    if (macrosClipBoard[index] != null) {
                        result[index] = new MacroRef(macrosClipBoard[index].getId());
                    }
                }
            }
            catch (Exception ex) {
                LOGGER.warn("Get macros from clipboard failed.", ex);
                throw new RuntimeException(ex);
            }
        }
        return result;
    }

    private void fireDelete(int[] rows) {

        // delete the rows
        LOGGER.info("Delete the aspect row at index: {}", rows[0]);
        if (rows.length == 0) {
            return;
        }

        // first move the labels and assigned macros up and then delete the last aspect
        final AccessoryTableModel accessoryTableModel = tableModel;

        // move all aspects after the delete aspect 1 index up
        int currentIndex = 0;
        for (currentIndex = rows[0]; currentIndex < (accessoryTableModel.getRowCount() - 1); currentIndex++) {
            // move the labels and assigned macros up

            LOGGER.info("Current index: {}", currentIndex);

            AccessoryAspect accessoryAspect = accessoryTableModel.getAccessoryAspectInstance(currentIndex + 1);

            if (accessoryAspect instanceof AccessoryAspectMacro) {
                AccessoryAspectMacro source = (AccessoryAspectMacro) accessoryAspect;
                LOGGER.info("Current source label: {}, macro: {}", source, source.getMacroRef());

                // update the table model
                accessoryTableModel
                    .setValueAt(source.getMacroRef(), currentIndex, AccessoryTableModel.COLUMN_MACRO_REF);
            }
            accessoryTableModel.setValueAt(accessoryAspect.getLabel(), currentIndex, AccessoryTableModel.COLUMN_LABEL);
        }

        LOGGER.info("Delete the aspect at index: {}", currentIndex);

        mainModel.getSelectedAccessory().removeAspect(currentIndex);

        fireAspectLabelChanged(selectedAccessoryModel.getSelectedAccessory().getId(), currentIndex);

        accessoryTableModel.fireTableDataChanged();
    }

    private MacroRef findNextFreeMacro() {

        SortedSet<Integer> macroIds = new TreeSet<>();
        for (Accessory accessory : mainModel.getAccessories()) {

            for (MacroRef aspect : accessory.getAspects()) {

                macroIds.add(aspect.getId());
            }
        }

        // macro id starts from 0
        int macroId = 0;
        for (Integer currentId : macroIds) {
            if (currentId.intValue() == macroId) {
                macroId++;
            }
            else {
                LOGGER.info("Found free macroId: {}", macroId);
            }
        }
        MacroRef macroRef = null;
        if (macroId < mainModel.getMacros().size()) {
            LOGGER.info("Create new MacroRef with macroId: {}", macroId);
            macroRef = new MacroRef(macroId);
        }
        else {
            macroRef = new MacroRef();
        }
        return macroRef;
    }

    private void fireInsertEmptyAfter(int row) {
        MacroRef macroRef = findNextFreeMacro();

        // TODO remove aspect must refresh the aspect labels of the accessory

        mainModel.getSelectedAccessory().addAspectAfter(row, macroRef);
    }

    private void fireInsertEmptyBefore(int row) {
        MacroRef macroRef = findNextFreeMacro();
        mainModel.getSelectedAccessory().addAspectBefore(row >= 0 ? row : 0, macroRef);
    }

    private void fireCopy(int[] rows) {
        setMacrosToClipboard(getMacros(rows));
    }

    private void fireCut(int[] rows) {
        MacroRef[] macros = getMacros(rows);
        setMacrosToClipboard(macros);
        for (int index = rows.length - 1; index >= 0; index--) {
            mainModel.getSelectedAccessory().removeAspect(rows[index]);
        }
    }

    private void firePasteAfter(int row) {
        MacroRef[] macros = getMacrosFromClipboard();
        if (macros != null) {
            mainModel.getSelectedAccessory().addAspectsAfter(row, macros);
        }
        else {
            MacroRef macroRef = findNextFreeMacro();
            mainModel.getSelectedAccessory().addAspectAfter(row, macroRef);
        }
    }

    public void changeLabel(AccessoryAspectTargetType aspectTargetType) {
        LOGGER.info("Change the label of the aspect: {}", aspectTargetType);

        int accessoryNumber = aspectTargetType.getAccessoryNum();
        int aspectNumber = aspectTargetType.getPortNum();
        String aspectLabel = aspectTargetType.getLabel();

        // set the aspect label
        Accessory selectedAccessory = selectedAccessoryModel.getSelectedAccessory();
        if (selectedAccessory != null && selectedAccessory.getId() == accessoryNumber) {
            // make sure the row with the aspect exists
            if (tableModel.getRowCount() < (aspectNumber + 1)) {
                for (int row = tableModel.getRowCount(); row < (aspectNumber + 1); row++) {
                    LOGGER.info("Add new row: {}", row);
                    tableModel.addRow(new AccessoryAspectMacro(aspectNumber, null));
                }
            }
            tableModel.setValueAt(aspectLabel, aspectNumber, AccessoryTableModel.COLUMN_LABEL);
        }
        else {
            accessoryPanelController.setAccessoryAspectLabel(accessoryNumber, aspectNumber, aspectLabel, true);
        }
    }

    public void refreshView() {
        macrosChanged();

        final Accessory accessory = selectedAccessoryModel.getSelectedAccessory();
        if (accessory != null) {
            Integer currentAspect = accessory.getAccessoryState().getAspect();
            refreshCurrentAspectLabel(accessory, currentAspect, accessory.getAccessoryState());
        }
    }
}
