package org.bidib.wizard.mvc.main.view.panel;

import java.awt.BorderLayout;
import java.util.LinkedList;
import java.util.List;

import javax.swing.JPanel;
import javax.swing.JScrollPane;
import javax.swing.ListSelectionModel;
import javax.swing.border.EmptyBorder;
import javax.swing.event.TableModelEvent;
import javax.swing.event.TableModelListener;

import org.bidib.wizard.api.locale.Resources;
import org.bidib.wizard.api.model.listener.PortListListener;
import org.bidib.wizard.api.model.listener.PortListener;
import org.bidib.wizard.api.model.listener.PortListenerProvider;
import org.bidib.wizard.api.utils.PortListUtils;
import org.bidib.wizard.common.script.node.types.TargetType;
import org.bidib.wizard.model.ports.Port;
import org.bidib.wizard.model.status.BidibStatus;
import org.bidib.wizard.client.common.model.SimplePortTableModel;
import org.bidib.wizard.client.common.model.listener.PortModelListener;
import org.bidib.wizard.client.common.table.PortTable;
import org.bidib.wizard.nodescript.script.node.ChangeLabelSupport;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.jidesoft.swing.DefaultOverlayable;
import com.jidesoft.swing.StyledLabelBuilder;

public abstract class SimplePortListPanel<S extends BidibStatus, P extends Port<S>, L extends PortListener<P>, M extends PortModelListener<P>>
    extends JPanel
    implements PortListListener, ChangeLabelSupport, PortListenerProvider<P> {
    private static final long serialVersionUID = 1L;

    protected final Logger LOGGER = LoggerFactory.getLogger(getClass());

    protected SimplePortTableModel<S, P, M> tableModel;

    protected PortTable table;

    protected L portListener;

    public SimplePortListPanel(final SimplePortTableModel<S, P, M> tableModel, String emptyTableText) {
        this.tableModel = tableModel;

        // this.ports = ports;
        setLayout(new BorderLayout());

        setBorder(new EmptyBorder(5, 5, 5, 5));

        createTable(tableModel, emptyTableText);

        final DefaultOverlayable overlayTable = new DefaultOverlayable(new JScrollPane(table));
        table.getModel().addTableModelListener(new TableModelListener() {
            @Override
            public void tableChanged(TableModelEvent e) {
                overlayTable.setOverlayVisible(table.getModel().getRowCount() == 0);
            }
        });

        overlayTable
            .addOverlayComponent(StyledLabelBuilder.createStyledLabel("{" + table.getEmptyTableText() + ":f:gray}"));

        add(overlayTable, BorderLayout.CENTER);
    }

    protected SimplePortTableModel<S, P, M> getTableModel() {
        return tableModel;
    }

    protected void createTable(final SimplePortTableModel<S, P, M> tableModel, String emptyTableText) {

        table = createPortTable(tableModel, emptyTableText);
        table.adjustRowHeight();

        // do not allow drag columns to other position
        table.getTableHeader().setReorderingAllowed(false);

        table.putClientProperty("terminateEditOnFocusLost", Boolean.TRUE);

        // disabled sorting
        table.setSortable(false);
        table.setSelectionMode(ListSelectionModel.SINGLE_SELECTION);
    }

    protected PortTable createPortTable(final SimplePortTableModel<S, P, M> tableModel, String emptyTableText) {
        return new PortTable(tableModel, emptyTableText) {
            private static final long serialVersionUID = 1L;

            @Override
            public void clearTable() {
            }
        };
    }

    public void setPortListener(final L listener) {
        portListener = listener;

    }

    @Override
    public String getName() {
        return Resources.getString(getClass(), "name");
    }

    @Override
    public void listChanged() {
        if (tableModel == null) {
            return;
        }

        LOGGER.info("The port list has changed in SimplePortListPanel.");

        // refreshPorts();

        tableModel.setRowCount(0);

        List<P> ports = new LinkedList<>();
        ports.addAll(getPorts());
        synchronized (ports) {
            for (P port : ports) {
                LOGGER.info("Adding row for port: {}", port);
                tableModel.addRow(port);
            }
        }

        LOGGER.info("The port list has changed has finished in SimplePortListPanel.");
    }

    @Override
    public void changeLabel(TargetType portType) {
        int portNum = portType.getPortNum();
        String label = portType.getLabel();

        P port = null;
        List<P> ports = getPorts();
        synchronized (ports) {
            port = PortListUtils.findPortByPortNumber(ports, portNum);
        }

        if (port != null) {
            LOGGER.info("Set the port label: {}", label);
            port.setLabel(label);

            try {
                portListener.labelChanged(port, label);
            }
            catch (Exception ex) {
                LOGGER.warn("Change port label failed.", ex);
            }
        }
    }

    /**
     * Get the list of ports.
     * 
     * @return the list of ports
     */
    protected abstract List<P> getPorts();

    @Override
    public PortListener<P> getPortListener() {
        return portListener;
    }

}
