package org.bidib.wizard.mvc.tips.controller;

import java.io.File;
import java.io.FileInputStream;
import java.io.InputStream;
import java.net.URL;
import java.util.Locale;

import org.apache.commons.lang3.time.StopWatch;
import org.bidib.wizard.core.service.SettingsService;
import org.bidib.wizard.mvc.main.view.MainView;
import org.bidib.wizard.mvc.tips.view.TipOfDayView;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;

import com.jidesoft.tipoftheday.TipOfTheDaySource;

public class TipOfDayController {

    private static final Logger LOGGER = LoggerFactory.getLogger(TipOfDayController.class);

    @Autowired
    private SettingsService settingsService;

    private final MainView mainView;

    public TipOfDayController(final MainView mainView) {
        this.mainView = mainView;
    }

    /**
     * Show the tips of day view.
     * 
     * @param listener
     *            the closed listener
     */
    public void start(final TipOfDayClosedListener listener) {

        final TipOfDayView todView = new TipOfDayView();
        final TipOfTheDaySource tipOfTheDaySource = getTipOfDaySource();
        todView.showTipsOfTheDay(this.mainView.getFrame(), listener, settingsService, tipOfTheDaySource);
    }

    private TipOfTheDaySource getTipOfDaySource() {

        TipOfTheDaySource source = null;

        try {
            final StopWatch sw = StopWatch.createStarted();

            Locale locale = Locale.getDefault();
            String lang = locale.getLanguage();

            source = findTipOfDaySource(lang);
            if (source == null) {
                // fallback to english
                source = findTipOfDaySource("en");
            }

            sw.stop();

            LOGGER.info("Load tips data took: {}", sw.toString());
        }
        catch (Exception ex) {
            LOGGER.warn("Get tips failed.", ex);
        }

        return source;
    }

    private TipOfTheDaySource findTipOfDaySource(String lang) {

        TipOfTheDaySource source = null;
        String filename = "tips_" + lang + ".properties";

        LOGGER.info("Prepared tips file: {}", filename);

        String installationPath = this.settingsService.getInstallationPath();
        LOGGER.info("Current installation path: {}", installationPath);

        File file = new File(installationPath);
        file = new File(file.getAbsoluteFile(), "data/tips");

        String labelPath = settingsService.getMiscSettings().getBidibConfigDir();
        File searchPathLabelPath = new File(labelPath, "data/tips");

        String[] searchPaths = { searchPathLabelPath.getAbsolutePath(), file.getAbsolutePath(), "classpath:/tips" };

        for (String searchPath : searchPaths) {

            if (searchPath.startsWith("classpath:")) {
                // search in the classpath

                int beginIndex = "classpath:".length();
                String lookup = searchPath.substring(beginIndex) + "/" + filename;
                LOGGER.info("Lookup tips file internally: {}", lookup);

                URL pathString = TipOfDayController.class.getResource(lookup);
                LOGGER.info("Prepared pathString: {}", pathString);

                if (pathString == null) {
                    LOGGER.info("No resource for lookup '{}' found.", lookup);
                    continue;
                }

                // load tips from classpath
                try {
                    source = new FileTipOfTheDaySource(TipOfDayController.class.getResourceAsStream(lookup));

                    break;
                }
                catch (Exception ex) {
                    LOGGER.warn("Load tips from classpath failed.", ex);
                }
            }
            else {
                LOGGER.info("Search for files in searchPath: {}", searchPath);
                File tipsFile = new File(searchPath, filename.toString());
                if (tipsFile.exists()) {
                    LOGGER.info("Found tips file: {}", tipsFile.getAbsolutePath());

                    // load tips from file
                    try (InputStream is = new FileInputStream(tipsFile)) {
                        source = new FileTipOfTheDaySource(is);

                        break;
                    }
                    catch (Exception ex) {
                        LOGGER.warn("Load tips from file failed.", ex);
                    }
                }
            }
        }
        return source;
    }
}
