package org.bidib.wizard.utils;

import java.io.File;
import java.io.FileNotFoundException;
import java.math.BigInteger;
import java.net.URISyntaxException;
import java.net.URL;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import org.assertj.core.api.Assertions;
import org.bidib.jbidibc.core.schema.BidibFactory;
import org.bidib.jbidibc.core.schema.bidib2.BiDiB;
import org.bidib.jbidibc.core.schema.bidiblabels.NodeLabels;
import org.bidib.jbidibc.messages.ProtocolVersion;
import org.bidib.jbidibc.messages.SoftwareVersion;
import org.bidib.jbidibc.messages.StringData;
import org.bidib.wizard.api.model.NodeInterface;
import org.bidib.wizard.api.service.node.NodeService;
import org.bidib.wizard.api.service.node.SwitchingNodeService;
import org.bidib.wizard.client.common.view.cvdef.CvDefinitionTreeModelRegistry;
import org.bidib.wizard.client.common.view.cvdef.CvNode;
import org.bidib.wizard.client.common.view.cvdef.DefaultCvDefinitionTreeModelRegistry;
import org.bidib.wizard.common.labels.WizardLabelFactory;
import org.bidib.wizard.common.labels.WizardLabelWrapper;
import org.bidib.wizard.core.node.Node;
import org.junit.jupiter.api.Test;
import org.mockito.Mockito;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class NodeUtilsTest {
    private static final Logger LOGGER = LoggerFactory.getLogger(NodeUtilsTest.class);

    @Test
    public void convertToBiDiB() {
        NodeInterface node = prepareNode();
        final Map<String, CvNode> cvNumberToNodeMap = new HashMap<>();
        String lang = "de";

        final NodeLabels nodeLabels = new NodeLabels();

        final WizardLabelWrapper wizardLabelWrapper = Mockito.mock(WizardLabelWrapper.class);
        final WizardLabelFactory wizardLabelFactory = Mockito.mock(WizardLabelFactory.class);
        Mockito.when(wizardLabelWrapper.getWizardLabelFactory()).thenReturn(wizardLabelFactory);
        Mockito.when(wizardLabelFactory.loadLabels(Mockito.anyLong())).thenReturn(nodeLabels);

        BiDiB bidib = NodeUtils.convertToBiDiB(node, cvNumberToNodeMap, lang, false, wizardLabelWrapper);

        Assertions.assertThat(bidib).isNotNull();
        Assertions.assertThat(bidib.getNodes()).isNotNull();
        List<org.bidib.jbidibc.core.schema.bidib2.Node> schemaNodes = bidib.getNodes().getNode();
        Assertions.assertThat(schemaNodes.size()).isEqualTo(1);
        org.bidib.jbidibc.core.schema.bidib2.Node schemaNode = schemaNodes.get(0);
        Assertions.assertThat(schemaNode.getUserName()).isEqualTo("Testnode");
    }

    @Test
    public void configureFromBiDiB() {

        String connectionId = "main";
        final NodeService nodeService = Mockito.mock(NodeService.class);
        final SwitchingNodeService switchingNodeService = Mockito.mock(SwitchingNodeService.class);

        final Map<String, Object> importParams = new HashMap<String, Object>();

        final CvDefinitionTreeModelRegistry cvDefinitionTreeModelRegistry = new DefaultCvDefinitionTreeModelRegistry();

        NodeInterface node = prepareNode();
        BiDiB bidib = new BiDiB();
        node =
            NodeUtils
                .configureFromBiDiB(connectionId, nodeService, switchingNodeService, cvDefinitionTreeModelRegistry,
                    node, importParams, bidib, null, false, false, false, false, false);

        Assertions.assertThat(node).isNotNull();
        Assertions.assertThat(node.getNode().getStoredString(StringData.INDEX_USERNAME)).isEqualTo("Testnode");
    }

    private NodeInterface prepareNode() {
        BigInteger bigInt = new BigInteger("45000d78007cea", 16);
        long uniqueId = bigInt.longValue();

        org.bidib.jbidibc.messages.Node coreNode =
            org.bidib.jbidibc.messages.Node.createNode(1, new byte[] { 1 }, uniqueId);
        coreNode.setStoredString(StringData.INDEX_USERNAME, "Testnode");
        coreNode.setSoftwareVersion(SoftwareVersion.parse("1.00.00"));
        coreNode.setProtocolVersion(ProtocolVersion.VERSION_0_6);

        Node node = new Node(coreNode);
        node.initialize();

        return node;
    }

    @Test
    public void loadFromBiDiBFile() throws URISyntaxException, FileNotFoundException {

        URL url = NodeUtilsTest.class.getResource("/org/bidib/wizard/utils/test.bidibnodes");

        File nodesFile = new File(url.toURI());
        LOGGER.info("Prepared file: {}", nodesFile.getAbsolutePath());

        BiDiB bidib = BidibFactory.loadBiDiBFile(nodesFile);

        Assertions.assertThat(bidib).isNotNull();
        LOGGER.info("Loaded bidib: {}", bidib);

        // no protocol messages
        Assertions.assertThat(bidib.getProtocol()).isNull();

        // only nodes
        Assertions.assertThat(bidib.getNodes()).isNotNull();

        List<org.bidib.jbidibc.core.schema.bidib2.Node> schemaNodes = bidib.getNodes().getNode();
        Assertions.assertThat(schemaNodes.size()).isEqualTo(1);
        org.bidib.jbidibc.core.schema.bidib2.Node schemaNode = schemaNodes.get(0);
        Assertions.assertThat(schemaNode.getUserName()).isEqualTo("LightControl Simulation");
        Assertions.assertThat(schemaNode.getProtocol()).isEqualTo("0.5");
        Assertions.assertThat(schemaNode.getFirmwareRelease()).isEqualTo("1.0.0");
    }
}
