package org.bidib.wizard.main;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;
import java.util.Locale;
import java.util.Optional;

import javax.swing.JFrame;
import javax.swing.JOptionPane;
import javax.swing.SwingUtilities;
import javax.swing.WindowConstants;

import org.apache.commons.io.FilenameUtils;
import org.apache.commons.lang3.StringUtils;
import org.bidib.wizard.common.utils.ImageUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.boot.WebApplicationType;
import org.springframework.boot.autoconfigure.SpringBootApplication;
import org.springframework.boot.builder.SpringApplicationBuilder;

@SpringBootApplication(scanBasePackages = { "org.bidib.wizard.core.model", "org.bidib.wizard.core.service",
    "org.bidib.wizard.core.config", "org.bidib.wizard.common.context", "org.bidib.wizard.config",
    "org.bidib.wizard.startup", "org.bidib.wizard.discovery.config" })
// @EnableAutoConfiguration(exclude = DistributedBidibAutoConfig.class)
public class BiDiBWizardStandaloneApplication {

    private static final Logger LOGGER = LoggerFactory.getLogger(BiDiBWizardStandaloneApplication.class);

    public static void main(String[] args) {

        Locale locale = Locale.getDefault();
        LOGGER.info("Current locale: {}", locale);

        List<String> argsList = new ArrayList<>();
        argsList.addAll(Arrays.asList(args));

        final SpringApplicationBuilder builder =
            new SpringApplicationBuilder()
                .sources(BiDiBWizardStandaloneApplication.class).headless(false).web(WebApplicationType.NONE)
                .properties("wizard-app-type=standalone");

        if (!"restartedMain".equals(Thread.currentThread().getName())) {
            // add these extensions if the thread is not the devutils restarted main thread
            // ConfigFileApplicationListener.CONFIG_NAME_PROPERTY

            Optional<String> wizardConfigFileName =
                argsList.stream().filter(val -> val.startsWith("--wizard.settings.file-name")).findFirst();
            if (wizardConfigFileName.isPresent()) {
                LOGGER.info("Keep the provided data from args: {}", wizardConfigFileName.get());
                String fileName = wizardConfigFileName.get().split("=")[1];

                fileName = FilenameUtils.getBaseName(fileName);

                LOGGER.info("Add the fileName: {}", fileName);

                builder.properties("spring.config.name=application," + fileName);
            }
            else {
                builder.properties("spring.config.name=application,wizard");
            }

            Optional<String> wizardConfigFileLocation =
                argsList.stream().filter(val -> val.startsWith("--wizard.settings.file-location")).findFirst();
            if (wizardConfigFileLocation.isPresent()) {
                LOGGER.info("Keep the provided data from args: {}", wizardConfigFileLocation.get());

                String location = wizardConfigFileLocation.get().split("=")[1];
                location = location.replaceAll("\\\\", "/");
                if (!location.endsWith("/")) {
                    location += "/";
                }

                LOGGER.info("Set the optional additional-location: {}", location);

                builder.properties("spring.config.additional-location=optional:file:///" + location);
            }
            // else {
            // String subDir = WizardUtils.getDefaultConfigSubDir();
            // builder.properties("spring.config.additional-location=optional:file:///${user.home}/"+
            // subDir + "/");
            // }
        }

        // get the preferences path from a jvm property (set with
        // -Dbidib.preferencesPath=....
        String preferencesPath = System.getProperty("bidib.preferencesPath");

        // if the path is not set, use the value from the environment
        if (StringUtils.isBlank(preferencesPath)) {
            preferencesPath = System.getenv("bidib.preferencesPath");
            // if the path is not set use the user.home
            if (StringUtils.isBlank(preferencesPath)) {
                preferencesPath = System.getProperty("user.home");

                System.setProperty("bidib.preferencesPath", preferencesPath);
            }
        }

        try {
            builder.run(argsList.toArray(new String[0]));
        }
        catch (Exception ex) {
            LOGGER.warn("Start application failed.", ex);

            try {
                SwingUtilities.invokeAndWait(() -> {
                    JFrame frame = new JFrame("BiDiB-Wizard");
                    frame.setDefaultCloseOperation(WindowConstants.EXIT_ON_CLOSE);
                    try {
                        frame
                            .setIconImage(ImageUtils
                                .createImageIcon(BiDiBWizardStandaloneApplication.class,
                                    "/icons/wizard-logo2-48x48.png")
                                .getImage());
                    }
                    catch (Exception ex2) {
                        LOGGER.warn("Load application icon failed.");
                    }
                    frame.setLocationRelativeTo(null);
                    frame.setVisible(true);

                    JOptionPane
                        .showMessageDialog(frame,
                            "Start application failed.\r\nCheck configuration files (*.yml).\r\nReason: "
                                + ex.getMessage().replaceAll("(.{100})", "$1\n"),
                            "BiDiB-Wizard", JOptionPane.ERROR_MESSAGE);
                });
            }
            catch (Exception ex1) {
                LOGGER.warn("Show exception dialog failed.", ex1);
            }

            System.exit(1);
        }
    }

}
