package org.bidib.wizard.mvc.backup.model;

import java.beans.PropertyChangeListener;
import java.io.File;
import java.util.Collections;
import java.util.LinkedList;
import java.util.List;

import javax.swing.SwingUtilities;

import org.bidib.wizard.api.model.NodeInterface;
import org.bidib.wizard.client.common.view.statusbar.StatusBarPublisher;
import org.bidib.wizard.common.labels.WizardLabelWrapper;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.jgoodies.binding.beans.Model;
import com.jgoodies.common.collect.ArrayListModel;

public class BackupTableModel extends Model {

    private static final long serialVersionUID = 1L;

    private static final Logger LOGGER = LoggerFactory.getLogger(BackupTableModel.class);

    public static final String PROPERTY_NODES = "nodes";

    public static final String PROPERTY_BACKUP_DIRECTORY = "backupDirectory";

    public static final String PROPERTY_BACKUP_ENABLED = "backupEnabled";

    private ArrayListModel<NodeBackupModel> nodeBackupList = new ArrayListModel<>();

    private File backupDirectory;

    private final PropertyChangeListener pcs;

    public BackupTableModel(StatusBarPublisher<String, Integer> publisher) {

        this.pcs = evt -> {
            switch (evt.getPropertyName()) {
                case NodeBackupModel.PROPERTY_PROGRESS:
                    LOGGER.info("The progress has changed: {}, source: {}", evt.getNewValue(), evt.getSource());
                    fireProgressChanged((NodeBackupModel) evt.getSource());
                    break;
                default:
                    break;
            }
        };

    }

    private void fireProgressChanged(NodeBackupModel source) {
        int fireChangeIndex = -1;
        synchronized (nodeBackupList) {

            for (NodeBackupModel nodeBackup : nodeBackupList) {
                if (nodeBackup.getNode().getUniqueId() == source.getNode().getUniqueId()) {
                    LOGGER.trace("Found nodeBackup to update: {}", nodeBackup);

                    int index = nodeBackupList.indexOf(nodeBackup);
                    fireChangeIndex = index;
                    break;
                }
            }
        }

        if (fireChangeIndex > -1) {
            nodeBackupList.fireContentsChanged(fireChangeIndex);
        }
    }

    public void addNode(final NodeInterface node, final WizardLabelWrapper wizardLabelWrapper) {

        SwingUtilities.invokeLater(() -> {
            synchronized (nodeBackupList) {
                NodeBackupModel nodeBackup = new NodeBackupModel(node, wizardLabelWrapper);

                if (!nodeBackupList.contains(nodeBackup)) {
                    LOGGER.info("Add nodeBackup to nodeBackup list: {}", node);

                    nodeBackup.addPropertyChangeListener(pcs);

                    nodeBackupList.add(nodeBackup);
                }
            }
        });
    }

    public void removeNode(final NodeInterface node) {
        SwingUtilities.invokeLater(() -> {

            synchronized (nodeBackupList) {
                LOGGER.info("Remove node from nodeBackup list: {}", node);

                List<NodeBackupModel> oldValue = new LinkedList<>(nodeBackupList);
                int index = nodeBackupList.indexOf(new NodeBackupModel(node, null));
                if (index > -1) {
                    NodeBackupModel removed = nodeBackupList.remove(index);
                    if (removed != null) {
                        LOGGER.info("Removed node: {}", node);

                        removed.removePropertyChangeListener(pcs);

                    }

                    firePropertyChange(PROPERTY_NODES, oldValue, nodeBackupList);
                }
            }
        });

    }

    public ArrayListModel<NodeBackupModel> getNodeBackupListModel() {
        return nodeBackupList;
    }

    public List<NodeBackupModel> getNodes() {
        return Collections.unmodifiableList(nodeBackupList);
    }

    /**
     * @return the backupDirectory
     */
    public File getBackupDirectory() {
        return backupDirectory;
    }

    /**
     * @param backupDirectory
     *            the backupDirectory to set
     */
    public void setBackupDirectory(File backupDirectory) {
        File oldValue = this.backupDirectory;
        boolean oldBackupEnabled = isBackupEnabled();

        this.backupDirectory = backupDirectory;

        firePropertyChange(PROPERTY_BACKUP_DIRECTORY, oldValue, this.backupDirectory);

        firePropertyChange(PROPERTY_BACKUP_ENABLED, oldBackupEnabled, isBackupEnabled());
    }

    public boolean isBackupEnabled() {
        return this.backupDirectory != null && this.backupDirectory.exists();
    }

}
