package org.bidib.wizard.mvc.booster.model;

import java.beans.IndexedPropertyChangeEvent;
import java.beans.PropertyChangeEvent;
import java.beans.PropertyChangeListener;

import javax.swing.SwingUtilities;

import org.apache.commons.lang3.StringUtils;
import org.bidib.jbidibc.core.schema.bidiblabels.NodeLabels;
import org.bidib.jbidibc.messages.StringData;
import org.bidib.jbidibc.messages.enums.BoosterControl;
import org.bidib.jbidibc.messages.enums.BoosterState;
import org.bidib.jbidibc.messages.enums.CommandStationState;
import org.bidib.jbidibc.messages.enums.FeatureEnum;
import org.bidib.jbidibc.messages.utils.ByteUtils;
import org.bidib.jbidibc.messages.utils.NodeUtils;
import org.bidib.wizard.api.model.NodeInterface;
import org.bidib.wizard.client.common.view.BidibNodeNameUtils;
import org.bidib.wizard.common.labels.WizardLabelWrapper;
import org.bidib.wizard.model.status.BoosterStatus;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.jgoodies.binding.beans.Model;

public class BoosterModel extends Model {
    private static final Logger LOGGER = LoggerFactory.getLogger(BoosterModel.class);

    private static final long serialVersionUID = 1L;

    public static final String PROPERTY_BOOSTER_STATUS = "boosterStatus";

    public static final String PROPERTY_COMMANDSTATION_STATUS = "commandStationState";

    public static final String PROPERTY_BOOSTER_CURRENT = "boosterCurrent";

    public static final String PROPERTY_BOOSTER_VOLTAGE = "boosterVoltage";

    public static final String PROPERTY_BOOSTER_TEMPERATURE = "boosterTemperature";

    public static final String PROPERTY_BOOSTER_LABEL = "nodeLabel";

    private final NodeInterface booster;

    private final String uniqueId;

    private BoosterState state;

    private CommandStationState commandStationState;

    private Integer current;

    private Integer voltage;

    private Integer temperature;

    private String nodeLabel;

    private long lastCurrentUpdate;

    private PropertyChangeListener propertyChangeListener;

    private final NodeLabels nodeLabels;

    private final PropertyChangeListener boosterListener;

    public BoosterModel(final NodeInterface booster, final WizardLabelWrapper wizardLabelWrapper,
        final PropertyChangeListener boosterListener) {
        this.booster = booster;
        this.boosterListener = boosterListener;
        this.uniqueId = ByteUtils.getUniqueIdAsString(booster.getUniqueId());

        this.nodeLabels =
            wizardLabelWrapper != null ? wizardLabelWrapper.getWizardLabelFactory().loadLabels(booster.getUniqueId())
                : null;

        LOGGER.debug("Create new BoosterModel for booster: {}", booster);
    }

    public void registerNode() {
        propertyChangeListener = new PropertyChangeListener() {

            @Override
            public void propertyChange(PropertyChangeEvent evt) {
                // update the label with the name ...
                LOGGER.trace("Property has changed, name: {}, value: {}", evt.getPropertyName(), evt.getNewValue());

                switch (evt.getPropertyName()) {
                    // the label has changed
                    case org.bidib.jbidibc.messages.Node.PROPERTY_USERNAME:
                        String nodeLabel = prepareNodeLabel();
                        setNodeLabel(nodeLabel);

                        final PropertyChangeEvent evt1 =
                            new PropertyChangeEvent(BoosterModel.this, PROPERTY_BOOSTER_LABEL, null, nodeLabel);
                        boosterListener.propertyChange(evt1);
                        break;
                    case org.bidib.jbidibc.messages.Node.PROPERTY_FEATURES:
                        LOGGER.info("The features of the node have changed.");

                        if (evt instanceof IndexedPropertyChangeEvent) {
                            IndexedPropertyChangeEvent indexedPropertyChangeEvent = (IndexedPropertyChangeEvent) evt;
                            if (FeatureEnum.FEATURE_BST_AMPERE.getNumber() == indexedPropertyChangeEvent.getIndex()) {
                                SwingUtilities.invokeLater(() -> fetchMaxBoosterCurrent());
                            }
                        }
                        else {
                            SwingUtilities.invokeLater(() -> fetchMaxBoosterCurrent());
                        }
                        break;

                    case NodeInterface.PROPERTY_INITIAL_LOAD_FINISHED:
                        LOGGER
                            .info(
                                "The initial load of the booster node has finished. Fetch the max booster current from the features for node: {}",
                                booster);
                        SwingUtilities.invokeLater(() -> fetchBoosterState());
                        break;
                    default:
                        break;
                }
            }
        };
        LOGGER.info("Add property change listener for booster: {}", booster);
        this.booster.addPropertyChangeListener(propertyChangeListener);
    }

    public void freeNode() {

        LOGGER.info("Remove property change listener for booster: {}", booster);
        this.booster.removePropertyChangeListener(propertyChangeListener);
    }

    /**
     * @return the booster node
     */
    public NodeInterface getBooster() {
        return booster;
    }

    /**
     * @return the uniqueId
     */
    public String getUniqueId() {
        return uniqueId;
    }

    /**
     * @return the node address
     */
    public byte[] getNodeAddress() {
        return booster.getNode().getAddr();
    }

    /**
     * @param nodeLabel
     *            the node label to set
     */
    public void setNodeLabel(String nodeLabel) {
        String oldValue = this.nodeLabel;

        this.nodeLabel = nodeLabel;

        firePropertyChange(PROPERTY_BOOSTER_LABEL, oldValue, this.nodeLabel);
    }

    /**
     * @return the node label
     */
    public String getNodeLabel() {

        BidibNodeNameUtils.NodeLabelData labelData = BidibNodeNameUtils.prepareLabel(booster, nodeLabels, false, true);
        String userName = labelData.getNodeLabel();
        if (StringUtils.isNotBlank(userName)) {
            return userName;
        }

        return nodeLabel;
    }

    private void fetchBoosterState() {

        fetchMaxBoosterCurrent();

        try {
            if (NodeUtils.hasBoosterFunctions(booster.getUniqueId()) && booster.getBoosterNode() != null) {
                BoosterStatus boosterStatus = booster.getBoosterNode().getBoosterStatus();
                LOGGER.info("Update the booster status: {}", boosterStatus);
                setState(boosterStatus != null ? boosterStatus.getBoosterState() : null);
            }
        }
        catch (Exception ex) {
            LOGGER.warn("Get booster state of node failed.", ex);
        }

        try {
            if (NodeUtils.hasCommandStationFunctions(booster.getUniqueId())
                && booster.getCommandStationNode() != null) {
                CommandStationState csStatus = booster.getCommandStationNode().getCommandStationState();
                LOGGER.info("Update the command station state: {}", csStatus);
                setCommandStationState(csStatus);
            }
        }
        catch (Exception ex) {
            LOGGER.warn("Get comamndStation state of node failed.", ex);
        }
    }

    protected void fetchMaxBoosterCurrent() {

        try {
            if (booster.getBoosterNode() != null) {
                booster.getBoosterNode().fetchMaxBoosterCurrent();
            }
        }
        catch (Exception ex) {
            LOGGER.warn("Get FEATURE_BST_AMPERE from node failed.", ex);
        }
    }

    /**
     * @return the state
     */
    public BoosterState getState() {
        return state;
    }

    /**
     * @param state
     *            the state to set
     */
    public void setState(BoosterState state) {
        BoosterState oldValue = this.state;
        this.state = state;

        firePropertyChange(PROPERTY_BOOSTER_STATUS, oldValue, this.state);
    }

    /**
     * @return the commandStationState
     */
    public CommandStationState getCommandStationState() {
        return commandStationState;
    }

    /**
     * @param commandStationState
     *            the commandStationState to set
     */
    public void setCommandStationState(CommandStationState commandStationState) {
        CommandStationState oldValue = this.commandStationState;

        this.commandStationState = commandStationState;
        firePropertyChange(PROPERTY_COMMANDSTATION_STATUS, oldValue, this.commandStationState);
    }

    /**
     * @return the boosterControl
     */
    public BoosterControl getBoosterControl() {
        if (NodeUtils.hasAccessoryFunctions(booster.getUniqueId())) {
            if (booster.getBoosterNode() != null) {
                return booster.getBoosterNode().getBoosterControl();
            }
        }
        return null;
    }

    /**
     * @return the maxCurrent
     */
    public Integer getMaxCurrent() {
        if (booster.getBoosterNode() != null) {
            return booster.getBoosterNode().getBoosterMaximumCurrent();
        }
        return null;
    }

    /**
     * @return the current
     */
    public Integer getCurrent() {
        return current;
    }

    /**
     * @param current
     *            the current to set
     */
    public void setCurrent(Integer current) {
        Integer oldValue = this.current;

        this.current = current;

        // set the last current update time
        setLastCurrentUpdate(System.currentTimeMillis());

        firePropertyChange(PROPERTY_BOOSTER_CURRENT, oldValue, this.current);
    }

    /**
     * @return the lastCurrentUpdate
     */
    public long getLastCurrentUpdate() {
        return lastCurrentUpdate;
    }

    /**
     * @param lastCurrentUpdate
     *            the lastCurrentUpdate to set
     */
    public void setLastCurrentUpdate(long lastCurrentUpdate) {
        this.lastCurrentUpdate = lastCurrentUpdate;
    }

    /**
     * @return the voltage
     */
    public Integer getVoltage() {
        return voltage;
    }

    /**
     * @param voltage
     *            the voltage to set
     */
    public void setVoltage(Integer voltage) {
        Integer oldValue = this.voltage;

        this.voltage = voltage;

        firePropertyChange(PROPERTY_BOOSTER_VOLTAGE, oldValue, this.voltage);
    }

    /**
     * @return the temperature
     */
    public Integer getTemperature() {
        return temperature;
    }

    /**
     * @param temperature
     *            the temperature to set
     */
    public void setTemperature(Integer temperature) {
        Integer oldValue = this.temperature;

        this.temperature = temperature;

        firePropertyChange(PROPERTY_BOOSTER_TEMPERATURE, oldValue, this.temperature);
    }

    @Override
    public boolean equals(Object obj) {
        if (obj instanceof BoosterModel) {
            BoosterModel other = (BoosterModel) obj;
            if (booster.equals(other.getBooster())) {
                return true;
            }
        }
        return false;
    }

    @Override
    public int hashCode() {
        return booster.hashCode();
    }

    public String prepareNodeLabel() {
        String nodeLabel = booster.getLabel();
        if (StringUtils.isBlank(nodeLabel)) {
            // try to get the product name
            String productString = booster.getNode().getStoredString(StringData.INDEX_PRODUCTNAME);
            if (StringUtils.isNotBlank(productString)) {
                nodeLabel = productString;
            }
        }
        return nodeLabel;
    }
}
