package org.bidib.wizard.mvc.loco.view.speedo;

import java.awt.BorderLayout;
import java.awt.Component;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.util.concurrent.atomic.AtomicBoolean;

import javax.swing.JButton;
import javax.swing.JDialog;
import javax.swing.JLabel;
import javax.swing.JOptionPane;
import javax.swing.JPanel;
import javax.swing.JProgressBar;
import javax.swing.SwingUtilities;

import org.bidib.wizard.api.locale.Resources;
import org.bidib.wizard.client.common.dialog.EscapeDialog;
import org.bidib.wizard.mvc.loco.view.listener.MeasurementViewListener;
import org.bidib.wizard.mvc.loco.view.listener.ProgressStatusCallback;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.jgoodies.forms.builder.ButtonBarBuilder;
import com.jgoodies.forms.builder.FormBuilder;
import com.jgoodies.forms.factories.Paddings;

public class MeasurementProgressDialog extends EscapeDialog {
    private static final long serialVersionUID = 1L;

    private static final Logger LOGGER = LoggerFactory.getLogger(MeasurementProgressDialog.class);

    private static final String ENCODED_DIALOG_COLUMN_SPECS = "pref, 10dlu, max(300dlu;pref)";

    private int result = JOptionPane.CANCEL_OPTION;

    private final AtomicBoolean continueMeasurementHolder = new AtomicBoolean(true);

    private final ProgressStatusCallback callback;

    final MeasurementViewListener listener;

    private final JProgressBar progressBar = new JProgressBar(0, 100);

    private JLabel infoLabel;

    private JButton cancel;

    public MeasurementProgressDialog(final Component parent, boolean modal, final MeasurementViewListener listener) {
        super(JOptionPane.getFrameForComponent(parent), Resources.getString(MeasurementProgressDialog.class, "title"),
            modal);
        this.listener = listener;

        setDefaultCloseOperation(JDialog.DO_NOTHING_ON_CLOSE);

        getContentPane().setLayout(new BorderLayout());

        // DefaultFormBuilder builder = new DefaultFormBuilder(new FormLayout(ENCODED_DIALOG_COLUMN_SPECS));
        FormBuilder builder =
            FormBuilder.create().columns(ENCODED_DIALOG_COLUMN_SPECS).rows("fill:50dlu:grow, 3dlu, pref, 3dlu, pref");
        builder.border(Paddings.DIALOG);

        // builder.appendRow("fill:50dlu:grow");

        infoLabel = new JLabel();
        builder.add(infoLabel).xyw(1, 1, 3);

        builder.add(Resources.getString(getClass(), "progress")).xy(1, 3);
        builder.add(progressBar).xy(3, 3);

        progressBar.setIndeterminate(true);

        cancel = new JButton(Resources.getString(getClass(), "cancel"));
        cancel.setEnabled(false);

        cancel.addActionListener(new ActionListener() {
            @Override
            public void actionPerformed(ActionEvent e) {
                fireCancel();
            }
        });

        JPanel buttons = new ButtonBarBuilder().addGlue().addButton(cancel).build();

        // builder.nextLine();
        builder.add(buttons).xyw(1, 5, 3);

        getContentPane().add(builder.build());

        pack();
        setLocationRelativeTo(parent);

        callback = new ProgressStatusCallback() {

            @Override
            public void progressChanged(final int progressValue) {
                LOGGER.info("The progress has changed, progress: {}", progressValue);

                SwingUtilities.invokeLater(new Runnable() {
                    @Override
                    public void run() {
                        progressBar.setValue(progressValue);
                    }
                });
            }

            @Override
            public void actionFinished() {
                LOGGER.info("ActionFinish was called. Close the dialog.");
                fireClose();
            }

            @Override
            public void textChanged(String text) {
                LOGGER.info("The text has changed, text: {}", text);

                SwingUtilities.invokeLater(new Runnable() {
                    @Override
                    public void run() {
                        infoLabel.setText(text);
                    }
                });
            }
        };

    }

    public void startAction() {

        LOGGER.info("Start measurement.");
        try {
            if (listener != null) {
                listener.startMeasurement(continueMeasurementHolder, callback);
            }
        }
        catch (Exception ex) {
            LOGGER.warn("Start measurement failed.", ex);
        }

        // enable the cancel button
        cancel.setEnabled(true);

        setVisible(true);

    }

    private void fireCancel() {
        LOGGER.info("Cancel the measurement, set the continueMeasurementHolder to false.");
        continueMeasurementHolder.set(false);

        if (listener != null) {
            listener.stopMeasurement(continueMeasurementHolder, callback);
        }
        // fireClose();
    }

    private void fireClose() {
        LOGGER.info("Close the dialog.");

        if (SwingUtilities.isEventDispatchThread()) {
            try {

                if (listener != null) {
                    listener.stopMeasurement(continueMeasurementHolder, callback);
                }
            }
            catch (Exception ex) {
                LOGGER.warn("Wait for termination of worker was interrupted.", ex);
            }

            setVisible(false);
        }
        else {
            SwingUtilities.invokeLater(new Runnable() {

                @Override
                public void run() {
                    LOGGER.info("Close the dialog from AWT thread.");
                    try {
                        if (listener != null) {
                            listener.stopMeasurement(continueMeasurementHolder, callback);
                        }
                    }
                    catch (Exception ex) {
                        LOGGER.warn("Wait for termination of worker was interrupted.", ex);
                    }

                    setVisible(false);
                }
            });
        }
    }

    public int getResult() {
        return result;
    }

    public void stop() {
        LOGGER.info("Stop the measurement.");
        fireCancel();
    }
}
