package org.bidib.wizard.mvc.locolist.view.wizard;

import java.awt.Component;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.io.File;
import java.util.ArrayList;
import java.util.List;
import java.util.stream.Collectors;

import javax.swing.DefaultListCellRenderer;
import javax.swing.JButton;
import javax.swing.JComboBox;
import javax.swing.JList;
import javax.swing.JOptionPane;
import javax.swing.JPanel;
import javax.swing.filechooser.FileFilter;

import org.apache.commons.collections4.CollectionUtils;
import org.apache.commons.io.FilenameUtils;
import org.apache.commons.lang3.StringUtils;
import org.bidib.jbidibc.core.schema.LocoLibraryFactory;
import org.bidib.jbidibc.core.schema.locolibrary.v1_0.LocoLibrary;
import org.bidib.jbidibc.core.schema.locolibrary.v1_0.LocoLibraryEntryType;
import org.bidib.wizard.api.locale.Resources;
import org.bidib.wizard.core.dialog.FileDialog;
import org.bidib.wizard.model.locolist.LocoListModel;
import org.bidib.wizard.model.status.SpeedSteps;
import org.bidib.wizard.mvc.locolist.model.LocoLibraryModel;
import org.bidib.wizard.mvc.locolist.model.LocoTableModel;
import org.oxbow.swingbits.dialog.task.TaskDialog;
import org.oxbow.swingbits.dialog.task.TaskDialogs;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.jgoodies.forms.builder.ButtonBarBuilder;
import com.jgoodies.forms.builder.FormBuilder;
import com.jidesoft.dialog.ButtonNames;
import com.jidesoft.dialog.PageEvent;
import com.jidesoft.dialog.PageListener;
import com.jidesoft.wizard.WelcomeWizardPage;

public class SelectLocoPage extends WelcomeWizardPage {
    private static final long serialVersionUID = 1L;

    private static final Logger LOGGER = LoggerFactory.getLogger(SelectLocoPage.class);

    private static final String ENCODED_DIALOG_COLUMN_SPECS = "pref, 10dlu, pref:grow";

    private static final String ENCODED_DIALOG_ROW_SPECS = "pref";

    private final LocoTableModel locoTableModel;

    private final LocoLibraryModel locoLibrary;

    private JComboBox<LocoListModel> locoCombos[];

    public SelectLocoPage(final LocoTableModel locoTableModel, final LocoLibraryModel locoLibrary) {
        super(Resources.getString(SelectLocoPage.class, "title"),
            Resources.getString(SelectLocoPage.class, "description"));
        this.locoTableModel = locoTableModel;
        this.locoLibrary = locoLibrary;
    }

    @Override
    protected void initContentPane() {
        super.initContentPane();

        final FormBuilder builder =
            FormBuilder
                .create().columns(ENCODED_DIALOG_COLUMN_SPECS).rows(ENCODED_DIALOG_ROW_SPECS).panel(new JPanel());

        List<LocoListModel> locos = new ArrayList<>();
        locos.add(null);
        locos.addAll(this.locoTableModel.getLocos());

        locoCombos = new JComboBox[8];
        int row = 3;
        for (int index = 0; index < 8; index++) {
            JComboBox<LocoListModel> comboLoco = new JComboBox<>(locos.toArray(new LocoListModel[0]));
            comboLoco.setRenderer(new LocoListRenderer());

            locoCombos[index] = comboLoco;

            builder.appendRows("3dlu, pref");
            builder.add(Integer.toString(index)).xy(1, row);
            builder.add(comboLoco).xy(3, row);

            row += 2;
        }

        // builder.appendParagraphGapRow();
        // builder.nextLine(2);

        // buttons
        JButton loadButton = new JButton(Resources.getString(getClass(), "load"));

        loadButton.addActionListener(new ActionListener() {
            @Override
            public void actionPerformed(ActionEvent e) {
                fireLoad();
            }
        });

        JButton saveButton = new JButton(Resources.getString(getClass(), "save"));

        saveButton.addActionListener(new ActionListener() {
            @Override
            public void actionPerformed(ActionEvent e) {
                fireSave();
            }
        });

        JPanel buttons = new ButtonBarBuilder().addGlue().addButton(loadButton, saveButton).build();
        builder.appendRows("9dlu, pref");
        builder.add(buttons).xyw(1, row, 3);

        addComponent(builder.build(), true);

        addPageListener(new PageListener() {

            @Override
            public void pageEventFired(PageEvent e) {

                switch (e.getID()) {
                    case PageEvent.PAGE_CLOSING:

                        if (e.getSource() instanceof JButton) {
                            JButton button = (JButton) e.getSource();
                            if (ButtonNames.NEXT.equals(button.getName())) {
                                LOGGER
                                    .info("The next button was pressed, start publishing the loco list to the mouse.");

                                locoLibrary.setPushToMouse(false);
                                locoLibrary.getLocoList().clear();

                                for (JComboBox<LocoListModel> combo : locoCombos) {
                                    Object selectedItem = combo.getSelectedItem();
                                    if (selectedItem instanceof LocoListModel) {
                                        LocoListModel locoModel = (LocoListModel) selectedItem;
                                        locoLibrary.getLocoList().add(locoModel);
                                    }
                                    else {
                                        locoLibrary.getLocoList().add(null);
                                    }
                                }

                                LOGGER.info("Filled loco library and trigger push.");
                                locoLibrary.setPushToMouse(true);
                            }
                        }
                        break;
                    default:
                        break;
                }
            }
        });
    }

    protected void fireSave() {
        LocoLibrary locoLibrary = new LocoLibrary();

        int index = 0;
        for (JComboBox<LocoListModel> combo : locoCombos) {
            Object selectedItem = combo.getSelectedItem();
            if (selectedItem instanceof LocoListModel) {
                LocoListModel locoModel = (LocoListModel) selectedItem;

                LocoLibraryEntryType locoListEntry = new LocoLibraryEntryType();
                locoListEntry.setIndex(index);
                locoListEntry.setAddress(locoModel.getLocoAddress());
                locoListEntry.setName(locoModel.getLocoName());
                locoListEntry.setSpeedSteps(SpeedSteps.valueOf(locoModel.getSpeedSteps()));
                locoLibrary.getLocoList().add(locoListEntry);
            }
            index++;
        }

        final File[] libraryFile = new File[1];
        FileFilter ff = new XmlFileFilter();

        String defaultFilename = "locoLibrary-1.0.xml";
        String workingDirectory = null;

        FileDialog dialog = new FileDialog(this, FileDialog.SAVE, workingDirectory, defaultFilename, ff) {
            @Override
            public void approve(String selectedFile) {
                File file = new File(selectedFile);
                LOGGER.info("User selected file to save the loco library: {}", file);
                libraryFile[0] = file;
            }
        };
        dialog.showDialog();

        if (libraryFile[0] != null) {
            try {
                LocoLibraryFactory.saveLocoLibrary(locoLibrary, libraryFile[0]);
            }
            catch (Exception ex) {
                LOGGER.warn("Save loco library failed.", ex);
                // TODO: handle exception

                TaskDialogs
                    .build(JOptionPane.getFrameForComponent(this),
                        Resources.getString(SelectLocoPage.class, "save-failed.instruction"), ex.getMessage())
                    .title(Resources.getString(SelectLocoPage.class, "save-failed.title"))
                    .icon(TaskDialog.StandardIcon.WARNING).inform();
            }
        }
    }

    protected void fireLoad() {

        final File[] libraryFile = new File[1];
        FileFilter ff = new XmlFileFilter();
        String workingDirectory = null;

        FileDialog dialog = new FileDialog(this, FileDialog.OPEN, workingDirectory, null, ff) {
            @Override
            public void approve(String selectedFile) {
                File file = new File(selectedFile);
                LOGGER.info("User selected file to open the loco library: {}", file);
                libraryFile[0] = file;
            }
        };
        dialog.showDialog();

        if (libraryFile[0] != null) {
            try {
                for (JComboBox<LocoListModel> combo : locoCombos) {
                    combo.setSelectedItem(null);
                }

                LocoLibrary locoLibrary = LocoLibraryFactory.getLocoLibrary(libraryFile[0]);
                if (locoLibrary != null && CollectionUtils.isNotEmpty(locoLibrary.getLocoList())) {
                    LOGGER.info("Load the loco items.");

                    for (LocoLibraryEntryType locoListEntry : locoLibrary
                        .getLocoList().stream().sorted((f1, f2) -> Integer.compare(f1.getIndex(), f2.getIndex()))
                        .collect(Collectors.toList())) {

                        int index = locoListEntry.getIndex();
                        if (index > -1 && index < locoCombos.length) {
                            JComboBox<LocoListModel> combo = locoCombos[index];
                            LocoListModel locoModel = new LocoListModel(locoListEntry.getAddress());
                            locoModel.setLocoName(locoListEntry.getName());
                            locoModel.setSpeedSteps(SpeedSteps.valueOf(locoListEntry.getSpeedSteps()));
                            combo.setSelectedItem(locoModel);
                        }
                    }

                }

            }
            catch (Exception ex) {
                LOGGER.warn("Load loco library failed.", ex);

                TaskDialogs
                    .build(JOptionPane.getFrameForComponent(this),
                        Resources.getString(SelectLocoPage.class, "load-failed.instruction"), ex.getMessage())
                    .title(Resources.getString(SelectLocoPage.class, "load-failed.title"))
                    .icon(TaskDialog.StandardIcon.WARNING).inform();
            }
        }

    }

    private static final class LocoListRenderer extends DefaultListCellRenderer {

        private static final long serialVersionUID = 1L;

        @Override
        public Component getListCellRendererComponent(
            JList<?> list, Object value, int index, boolean isSelected, boolean cellHasFocus) {
            Component comp = super.getListCellRendererComponent(list, value, index, isSelected, cellHasFocus);
            if (value instanceof LocoListModel) {
                LocoListModel locoModel = (LocoListModel) value;
                setText(Integer.toString(locoModel.getLocoAddress()) + " : "
                    + (StringUtils.isNotBlank(locoModel.getLocoName()) ? locoModel.getLocoName()
                        : Integer.toString(locoModel.getLocoAddress())));
            }
            else {
                // set the empty string
                setText(" ");
            }
            return comp;
        }
    }

    private final class XmlFileFilter extends FileFilter {

        public static final String SUFFIX_XML = "xml";

        @Override
        public boolean accept(File file) {
            boolean result = false;

            if (file != null) {
                if (file.isDirectory()) {
                    result = true;
                }
                else if (file.toString() != null) {
                    String extension = FilenameUtils.getExtension(file.toString());
                    if (SUFFIX_XML.equalsIgnoreCase(extension)) {

                        result = true;
                    }
                }
            }
            return result;
        }

        @Override
        public String getDescription() {
            return Resources.getString(SelectLocoPage.class, "filter") + " (*." + SUFFIX_XML + ")";
        }
    }

}