package org.bidib.wizard.mvc.main.controller;

import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;

import javax.swing.SwingUtilities;
import javax.swing.Timer;

import org.bidib.jbidibc.messages.enums.BoosterControl;
import org.bidib.jbidibc.messages.enums.CommandStationState;
import org.bidib.jbidibc.messages.utils.NodeUtils;
import org.bidib.wizard.api.model.BoosterNodeInterface;
import org.bidib.wizard.api.model.CommandStationNodeInterface;
import org.bidib.wizard.api.model.NodeInterface;
import org.bidib.wizard.api.model.listener.BoosterStatusListener;
import org.bidib.wizard.api.model.listener.CommandStationStatusListener;
import org.bidib.wizard.api.model.listener.DefaultNodeListListener;
import org.bidib.wizard.api.service.node.BoosterService;
import org.bidib.wizard.api.service.node.CommandStationService;
import org.bidib.wizard.core.model.connection.ConnectionRegistry;
import org.bidib.wizard.model.status.BoosterStatus;
import org.bidib.wizard.model.status.CommandStationStatus;
import org.bidib.wizard.mvc.main.model.MainModel;
import org.bidib.wizard.mvc.main.view.panel.BoosterPanel;
import org.bidib.wizard.mvc.main.view.panel.listener.StatusListener;
import org.bidib.wizard.mvc.main.view.panel.listener.TabVisibilityListener;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;

public class BoosterPanelController extends DefaultNodeListListener {

    private static final Logger LOGGER = LoggerFactory.getLogger(BoosterPanelController.class);

    private final MainModel mainModel;

    private BoosterPanel boosterPanel;

    private Timer boosterCurrentTimer;

    private NodeInterface selectedNode;

    private final BoosterStatusListener boosterStatusListener;

    private final CommandStationStatusListener commandStationStatusListener;

    @Autowired
    private BoosterService boosterService;

    @Autowired
    private CommandStationService commandStationService;

    private static final long CURRENT_UPDATE_TIMEOUT = 3000;

    public BoosterPanelController(final MainModel mainModel) {
        this.mainModel = mainModel;

        boosterStatusListener = new BoosterStatusListener() {
            @Override
            public void currentChanged(final NodeInterface node, final Integer current, long timestamp) {
                if (SwingUtilities.isEventDispatchThread()) {
                    internalCurrentChanged(current, timestamp);
                }
                else {
                    SwingUtilities.invokeLater(new Runnable() {
                        @Override
                        public void run() {
                            internalCurrentChanged(current, timestamp);
                        }
                    });
                }
            }

            private void internalCurrentChanged(Integer current, long timestamp) {
                if (boosterPanel != null) {
                    boosterPanel.boosterCurrentChanged(current, timestamp);
                }
            }

            @Override
            public void maximumCurrentChanged(final NodeInterface node, final Integer maximumCurrent) {

                if (SwingUtilities.isEventDispatchThread()) {
                    internalMaximumCurrentChanged(maximumCurrent);
                }
                else {
                    SwingUtilities.invokeLater(new Runnable() {
                        @Override
                        public void run() {
                            internalMaximumCurrentChanged(maximumCurrent);
                        }
                    });
                }
            }

            private void internalMaximumCurrentChanged(Integer maximumCurrent) {
                if (boosterPanel != null) {
                    boosterPanel.boosterMaximumCurrentChanged(maximumCurrent);
                }
            }

            @Override
            public void stateChanged(final NodeInterface node, final BoosterStatus status) {
                LOGGER.info("The booster state has changed: {}", status);
                if (SwingUtilities.isEventDispatchThread()) {
                    internalStateChanged(status);
                }
                else {
                    SwingUtilities.invokeLater(new Runnable() {
                        @Override
                        public void run() {
                            internalStateChanged(status);
                        }
                    });
                }
            }

            private void internalStateChanged(BoosterStatus status) {
                LOGGER.info("Status of booster has changed: {}", status);
                if (boosterPanel != null) {
                    boosterPanel.boosterStateChanged(status);
                }
            }

            @Override
            public void temperatureChanged(final NodeInterface node, final Integer temperature) {
                if (SwingUtilities.isEventDispatchThread()) {
                    internalTemperatureChanged(temperature);
                }
                else {
                    SwingUtilities.invokeLater(new Runnable() {
                        @Override
                        public void run() {
                            internalTemperatureChanged(temperature);
                        }
                    });
                }
            }

            private void internalTemperatureChanged(Integer temperature) {
                if (boosterPanel != null) {
                    boosterPanel.temperatureChanged(temperature);
                }
            }

            @Override
            public void voltageChanged(final NodeInterface node, final Integer voltage) {
                if (SwingUtilities.isEventDispatchThread()) {
                    internalVoltageChanged(voltage);
                }
                else {
                    SwingUtilities.invokeLater(new Runnable() {
                        @Override
                        public void run() {
                            internalVoltageChanged(voltage);
                        }
                    });
                }
            }

            private void internalVoltageChanged(Integer voltage) {
                if (boosterPanel != null) {
                    boosterPanel.voltageChanged(voltage);
                }
            }

            @Override
            public void controlChanged(NodeInterface node, BoosterControl control) {
                // TODO Auto-generated method stub

            }
        };

        // // prepare the status listener for command station status messages
        commandStationStatusListener = new CommandStationStatusListener() {

            @Override
            public void statusChanged(final NodeInterface node, final CommandStationStatus status) {
                LOGGER.info("The command station status has changed, node: {}, status: {}", node, status);
                if (SwingUtilities.isEventDispatchThread()) {
                    internalStateChanged(node, status);
                }
                else {
                    SwingUtilities.invokeLater(new Runnable() {
                        @Override
                        public void run() {
                            internalStateChanged(node, status);
                        }
                    });
                }
            }

            private void internalStateChanged(NodeInterface node, final CommandStationStatus status) {
                LOGGER.info("Status of command station has changed: {}", status);

                if (boosterPanel != null && node.equals(BoosterPanelController.this.selectedNode)) {
                    boosterPanel.commandStationStatusChanged(status);
                }
            }
        };

    }

    public BoosterPanel createPanel(final TabVisibilityListener tabVisibilityListener) {
        LOGGER.info("Create new booster panel.");

        final BoosterPanel boosterPanel = new BoosterPanel(mainModel);

        boosterPanel.addStatusListener(new StatusListener() {
            @Override
            public void switchedOff() {
                NodeInterface node = mainModel.getSelectedNode();
                boosterService
                    .setBoosterState(ConnectionRegistry.CONNECTION_ID_MAIN, node.getBoosterNode(), BoosterStatus.OFF);
            }

            @Override
            public void switchedOn() {
                NodeInterface node = mainModel.getSelectedNode();
                boosterService
                    .setBoosterState(ConnectionRegistry.CONNECTION_ID_MAIN, node.getBoosterNode(), BoosterStatus.ON);
            }

            @Override
            public void queryBoosterState() {
                // trigger the booster state
                NodeInterface node = mainModel.getSelectedNode();
                boosterService.queryBoosterState(ConnectionRegistry.CONNECTION_ID_MAIN, node.getBoosterNode());
            }

            @Override
            public void switchedCommandStationOn(boolean ignoreWatchDog) {

                LOGGER.info("Switch command station on, ignoreWatchDog: {}", ignoreWatchDog);

                setCommandStationState(ignoreWatchDog ? CommandStationStatus.GO_IGN_WD : CommandStationStatus.GO);
            }

            @Override
            public void switchedCommandStationStop() {
                LOGGER.info("Switch command station to stop.");
                setCommandStationState(CommandStationStatus.STOP);
            }

            @Override
            public void switchedCommandStationSoftStop() {
                LOGGER.info("Switch command station to softStop.");
                setCommandStationState(CommandStationStatus.SOFTSTOP);
            }

            @Override
            public void switchedCommandStationOff() {
                LOGGER.info("Switch command station off.");
                setCommandStationState(CommandStationStatus.OFF);
            }

            private void setCommandStationState(CommandStationStatus status) {

                final NodeInterface node = mainModel.getSelectedNode();

                if (node != null && node.getCommandStationNode() != null) {
                    commandStationService
                        .setCommandStationState(ConnectionRegistry.CONNECTION_ID_MAIN, node.getCommandStationNode(),
                            status);
                }
                else {
                    LOGGER.warn("No command station node available.");
                }
            }
        });

        try {
            // start the booster current timer
            boosterCurrentTimer = new Timer(1000, new ActionListener() {

                @Override
                public void actionPerformed(ActionEvent e) {
                    LOGGER.trace("The booster current timer has elapsed.");

                    long now = System.currentTimeMillis();

                    try {
                        final NodeInterface selectedNode = mainModel.getSelectedNode();
                        if (selectedNode != null && NodeUtils.hasBoosterFunctions(selectedNode.getUniqueId())) {
                            BoosterNodeInterface boosterNode = selectedNode.getBoosterNode();
                            if (boosterNode != null && boosterNode.getBoosterCurrent() != null
                                && boosterNode.getLastCurrentUpdate() != null) {

                                if (boosterNode.getBoosterCurrent() >= 0
                                    && boosterNode.getLastCurrentUpdate() < (now - CURRENT_UPDATE_TIMEOUT)) {
                                    // the current value is outdated -> clear the value
                                    LOGGER
                                        .info("the current value is outdated -> clear the value, node: {}",
                                            selectedNode);

                                    boosterNode.setBoosterCurrent(null, System.currentTimeMillis());
                                }
                            }
                        }
                    }
                    catch (Exception ex) {
                        LOGGER.warn("Clear the outdated booster current value failed.", ex);
                    }
                }
            });
            boosterCurrentTimer.setCoalesce(true);
            boosterCurrentTimer.start();
        }
        catch (Exception ex) {
            LOGGER.warn("Start the booster current timer failed.", ex);
        }

        this.boosterPanel = boosterPanel;

        return boosterPanel;
    }

    @Override
    public void nodeChanged(final NodeInterface node) {
        // LOGGER.trace("The selected node has changed.");

        updateComponentState();
    }

    private void updateComponentState() {

        final NodeInterface node = mainModel.getSelectedNode();

        if (selectedNode != null && selectedNode.equals(node)) {
            LOGGER.debug("Node is selected already: {}", node);
            return;
        }

        if (selectedNode != null) {
            BoosterNodeInterface boosterNode = selectedNode.getBoosterNode();
            if (boosterNode != null) {
                LOGGER.debug("Remove booster status listener from boosterNode: {}", boosterNode);
                boosterNode.removeBoosterStatusListener(boosterStatusListener);
            }
            else {
                LOGGER.debug("The selected node has no booster functions.");
            }

            CommandStationNodeInterface commandStationNode = selectedNode.getCommandStationNode();
            if (commandStationNode != null) {
                commandStationNode.removeCommandStationStatusListener(commandStationStatusListener);
            }
            else {
                LOGGER.debug("The selected node has no command station functions.");
            }
        }

        // release the previously selected node
        selectedNode = null;

        boolean isBooster = false;
        if (node != null && NodeUtils.hasBoosterFunctions(node.getUniqueId())) {
            isBooster = true;
        }

        boolean isCommandStation = false;
        if (node != null && NodeUtils.hasCommandStationFunctions(node.getUniqueId())) {
            isCommandStation = true;
        }

        if (isBooster || isCommandStation) {
            // set the new selected node
            selectedNode = node;
        }

        LOGGER
            .info("The selected node has booster functions: {}, has command station functions: {}, node: {}", isBooster,
                isCommandStation, node);

        BoosterNodeInterface boosterNode = null;
        CommandStationNodeInterface commandStationNode = null;

        if (selectedNode != null) {
            boosterNode = selectedNode.getBoosterNode();
            if (boosterNode != null) {
                LOGGER.debug("Add booster status listener to boosterNode: {}", boosterNode);
                boosterNode.addBoosterStatusListener(boosterStatusListener);
            }
            else {
                LOGGER.debug("The new selected node has no booster functions.");
            }

            commandStationNode = selectedNode.getCommandStationNode();
            if (commandStationNode != null) {
                LOGGER.debug("Adding command station status listener to node: {}", commandStationNode);
                commandStationNode.addCommandStationStatusListener(commandStationStatusListener);
            }
            else {
                LOGGER.debug("The new selected node has no command station functions.");
            }
        }

        if (boosterPanel != null) {
            boosterPanel.nodeChanged();
        }

        if (boosterNode != null) {

            // request the voltage
            Integer voltage = boosterNode.getBoosterVoltage();
            LOGGER.debug("Set the voltage: {}", voltage);
            boosterStatusListener.voltageChanged(selectedNode, voltage);
            boosterStatusListener.stateChanged(node, boosterNode.getBoosterStatus());
            boosterStatusListener.maximumCurrentChanged(node, boosterNode.getBoosterMaximumCurrent());
            boosterStatusListener.currentChanged(node, boosterNode.getBoosterCurrent(), System.currentTimeMillis());
            boosterStatusListener.temperatureChanged(node, boosterNode.getBoosterTemperature());
        }

        if (commandStationNode != null) {
            CommandStationState csState = commandStationNode.getCommandStationState();

            if (csState == null) {
                LOGGER.warn("The command station status is not available. Use OFF as state.");
                csState = CommandStationState.OFF;
            }
            final CommandStationStatus csStatus = CommandStationStatus.valueOf(csState);
            commandStationStatusListener.statusChanged(node, csStatus);
        }
    }

}
